package eu.dnetlib.clients.functionality.alert.ws;

import java.net.URI;
import java.net.URL;
import java.util.Date;
import java.util.SortedSet;
import java.util.TreeSet;

import eu.dnetlib.api.functionality.AlertService;
import eu.dnetlib.api.functionality.AlertServiceException;
import eu.dnetlib.clients.ws.BaseWebServiceClient;
import eu.dnetlib.domain.functionality.AlertSubscription;
import eu.dnetlib.domain.functionality.AlertTemplate;
import eu.dnetlib.domain.functionality.NotificationEvent;
import eu.dnetlib.domain.functionality.ObjectPage;
import eu.dnetlib.domain.functionality.ResultPage;

/**
 * This class implements an alert service using an alert web service.
 * @author thanos@di.uoa.gr
 * @see eu.dnetlib.api.functionality.AlertService
 * @see AlertWebService
 * @see eu.dnetlib.api.functionality.AlertServiceException
 *
 */
public class AlertWebServiceClient extends BaseWebServiceClient<AlertWebService> implements AlertService {
	@Override
	public SortedSet<String> getSupportedAlertModes() {
		final SortedSet<String> supportedAlertModes = webService.getSupportedAlertModes();
		return (supportedAlertModes == null) ? new TreeSet<String>() : supportedAlertModes;
	}
	
	@Override
	public ObjectPage<AlertTemplate> getTemplates(final int pageNumber, final int pageSize) throws AlertServiceException {
		try {
			return webService.getTemplates(pageNumber, pageSize);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error retrieving templates (page: " + pageNumber + ", page size: " + pageSize + ")", e);
		}
	}
	
	@Override
	public void addTemplate(final AlertTemplate template) throws AlertServiceException {
		try {
			webService.addTemplate(template);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error adding template " + template, e);
		}
	}
	
	@Override
	public void removeTemplate(final String templateId) throws AlertServiceException {
		try {
			webService.removeTemplate(templateId);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error removing template " + templateId, e);
		}
	}
	
	@Override
	public ObjectPage<AlertSubscription> getSubscriptions(final int pageNumber, final int pageSize) throws AlertServiceException {
		try {
			return webService.getSubscriptions(pageNumber, pageSize);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error retrieving subscriptions (page: " + pageNumber + ", page size: " + pageSize + ")", e);
		}
	}
	
	@Override
	public SortedSet<AlertSubscription> getSubscriptions(final String alertMode, final String subscriber, final int limit, final int offset) throws AlertServiceException {
		try {
			final SortedSet<AlertSubscription> subscriptions = webService.getSubscriptions(alertMode, subscriber, limit, offset);
			return (subscriptions == null) ? new TreeSet<AlertSubscription>() : subscriptions;
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error retrieving subscriptions with alert mode " + alertMode + " and subscriber like '" + subscriber + "' (limit: " + limit + ", offset: " + offset + ")", e);
		}
	}
	
	@Override
	public void addSubscription(final AlertSubscription subscription) throws AlertServiceException {
		try {
			webService.addSubscription(subscription);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error adding subscription " + subscription, e);
		}
	}
	
	@Override
	public void enableSubscription(final String templateId, final URL notificationService, final String queryId, final String resultId, final String alertMode, final URI subscriber) throws AlertServiceException {
		try {
			webService.enableSubscription(templateId, notificationService, queryId, resultId, alertMode, subscriber);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error enabling subscription (template: " + templateId + ", notification service: " + notificationService + ", query: " + queryId + ", result: " + resultId + ", alert mode: " +
					alertMode + ", subscriber: " + subscriber + ")", e);
		}
	}
	
	@Override
	public void disableSubscription(final String templateId, final URL notificationService, final String queryId, final String resultId, final String alertMode, final URI subscriber) throws AlertServiceException {
		try {
			webService.disableSubscription(templateId, notificationService, queryId, resultId, alertMode, subscriber);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error disabling subscription (template: " + templateId + ", notification service: " + notificationService + ", query: " + queryId + ", result: " + resultId + ", alert mode: " +
					alertMode + ", subscriber: " + subscriber + ")", e);
		}
	}

	@Override
	public void removeSubscription(final String templateId, final URL notificationService, final String queryId, final String resultId, final String alertMode, final URI subscriber) throws AlertServiceException {
		try {
			webService.removeSubscription(templateId, notificationService, queryId, resultId, alertMode, subscriber);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error removing subscription (template: " + templateId + ", notification service: " + notificationService + ", query: " + queryId + ", result: " + resultId + ", alert mode: " +
					alertMode + ", subscriber: " + subscriber + ")", e);
		}
	}
	
	@Override
	public int countAlertResults(final URL notificationService, final String queryId, final Date date, final String resultId) throws AlertServiceException {
		try {
			return webService.countAlertResults(notificationService, queryId, date, resultId);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error counting alert results (notification service: " + notificationService + ", query: " + queryId + ", date: " + date + ", result: " + resultId + ")", e);
		}
	}

	@Override
	public ResultPage getAlertResults(final URL notificationService, final String queryId, final String resultId, final Date fromDate, final Date toDate, final int limit, final int offset) throws AlertServiceException {
		try {
			return webService.getAlertResults(notificationService, queryId, resultId, fromDate, toDate, limit, offset);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error retrieving alert results (notification service: " + notificationService + ", query: " + queryId + ", result: " + resultId + ", from date: " + fromDate +
					", to date: " + toDate + ", limit: " + limit + ", offset: " + offset + ")", e);
		}
	}
	
	@Override
	public void alert(final URL notificationService, final NotificationEvent event) throws AlertServiceException {
		try {
			webService.alert(notificationService, event);
		} catch (final AlertWebServiceException e) {
			throw new AlertServiceException("error receiving notification from notification service " + notificationService + " about event " + event, e);
		}
	}
}
