package eu.dnetlib.clients.functionality.alert.ws;

import java.net.URI;
import java.net.URL;
import java.util.Date;
import java.util.SortedSet;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.api.functionality.AlertService;
import eu.dnetlib.api.functionality.AlertServiceException;
import eu.dnetlib.clients.ws.DriverWebService;
import eu.dnetlib.domain.functionality.AlertSubscription;
import eu.dnetlib.domain.functionality.AlertTemplate;
import eu.dnetlib.domain.functionality.NotificationEvent;
import eu.dnetlib.domain.functionality.ObjectPage;
import eu.dnetlib.domain.functionality.ResultPage;

/**
 * This interface declares the available methods of an alert service which is accessible as a web service.
 * @author thanos@di.uoa.gr
 * @see eu.dnetlib.api.functionality.AlertService
 * @see AlertWebServiceException
 *
 */
@WebService
public interface AlertWebService extends DriverWebService<AlertService> {
	/**
	 * Retrieve all the supported modes in which alerts can be sent.
	 * @return a sorted set containing all the supported modes in which alerts can be sent
	 */
	@WebMethod(operationName = "getSupportedAlertModes")
	public SortedSet<String> getSupportedAlertModes();
	
	/**
	 * Retrieve a page of templates.
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing templates
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getTemplates")
	public ObjectPage<AlertTemplate> getTemplates(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws AlertWebServiceException;
	
	/**
	 * Add a template.
	 * @param template the template to add
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addTemplate")
	public void addTemplate(@WebParam(name = "template") final AlertTemplate template) throws AlertWebServiceException;
	
	/**
	 * Remove a template.
	 * @param templateId the unique identifier of the template to remove
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeTemplate")
	public void removeTemplate(@WebParam(name = "templateId") final String templateId) throws AlertWebServiceException;
	
	/**
	 * Retrieve a page of subscriptions.
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing subscriptions
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getSubscriptions")
	public ObjectPage<AlertSubscription> getSubscriptions(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws AlertWebServiceException;
	
	/**
	 * Retrieve some subscriptions by alert mode and subscriber.
	 * @param alertMode part of the alert mode to match
	 * @param subscriber part of the subscriber URI to match
	 * @param limit the maximum number of subscriptions to retrieve
	 * @param offset the offset to start at 
	 * @return a sorted set of subscriptions
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getSubscriptionsByAlertModeAndSubscriber")
	public SortedSet<AlertSubscription> getSubscriptions(@WebParam(name = "alertMode") final String alertMode, @WebParam(name = "subscriber") final String subscriber, @WebParam(name = "limit") final int
			limit, @WebParam(name = "offset") final int offset) throws AlertWebServiceException; 

	/**
	 * Add a new subscription.
	 * @param subscription the subscription to add
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addSubscription")
	public void addSubscription(@WebParam(name = "subscription") final AlertSubscription subscription) throws AlertWebServiceException;
	
	/**
	 * Enable a subscription.
	 * @param templateId the unique identifier of the template of the subscription to enable
	 * @param notificationService the URL of the notification service of the subscription to enable
	 * @param queryId the unique identifier of the notification query of the subscription to enable
	 * @param resultId the unique identifier of the result of the subscription to enable
	 * @param alertMode the alert mode of the subscription to enable
	 * @param subscriber the URI of the subscriber of the subscription to enable 
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "enableSubscription")
	public void enableSubscription(@WebParam(name = "templateId") final String templateId, @WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "queryId") final String queryId,
			@WebParam(name = "resultId") final String resultId, @WebParam(name = "alertMode") final String alertMode, @WebParam(name = "subscriber") final URI subscriber) throws AlertWebServiceException;
	
	/**
	 * Disable a subscription.
	 * @param templateId the unique identifier of the template of the subscription to disable
	 * @param notificationService the URL of the notification service of the subscription to disable
	 * @param queryId the unique identifier of the notification query of the subscription to disable
	 * @param resultId the unique identifier of the result of the subscription to disable
	 * @param alertMode the alert mode of the subscription to disable
	 * @param subscriber the URI of the subscriber of the subscription to disable 
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "disableSubscription")
	public void disableSubscription(@WebParam(name = "templateId") final String templateId, @WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "queryId") final String queryId,
			@WebParam(name = "resultId") final String resultId, @WebParam(name = "alertMode") final String alertMode, @WebParam(name = "subscriber") final URI subscriber) throws AlertWebServiceException;
	
	/**
	 * Remove a subscription.
	 * @param templateId the unique identifier of the template of the subscription to remove
	 * @param notificationService the URL of the notification service of the subscription to remove
	 * @param queryId the unique identifier of the notification query of the subscription to remove
	 * @param resultId the unique identifier of the result of the subscription to remove
	 * @param alertMode the alert mode of the subscription to remove
	 * @param subscriber the URI of the subscriber of the subscription to remove 
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeSubscription")
	public void removeSubscription(@WebParam(name = "templateId") final String templateId, @WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "queryId") final String queryId,
			@WebParam(name = "resultId") final String resultId, @WebParam(name = "alertMode") final String alertMode, @WebParam(name = "subscriber") final URI subscriber) throws AlertWebServiceException;
	
	/**
	 * Count the results of an alert.
	 * @param templateId the unique identifier of the template of the alert results to count
	 * @param notificationService the URL of the notification service of the alert results to count
	 * @param queryId the unique identifier of the notification query of the alert results to count
	 * @param date the date that corresponds to the notification event of the alert results to count
	 * @param resultId the unique identifier of the notification result of the alert results to count
	 * @return the total number of the specified alert results
	 * @throws AlertServiceException if any errors occur
	 */
	@WebMethod(operationName = "countAlertResults")
	public int countAlertResults(@WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "queryId") final String queryId, @WebParam(name = "date") final Date date, @WebParam(name = "resultId")
			final String resultId) throws AlertWebServiceException;

	/**
	 * Retrieve a page of the results of an alert.
	 * @param notificationService the URL of the notification service of the alert results to retrieve
	 * @param queryId the unique identfier of the notification query of the alert results to retrieve
	 * @param resultId the unique identifier of the notification result of the alert results to retrieve
	 * @param fromDate the date from which to retrieve alert results
	 * @param toDate the date up to which to retrieve alert results
	 * @param limit the maximum number of alert results to retrieve
	 * @param offset the offset to start at
	 * @return a result page
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getAlertResults")
	public ResultPage getAlertResults(@WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "queryId") final String queryId, @WebParam(name = "resultId") final String resultId, @WebParam(name
			= "fromDate") final Date fromDate, @WebParam(name = "toDate") final Date toDate, @WebParam(name ="limit") final int limit, @WebParam(name = "offset") final int offset) throws AlertWebServiceException;
	
	/**
	 * Alert this service about an event generated by a notification service.
	 * @param notificationService the URL of the notification service
	 * @param event the notification event
	 * @throws AlertWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "alert")
	public void alert(@WebParam(name = "notificationService") final URL notificationService, @WebParam(name = "event") final NotificationEvent event) throws AlertWebServiceException;
}
