package eu.dnetlib.clients.functionality.notification.ws;

import java.net.URL;
import java.util.Date;
import java.util.SortedSet;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.api.functionality.NotificationService;
import eu.dnetlib.clients.ws.DriverWebService;
import eu.dnetlib.domain.functionality.NotificationEvent;
import eu.dnetlib.domain.functionality.NotificationQuery;
import eu.dnetlib.domain.functionality.NotificationResult;
import eu.dnetlib.domain.functionality.NotificationSchedule;
import eu.dnetlib.domain.functionality.NotificationSubscription;
import eu.dnetlib.domain.functionality.ObjectPage;
import eu.dnetlib.domain.functionality.ResultPage;

/**
 * This interface declares the available methods of a notification service which is accessible as a web service.
 * @author thanos@di.uoa.gr
 * @see eu.dnetlib.api.functionality.NotificationService
 * @see NotificationWebServiceException
 * 
 */
@WebService
public interface NotificationWebService extends DriverWebService<NotificationService> {
	/**
	 * Retrieve all the supported languages in which queries can be expressed.
	 * @return a set containing all the supported languages in which queries can be expressed
	 */
	@WebMethod(operationName = "getSupportedQueryLanguages")
	public SortedSet<String> getSupportedQueryLanguages();

	/**
	 * Retrieve a page of queries.
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing queries
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getQueries")
	public ObjectPage<NotificationQuery> getQueries(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws NotificationWebServiceException;
	
	/**
	 * Add a query.
	 * @param query the query to add
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addQuery")
	public void addQuery(@WebParam(name = "query") final NotificationQuery query) throws NotificationWebServiceException;
	
	/**
	 * Remove a query.
	 * @param queryId the unique identifier of the query to remove
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeQuery")
	public void removeQuery(final String queryId) throws NotificationWebServiceException;

	/**
	 * Execute a query and return the generated result.
	 * @param queryId the unique identifier of the query to execute
	 * @param resultId the unique result identifier to use
	 * @param fromDate the from date to use
	 * @param toDate the to date to use
	 * @param limit the limit to use
	 * @param offset the offset to use
	 * @return a result page containing the generated results
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "executeQuery")
	public ResultPage executeQuery(@WebParam(name = "queryId") final String queryId, @WebParam(name = "resultId") final String resultId, @WebParam(name = "fromDate") final Date fromDate, @WebParam(name = "toDate") final Date
			toDate, @WebParam(name = "limit") final int limit, @WebParam(name = "offset") final int offset) throws NotificationWebServiceException;

	/**
	 * Retrieve a page of schedules
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing schedules
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getSchedules")
	public ObjectPage<NotificationSchedule> getSchedules(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws NotificationWebServiceException;
	
	/**
	 * Add a schedule.
	 * @param schedule the schedule to add
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addSchedule")
	public void addSchedule(@WebParam(name = "schedule") final NotificationSchedule schedule) throws NotificationWebServiceException;
	
	/**
	 * Enable a schedule
	 * @param queryId the unique identifier of the query whose schedule to enable
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "enableSchedule")
	public void enableSchedule(@WebParam(name = "queryId") final String queryId) throws NotificationWebServiceException;

	/**
	 * Disable a schedule
	 * @param queryId the unique identifier of the query whose schedule to disable
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "disableSchedule")
	public void disableSchedule(@WebParam(name = "queryId") final String queryId) throws NotificationWebServiceException;
	
	/**
	 * Remove a schedule.
	 * @param queryId the unique identifier of the query of the schedule to remove
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeSchedule")
	public void removeSchedule(@WebParam(name = "queryId") final String queryId) throws NotificationWebServiceException;
	
	/**
	 * Retrieve a page of events.
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing events
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getEvents")
	public ObjectPage<NotificationEvent> getEvents(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws NotificationWebServiceException;
	
	/**
	 * Retrieve a page of results.
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing results
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getResults")
	public ObjectPage<NotificationResult> getResults(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws NotificationWebServiceException;
	
	/**
	 * Retrieve a result of an event of a query.
	 * @param queryId the unique identifier of the query of the event of the result to retrieve
	 * @param date the date corresponding to the event of the result to retrieve
	 * @param resultId the unique identifier of the result to retrieve
	 * @return the specified result of the specified event of the specified query or null if no such result exists
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getResult")
	public NotificationResult getResult(@WebParam(name = "queryId") final String queryId, @WebParam(name = "date") final Date date, @WebParam(name = "resultId") final String resultId) throws NotificationWebServiceException;
	
	/**
	 * Retrieve a result of the previous event of an event of a query.
	 * @param queryId the unique identifier of the query of the event of the result to retrieve
	 * @param date the date corresponding to the next event of the event whose result to retrieve 
	 * @param resultId the unique identifier of the result to retrieve
	 * @return the specified result of the previous event of the specified event of the specified query or null if no such result exists
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getPreviousResult")
	public NotificationResult getPreviousResult(@WebParam(name = "queryId") final String queryId, @WebParam(name = "date") final Date date, @WebParam(name = "resultId") final String resultId) throws
			NotificationWebServiceException;
	
	/**
	 * Retrieve a page of subscriptions
	 * @param pageNumber the number of the page to retrieve
	 * @param pageSize the size of the page to retrieve
	 * @return an object page containing subscriptions
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getSubscriptions")
	public ObjectPage<NotificationSubscription> getSubscriptions(@WebParam(name = "pageNumber") final int pageNumber, @WebParam(name = "pageSize") final int pageSize) throws NotificationWebServiceException;
	
	/**
	 * Add a subscription.
	 * @param subscription the subscription to add
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addSubscription")
	public void addSubscription(@WebParam(name = "subscription") final NotificationSubscription subscription) throws NotificationWebServiceException;
	
	/**
	 * Enable a subscription.
	 * @param queryId the unique identifier of the query of the schedule of the subscription to enable
	 * @param alertService the URL of the alert service to enable
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "enableSubscription")
	public void enableSubscription(@WebParam(name = "queryId") final String queryId, @WebParam(name = "alertService") final URL alertService) throws NotificationWebServiceException;
	
	/**
	 * Disable a subscription.
	 * @param queryId the unique identifier of the query of the schedule of the subscription to disable
	 * @param alertService the URL of the alert service to disable
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "disableSubscription")
	public void disableSubscription(@WebParam(name = "queryId") final String queryId, @WebParam(name = "alertService") final URL alertService) throws NotificationWebServiceException;
	
	/**
	 * Remove a subscription.
	 * @param queryId the unique identifier of the query of the schedule of the subscription to remove
	 * @param alertService the URL of the alert service to remove
	 * @throws NotificationWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeSubscription")
	public void removeSubscription(@WebParam(name = "queryId") final String queryId, @WebParam(name = "alertService") final URL alertService) throws NotificationWebServiceException;	
}
