package eu.dnetlib.clients.functionality.recommendation.ws;

import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.api.functionality.RecommendationService;
import eu.dnetlib.api.functionality.RecommendationServiceException;
import eu.dnetlib.clients.ws.DriverWebService;
import eu.dnetlib.domain.functionality.Recommendation;

@WebService
public interface RecommendationWebService extends
		DriverWebService<RecommendationService> {

	/**
	 * creates a new recommendation of type announcement
	 * 
	 * @param recommendationText
	 *            the text of the recommendation
	 * @param creationDate
	 *            the creation date of the announcement
	 * @param expirationDate
	 *            the date the announcement expires
	 * @return the id of the new recommendation
	 * @throws RecommendationServiceException
	 *             if the announcement cannot be created
	 */
	public String generateAnnouncement(
			@WebParam(name = "announcementIndex") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "announcementTitle") String announcementTitle,
			@WebParam(name = "announcementText") String announcementText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate)
			throws RecommendationWebServiceException;

	/**
	 * Swap the index of 2 announcements
	 * 
	 * @param announcementId1
	 *            the id of the fist announcement
	 * @param announcementId2
	 *            the id of the second announcement
	 */
	public void swapAnnouncements( 
			@WebParam(name = "announcementId1") String announcementId1,
			@WebParam(name = "announcementId2") String announcementId2 ) throws RecommendationWebServiceException;
	
	public void swapUserRecommendations( 
			@WebParam(name = "recommendationId1") String recommendationId1,
			@WebParam(name = "recommendationId2") String recommendationId2 ) throws RecommendationWebServiceException;
	
	public void swapCommunityRecommendations( 
			@WebParam(name = "communityId1") String communityId1,
			@WebParam(name = "communityId2") String communityId2 ) throws RecommendationWebServiceException;
		
	/**
	 * creates a new recommendation
	 * 
	 * @param rText
	 *            the recommendation text
	 * @param creationDate
	 *            the date the recommendation was created
	 * @param expirationDate
	 *            the date the recommendation expires
	 * @return the id of the recommendation
	 * @throws RecommendationWebServiceException
	 *             if the recommendation cannot be created
	 */
	public Recommendation generateRecommendation(
			@WebParam(name = "index") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "title") String title,
			@WebParam(name = "recommendationText") String recommendationText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate)
			throws RecommendationWebServiceException;

	/**
	 * creates a new recommendation of type community
	 * 
	 * @param recommendationText
	 *            the recommendation text
	 * @param creationDate
	 * @param expirationDate
	 * @param communityId
	 *            the ids of the associated communities
	 * @return the id of the recommendation
	 * @throws RecommendationWebServiceException
	 *             if the community recommendation cannot be created
	 */
	public String generateCommunityRecommendation(
			@WebParam(name = "index") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "title") String title,
			@WebParam(name = "recommendationText") String recommendationText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate,
			@WebParam(name = "communityIds") Set<String> communityIds)
			throws RecommendationWebServiceException;

	/**
	 * creates a new recommendation of type user
	 * 
	 * @param recommendationText
	 * @param creationDate
	 * @param expirationDate
	 * @return the id of the recommendation
	 * @throws RecommendationWebServiceException
	 *             if the user recommendation cannot be created
	 */
	public String generateUserRecommendation(
			@WebParam(name = "index") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "title") String title,
			@WebParam(name = "userId") String userId,
			@WebParam(name = "recommendationText") String recommendationText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate)
			throws RecommendationWebServiceException;

	/**
	 * retrieves the recommendation text of all announcements that haven't
	 * expired
	 * 
	 * @return the announcements' recommendation text
	 * @throws RecommendationServiceException
	 */
	public List<String> getAnnouncements()
			throws RecommendationServiceException;

	/**
	 * retrieves ths ids of all announcements
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<String> getAllAnnouncementIds()
			throws RecommendationServiceException;

	/**
	 * retrieves the recommendation text of all announcements
	 * 
	 * @return the recommendation text of all announcements
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getAllAnnouncements()
			throws RecommendationServiceException;

	/**
	 * retrieves all recommendations of type community, just the content
	 * 
	 * @return the recommendation text of all community recommendations
	 * @throws RecommendationServiceException
	 */
	@WebMethod(operationName = "getAllCommunityRecommendations")
	public List<String> getAllCommunityRecommendations()
			throws RecommendationServiceException;

	/**
	 * retrieves all recommendations Objects, not just the content of type community
	 * 
	 * @return the recommendation text of all community recommendations
	 * @throws RecommendationServiceException
	 */
	@WebMethod(operationName = "getAllCommunityRecommendationsObj")
	public List<Recommendation> getAllCommunityRecommendationsObj()
			throws RecommendationServiceException;
	
	@WebMethod(operationName = "getAllCommunityRecommendationsIds")
	public List<String> getCommunityRecommendations(
			@WebParam(name = "communityId") String communityId)
			throws RecommendationServiceException;

	@WebMethod(operationName = "getCommunityRecommendationsObjIds")
	public List<Recommendation> getCommunityRecommendationsObj(
			@WebParam(name = "communityId") String communityId)
			throws RecommendationServiceException;
	
	/**
	 * retrieves all community recommendations that are associated with a
	 * community
	 * 
	 * @param communityId
	 *            the id of the community
	 * @return the recommendation text of the community recommendations
	 * @throws RecommendationServiceException
	 */
	@WebMethod(operationName = "getAllCommunityRecommendationsByCommunityId")
	public List<String> getAllCommunityRecommendations(
			@WebParam(name = "communityId") String communityId)
			throws RecommendationServiceException;

	@WebMethod(operationName = "getAllCommunityRecommendationsByCommunityIdObj")
	public List<Recommendation> getAllCommunityRecommendationsObj(
			@WebParam(name = "communityId") String communityId)
			throws RecommendationServiceException;

	
	public List<String> getAllUserRecommendations(
			@WebParam(name = "userId") String userId)
			throws RecommendationServiceException;

	/**
	 * retrieves a recommendation
	 * 
	 * @param recommendationId
	 *            the recommendation id
	 * @return the recommendation or null if the recommendation does not exist
	 * @throws RecommendationServiceException
	 * 
	 */

	public Recommendation getRecommendation(
			@WebParam(name = "recommendationId") String recommendationId)
			throws RecommendationServiceException;

	/**
	 * retrieves the text of a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @return the recommendation text or null if the recommendation does not
	 *         exist
	 * @throws RecommendationServiceException
	 * 
	 */
	public String getRecommendationText(
			@WebParam(name = "recommendationId") String recommendationId)
			throws RecommendationServiceException;

	/**
	 * retrieves the recommendation text of a Set of recommendations
	 * 
	 * @param recommendationIds
	 *            the ids of the recommendations
	 * @return a Set of the recommendations' text
	 * @throws RecommendationServiceException
	 * 
	 */
	public List<String> getRecommendations(
			@WebParam(name = "recommendationIds") List<String> recommendationIds)
			throws RecommendationServiceException;

	/**
	 * removes a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @throws RecommendationWebServiceException
	 *             if the recommendation does not exist or cannot be updated
	 */
	public void removeRecommendation(
			@WebParam(name = "recommendationId") String recommendationId)
			throws RecommendationWebServiceException;

	/**
	 * removes a recommendation of type announcement
	 * 
	 * @param announcementId
	 *            the id of the announcement
	 * @throws RecommendationWebServiceException
	 *             if the announcement does not exist or cannot be updated
	 */
	public void removeAnnouncement(
			@WebParam(name = "announcementId") String announcementId)
			throws RecommendationWebServiceException;

	/**
	 * removes a recommendation of type community
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @param communityId
	 *            the id of the community
	 * @throws RecommendationWebServiceException
	 */
	public void removeCommunityRecommendation(
			@WebParam(name = "recommendationId") String recommendationId)
			throws RecommendationWebServiceException;

	public void removeUserRecommendation(
			@WebParam(name = "recommendationId") String recommendationId)
			throws RecommendationWebServiceException;

	/**
	 * updates a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @param recommendationText
	 *            the text of the recommendation
	 * @param creationDate
	 *            the date the recommendation is created
	 * @param expirationDate
	 *            the date the recommendation expires
	 * @throws RecommendationWebServiceException
	 *             if the recommendation does not exist or cannot be updated
	 */
	public void updateRecommendation(
			@WebParam(name = "recommendationId") String recommendationId,
			@WebParam(name = "index") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "title") String title,
			@WebParam(name = "recommendationText") String recommendationText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate)
			throws RecommendationWebServiceException;

	/**
	 * updates an announcement and the corresponding recommendation
	 * 
	 * @param announcementId
	 *            the id of the announcement
	 * @param recommendationText
	 *            the text of the corresponding recommendation
	 * @param creationDate
	 *            the creation date of the announcement
	 * @param expirationDate
	 *            the expiration date of the announcement
	 * @throws RecommendationWebServiceException
	 *             if the announcement recommendation does not exist or cannot
	 *             be updated
	 */
	public void updateAnnouncement(
			@WebParam(name = "announcementId") String announcementId,
			@WebParam(name = "announcementIndex") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "announcementTitle") String announcementTitle,
			@WebParam(name = "announcementText") String announcementText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate)
			throws RecommendationWebServiceException;

	public void updateCommunityRecommendation(
			@WebParam(name = "recommendationId") String recommendationId,
			@WebParam(name = "index") int index,
			@WebParam(name = "isActive") boolean active,
			@WebParam(name = "title") String title,
			@WebParam(name = "recommendationText") String recommendationText,
			@WebParam(name = "creationDate") Date creationDate,
			@WebParam(name = "expirationDate") Date expirationDate,
			@WebParam(name = "communityIds") Set<String> communityIds)
			throws RecommendationWebServiceException;

	/**
	 * sends an e-mail to all users with the recommendation text of an
	 * announcement
	 * 
	 * @param announcementId
	 *            the id of the announcement
	 * @throws MessagingException
	 *             if the e-mail cannot be sent
	 * @throws RecommendationWebServiceException
	 *             if the announcement or recommendation does not exist
	 */
	public void sendAnnouncement(
			@WebParam(name = "announcementId") String announcementId)
			throws RecommendationWebServiceException;

	public void addRecommendationToUser(
			@WebParam(name = "recommendationId") String recommendationId,
			@WebParam(name = "userId") String userId)
			throws RecommendationWebServiceException;

	/**
	 * gets all the community recommendation for a specific user
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<String> getCommunityRecommendationsForUser(
			@WebParam(name = "user") String userId)
			throws RecommendationServiceException;
	

	/**
	 * gets all the community recommendation objects for a specific user
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getCommunityRecommendationsForUserObj(
			@WebParam(name = "user") String userId)
			throws RecommendationServiceException;

}
