package eu.dnetlib.clients.functionality.community.ws;

import java.util.List;
import java.util.Set;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.api.functionality.CommunityService;
import eu.dnetlib.clients.ws.DriverWebService;
import eu.dnetlib.domain.functionality.Community;
import eu.dnetlib.domain.functionality.CommunitySearchCriteria;

/**
 * This interface describes the methods available for a community web service.
 * @author thanos@di.uoa.gr
 *
 */
@WebService
public interface CommunityWebService extends DriverWebService<CommunityService> {
	/**
	 * Create a new community.
	 * @param name the community name
	 * @param ownerId the id of the user that owns the community
	 * @return the id of the newly created community
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "createCommunity")
	public String createCommunity(
			@WebParam(name = "name") String name,
			@WebParam(name = "ownerId") String ownerId
			) throws CommunityWebServiceException;
	
	/**
	 * Update an existing community.
	 * @param id the id of the community to update
	 * @param name the new community name
	 * @param description the new community description
	 * @param owner the new owner of the community
	 * @param managers the new set of community managers
	 * @param collections the new set of community collections
	 * @param recommendations the new set of community recommendations
	 * @param authoritativeDocuments the new set of community authoritative documents
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "updateCommunity")
	public void updateCommunity(
			@WebParam(name = "id") String id,
			@WebParam(name = "name") String name,
			@WebParam(name = "description") String description,
			@WebParam(name = "owner") String owner,
			@WebParam(name = "managers") Set<String> managers,
			@WebParam(name = "collections") Set<String> collections,
			@WebParam(name = "recommendations") Set<String> recommendations,
			@WebParam(name = "authoritativeDocuments") Set<String> authoritativeDocuments
			) throws CommunityWebServiceException;

	/**
	 * Remove an existing community.
	 * @param id the id of the community to remove
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeCommunity") 
	public void removeCommunity(
			@WebParam(name = "id") String id
			) throws CommunityWebServiceException;
	
	/**
	 * Search a community by id.
	 * @param id the id to search for
	 * @return the community that corresponds to the specified id or null if no such community exists
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getCommunity")
	public Community getCommunity(
			@WebParam(name ="id") String id
			) throws CommunityWebServiceException;
	
	/**
	 * Get all the available community ids.
	 * @return a list containing all the available community ids
	 */
	@WebMethod(operationName = "getAllCommunities")
	public List<String> getAllCommunityIds() throws CommunityWebServiceException;
	
	/**
	 * Search communities by criteria.
	 * @param criteria the criteria to search for
	 * @return a list containing all the communities that match the specified criteria
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "searchCommunities")
	public List<Community> searchCommunities(
			@WebParam(name = "criteria") CommunitySearchCriteria criteria
			) throws CommunityWebServiceException;
	
	/**
	 * Search community ids by criteria.
	 * @param criteria the criteria to search for
	 * @return a list containing the ids of all the communities that match the specified criteria
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "searchCommunityIds")
	public List<String> searchCommunityIds(
			@WebParam(name = "criteria") CommunitySearchCriteria criteria
			) throws CommunityWebServiceException;
	
	/**
	 * Add a collection to a community.
	 * @param communityId the id of the community
	 * @param collectionId the id of the collection
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addCollection")
	public void addCollection(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "collectionId") String collectionId
			) throws CommunityWebServiceException;
	
	/**
	 * Remove a collection from a community.
	 * @param communityId the id of the community
	 * @param collectionId the id of the collection
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeCollection")
	public void removeCollection(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "collectionId") String collectionId
			) throws CommunityWebServiceException;
	
	/**
	 * Get all the collections of a community.
	 * @param communityId the id of the community
	 * @return a set containing the ids of all the collections of the specified community
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getCollections")
	public Set<String> getCollections(
			@WebParam(name = "communityId") String communityId
			) throws CommunityWebServiceException;
	
	/**
	 * Add a new user as community manager to this community.
	 * @param communityId the id of the community
	 * @param userId the id of the user
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addManager")
	public void addManager(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "userId") String userId
			) throws CommunityWebServiceException;
	
	/**
	 * Remove a a user as community manager from a community.
	 * @param communityId the id of the community
	 * @param userId the id of the user
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeManager")
	public void removeManager(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "userId") String userId
			) throws CommunityWebServiceException;
	
	/**
	 * Get all the users that are managers of a community. 
	 * @param communityId the id of the community
	 * @return a set containing the ids of all the users that are managers of this community 
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getManagers")
	public Set<String> getManagers(
			@WebParam(name = "communityId") String communityId
			) throws CommunityWebServiceException;
	
	/**
	 * Add a recommendation to a community.
	 * @param communityId the id of the community
	 * @param recommendationId the id of the recommendation
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addRecommendation")
	public void addRecommendation(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "recommendationId") String recommendationId
			) throws CommunityWebServiceException;

	/**
	 * Remove a recommendation from a community.
	 * @param communityId the id of the community
	 * @param recommendationId the id of the recommendation
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "removeRecommendation")
	public void removeRecommendation(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "recommendationId") String recommendationId
			) throws CommunityWebServiceException;
	
	/**
	 * Get all the recommendations of a community.
	 * @param communityId the id of the community
	 * @return a set containing the ids of all the recommendations of the specified community
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getRecommendations")
	public Set<String> getRecommendations(
			@WebParam(name = "communityId") String communityId
			) throws CommunityWebServiceException;
	
	/**
	 * Add an authoritative document to a community.
	 * @param communityId the id of the community
	 * @param documentId the id of the authoritative document
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "addAuthoritativeDocument")
	public void addAuthoritativeDocument(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "documentId") String documentId
			) throws CommunityWebServiceException;
	
	/**
	 * Remove an authoritative document from a community.
	 * @param communityId the id of the community
	 * @param documentId the id of the authoritative document
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName ="removeAuthoritativeDocument")
	public void removeAuthoritativeDocument(
			@WebParam(name = "communityId") String communityId,
			@WebParam(name = "documentId") String documentId
			) throws CommunityWebServiceException;
	
	/**
	 * Get all the authoritative documents of a community.
	 * @param communityId the id of the community
	 * @return a set containing the ids of all the authoritative documents
	 * @throws CommunityWebServiceException if any errors occur
	 */
	@WebMethod(operationName = "getAuthoritativeDocuments")
	public Set<String> getAuthoritativeDocuments(
			@WebParam(name = "communityId") String communityId
			) throws CommunityWebServiceException;
}
