package gr.uoa.di.driver.app;

import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.api.DriverServiceException;
import eu.dnetlib.domain.ActionType;
import eu.dnetlib.domain.EPR;
import eu.dnetlib.domain.ResourceType;
import eu.dnetlib.domain.ServiceIdentity;
import eu.dnetlib.domain.enabling.Notification;
import eu.dnetlib.domain.enabling.Subscription;
import gr.uoa.di.driver.enabling.issn.NotificationListener;
import gr.uoa.di.driver.enabling.issn.SNManager;
import gr.uoa.di.driver.enabling.issn.SubscriptionFactory;

public abstract class DriverServiceImpl implements DriverService {
	private static Logger logger = Logger.getLogger(DriverServiceImpl.class);
	
	private SubscriptionFactory subscriptionFactory = new SubscriptionFactory();
	private volatile EPR serviceEPR = null;
	private ServiceIdentity serviceIdentity = null;
	
	private SNManager snManager = null;
	
	private Map<Subscription, NotificationListener> pendingSubscriptions = new HashMap<Subscription, NotificationListener>();
	
	private volatile boolean inititialized = false;

	public void init() {
		logger.debug("Service identity: " + this.serviceIdentity);

		if (logger.isDebugEnabled())
			logger.debug("Registering pending subscriptions for service " + 
					this.getClass().getSimpleName() + " with epr " + 
					this.getServiceEPR());
		
		for (java.util.Map.Entry<Subscription, NotificationListener> entry:pendingSubscriptions.entrySet()) {
			Subscription sub = entry.getKey();
			NotificationListener listener = entry.getValue();
			
			sub.setEpr(this.getServiceEPR());
				
			snManager.subscribe(sub, listener);
		}
			
		pendingSubscriptions.clear();
		
		this.inititialized = true;
	}
	
	public void subscribe(ActionType actionType, ResourceType resourceType, NotificationListener listener) {
		Subscription sub = subscriptionFactory.createSubscription(actionType,
				resourceType, serviceEPR);
		
		this.addSubscription(sub, listener);		
	}
	
	public void subscribe(ActionType actionType, ResourceType resourceType, 
			String resourceId, NotificationListener listener) {
		Subscription sub = subscriptionFactory.createSubscription(actionType,
				resourceType, resourceId, serviceEPR);
		
		this.addSubscription(sub, listener);		
	}
	
	public void subscribe(ActionType actionType, ResourceType resourceType, 
			String resourceId, String path, NotificationListener listener) {
		Subscription sub = subscriptionFactory.createSubscription(actionType,
				resourceType, resourceId, path, serviceEPR);
		
		this.addSubscription(sub, listener);		
	}
	
	private void addSubscription(Subscription sub, NotificationListener listener) {
		if (!isInititialized()) {
			logger.debug("Service " + getClass().getSimpleName() + " not yet " +
					"initialized, adding subscription to queue.");

			pendingSubscriptions.put(sub, listener);
		} else if (snManager != null) {
			if (logger.isDebugEnabled())
				logger.debug("Adding subscription in service " + 
						getClass().getSimpleName() + " with epr " + getServiceEPR());
			
			this.snManager.subscribe(sub, listener);
		} else {
			logger.warn("Service is initialized, but no subscription manager " +
					"is present! Ignoring subscription.");
		}
	}

	@Override
	public ServiceIdentity identify() {
		return this.serviceIdentity;
	}

	@Override
	public void notify(Notification notification) throws DriverServiceException {
		snManager.notify(notification);
	}

	public boolean isInititialized() {
		return inititialized;
	}
	
	public void setServiceIdentity(ServiceIdentity serviceIdentity) {
		this.serviceIdentity = serviceIdentity;
	}

	public void setServiceEPR(EPR serviceEPR) {
		logger.debug("setting epr " + serviceEPR);
		
		this.serviceEPR = serviceEPR;
	}
	
	public EPR getServiceEPR() {
		return this.serviceEPR;
	}

	public SNManager getSnManager() {
		return snManager;
	}

	public void setSnManager(SNManager snManager) {
		this.snManager = snManager;
	}
}
