package gr.uoa.di.driver.util;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import eu.dnetlib.api.DriverService;

/**
 * An implementation of the service locator that keeps a cache of the service
 * and refreshes it periodically.
 * 
 * The actual task of service discovery is performed by a locator that is 
 * injected. 
 * 
 * @author <a href="mailto:antleb@di.uoa.gr">Antonis Lempesis</a>
 *
 * @param <S> the service type
 */
public class RefreshingServiceLocator<S extends DriverService> implements
		ServiceLocator<S> {
	/** The locator used to discover the service */
	private ServiceLocator<S> locator = null;
	/** the cached instance of the service */
	private S service = null;
	/** the executor that runs the refresh tasks */
	private ScheduledExecutorService executor = null;
	/** the discovery period in milliseconds*/
	private long period = 0;

	@Override
	public synchronized S getService() {
		if (service == null)
			init();
		
		return service;
	}

	/** 
	 * Called to initialize the locator. Performs the first service lookup and
	 * also schedules the periodic refresh of the cache. 
	 * 
	 * */
	private void init() {
		this.service = locator.getService();

		executor.scheduleWithFixedDelay(new RefreshTask(), period, period,
				TimeUnit.MILLISECONDS);
	}

	/**
	 * A very simple runnable that refreshed the cache. No synchronization is
	 * needed during cache refresh: The old instance will be returned while the
	 * locator is working to refresh the cache.
	 * 
	 * @author <a href="mailto:antleb@di.uoa.gr">Antonis Lempesis</a>
	 *
	 */
	private class RefreshTask implements Runnable {
		private Logger logger = Logger.getLogger(RefreshTask.class);

		@Override
		public void run() {
			try {
				logger.debug("Refreshing service cache");
				service = locator.getService();
			} catch (Exception e) {
				logger.info("Error refreshing service", e);
			}
		}
	}

	public ServiceLocator<S> getLocator() {
		return locator;
	}

	public void setLocator(ServiceLocator<S> locator) {
		this.locator = locator;
	}

	public ScheduledExecutorService getExecutor() {
		return executor;
	}

	public void setExecutor(ScheduledExecutorService executor) {
		this.executor = executor;
	}

	public long getPeriod() {
		return period;
	}

	public void setPeriod(long period) {
		this.period = period;
	}
}