package gr.uoa.di.driver.enabling.islookup.cache;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBException;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;

import eu.dnetlib.domain.ActionType;
import eu.dnetlib.domain.DriverResource;
import eu.dnetlib.domain.ResourceType;
import eu.dnetlib.domain.SearchCriteria;
import eu.dnetlib.domain.SecureDriverResource;
import eu.dnetlib.domain.enabling.Notification;
import eu.dnetlib.domain.enabling.SecurityProfile;
import eu.dnetlib.domain.enabling.SecurityProfileSearchCriteria;
import gr.uoa.di.driver.app.DriverServiceImpl;
import gr.uoa.di.driver.enabling.ISLookUp;
import gr.uoa.di.driver.enabling.ISLookUpException;
import gr.uoa.di.driver.enabling.issn.NotificationListener;
import gr.uoa.di.driver.xml.ResourceToXmlConverter;

public class DriverCacheManager<D extends DriverResource> {
	private Logger logger = Logger.getLogger(DriverCacheManager.class);
	
	private ResourceType resourceType = null;
	
	protected DriverServiceImpl service = null;

	private Ehcache idCache = null;
	private Ehcache criteriaCache = null;
	private ResourceToXmlConverter<D> converter = null;
	private ISLookUp<SecurityProfile> securityLookUp = null;
	private Comparator<D> comparator = null;

	public void init() {
		logger.debug("Initializing cache manager for object type " + resourceType);
		
		NotificationListener listener = new CacheNotificationListener();
		
		service.subscribe(ActionType.CREATE, resourceType, listener);
		service.subscribe(ActionType.UPDATE, resourceType, listener);
		service.subscribe(ActionType.DELETE, resourceType, listener);
	}
	
	private void processCreationEvent(Notification notification) {
		try {
			logger.debug("processing creation event");

			String xml = notification.getResource();
			D object = converter.XmlToObject(xml);
			String objectId = object.getResourceId();

			logger.debug("adding object with id: " + objectId);

			if (object instanceof SecureDriverResource) {
				SecurityProfile secProfile = this.getSecurityProfile(object.getResourceId());

				((SecureDriverResource) object).setSecurityProfile(secProfile);
			}

			idCache.put(new Element(objectId, xml));

			this.evaluateQueries(xml, object);
			
			logger.debug("Done processing create event");
		} catch (JAXBException je) {
			logger.error("Error processing create event", je);
		} catch (ISLookUpException ile) {
			logger.error("Error processing create event", ile);
		} catch (Exception e) {
			logger.error("Error processing create event", e);
		}
	}

	private void processUpdateEvent(Notification notification) {
		try {
			logger.debug("processing update event");

			String xml = notification.getResource();
			D object = converter.XmlToObject(xml);

			if (object instanceof SecureDriverResource) {
				logger.debug("profile has security profile");

				SecurityProfile secProfile = this.getSecurityProfile(object
						.getResourceId());

				((SecureDriverResource) object).setSecurityProfile(secProfile);
			}

			idCache.put(new Element(object.getResourceId(), xml));

			this.evaluateQueries(xml, object);
			
			logger.debug("Done processing update event");
		} catch (JAXBException je) {
			logger.error("Error processing update event", je);
		} catch (ISLookUpException ile) {
			logger.error("Error processing update event", ile);
		} catch (Exception e) {
			logger.error("Error processing update event", e);
		}
	}

	private SecurityProfile getSecurityProfile(String profileId) throws ISLookUpException {
		logger.debug("Getting sec profile from IS.");

		SecurityProfileSearchCriteria crit = new SecurityProfileSearchCriteria();

		crit.getDriverResourceIds().add(profileId);

		return securityLookUp.getUniqueResult(crit);
	}

	@SuppressWarnings("unchecked")
	private void processDeletionEvent(Notification notification) {
		try {
			logger.debug("processing deletion event...");
			D object = converter.XmlToObject(notification.getResource());
			String profileId = object.getResourceId();

			logger.debug("removing object with id: " + profileId);
			// remove object from id cache
			idCache.remove(profileId);

			// remove object from each result of a query
			for (Object key : criteriaCache.getKeysWithExpiryCheck()) {
				CriteriaCacheKey cacheKey = (CriteriaCacheKey) key;
				
				if (logger.isDebugEnabled())
					logger.debug("checking query: " + cacheKey.getXQuery());

				List<String> results = (List<String>) criteriaCache.get(key).getObjectValue();

				for (Iterator<String> iter = results.iterator(); iter.hasNext();) {
					String xml = iter.next();
					D result = converter.XmlToObject(xml);

					logger.debug("comparing with result: " + result.getResourceId());

					if (result.getResourceId().equals(profileId)) {
						logger.debug("object found in results. removing...");
						iter.remove();
					}
				}
			}
		} catch (JAXBException e) {
			logger.error("Error processing deletion event", e);
		} catch (Exception e) {
			logger.error("Error processing deletion event", e);
		}

		logger.debug("Done processing deletion event");
	}

	@SuppressWarnings( { "unchecked" })
	private void evaluateQueries(String xml, D object) throws JAXBException {
		String objectId = object.getResourceId();
		List<CriteriaCacheKey> queries = criteriaCache.getKeysWithExpiryCheck();
		
		logger.debug("Will evaluate " + queries.size() + " query(ies)");
		
		for (int i = 0; i < queries.size(); i++) {
			CriteriaCacheKey key = queries.get(i);
			SearchCriteria criteria = key.getCriteria();
			List<String> results = (List<String>) this.criteriaCache.get(key).getObjectValue();

			if (logger.isDebugEnabled())
				logger.debug("testing query: " + key.getXQuery());

			// search for object in results
			for (int j = 0; j < results.size(); j++) {
				String cachedXml = results.get(j);
				D result = converter.XmlToObject(cachedXml);
				String resultId = result.getResourceId();

				logger.debug("comparing with: " + resultId);

				if (objectId.equals(resultId)) {
					// remove result from list. Will add it again later, if
					// needed
					logger.debug("result found. Removing it");
					results.remove(j);

					break;
				}
			}

			// check if object must be included in results
			try {
				if (criteria.matches(object)) {
					logger.debug("results matches. adding it.");
					results.add(xml);

					if (comparator != null) {
						List<CacheEntryPair> pairList = new ArrayList<CacheEntryPair>();
						
						for (String cachedXml:results)
							pairList.add(new CacheEntryPair(cachedXml, converter.XmlToObject(cachedXml)));
						
						Collections.sort(pairList, new CacheEntryPairComparator(comparator));
						
						results.clear();
						for (CacheEntryPair pair:pairList)
							results.add(pair.xml);
					}
				} else {
					logger.debug("Object doesn't match query. Will not add");
				}
			} catch (UnsupportedOperationException uoe) {
				logger.warn("Search criteria have not implememented 'matches' " +
						"method. Clearing query cache");
				
				criteriaCache.removeAll();
				
				break;
			} catch (Exception e) {
				logger.error("Error updating queries", e);
			}
		}
		
		logger.debug("Done evaluating queries");
	}
	
	public void setConverter(ResourceToXmlConverter<D> converter) {
		this.converter = converter;
	}

	public void setIdCache(Ehcache idCache) {
		this.idCache = idCache;
	}

	public Ehcache getIdCache() {
		return idCache;
	}

	public void setSecurityLookUp(ISLookUp<SecurityProfile> securityLookUp) {
		this.securityLookUp = securityLookUp;
	}

	public void setComparator(Comparator<D> comparator) {
		this.comparator = comparator;
	}

	public ResourceType getResourceType() {
		return resourceType;
	}

	public void setResourceType(ResourceType resourceType) {
		this.resourceType = resourceType;
	}

	public Ehcache getCriteriaCache() {
		return criteriaCache;
	}

	public void setCriteriaCache(Ehcache criteriaCache) {
		this.criteriaCache = criteriaCache;
	}

	public void setService(DriverServiceImpl service) {
		this.service = service;
	}
	
	private class CacheEntryPair {
		private String xml = null;
		private D d = null;
		
		public CacheEntryPair(String xml, D d) {
			this.xml = xml;
			this.d = d;
		}
	}
	
	private class CacheEntryPairComparator implements Comparator<CacheEntryPair> {
		private Comparator<D> comparator = null;
		
		public CacheEntryPairComparator(Comparator<D> comparator) {
			this.comparator = comparator;
		}

		@Override
		public int compare(CacheEntryPair o1, CacheEntryPair o2) {
			return this.comparator.compare(o1.d, o2.d);
		}
	}
	
	protected class CacheNotificationListener implements NotificationListener {
		public void processNotification(Notification notification) {

			try {
				if (notification.getResourceType().equals(resourceType)) {
					switch (notification.getActionType()) {
					case CREATE:
						processCreationEvent(notification);
						break;
					case UPDATE:
						processUpdateEvent(notification);
						break;
					case DELETE:
						processDeletionEvent(notification);
						break;
					}
				}
			} catch (Exception e) {
				logger.error("Error updating cache", e);
			}
		}

	}
}
