package gr.uoa.di.driver.enabling.issn;

import eu.dnetlib.api.enabling.ISSNService;
import eu.dnetlib.api.enabling.ISSNServiceException;
import eu.dnetlib.domain.enabling.Subscription;
import gr.uoa.di.driver.util.ServiceLocator;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

/**
 * 
 * @author <a href="mailto:antleb@di.uoa.gr">Antonis Lempesis</a>
 *
 */
public class SubscriptionManagerImpl implements SubscriptionManager {
	private static Logger logger = Logger.getLogger(SubscriptionManagerImpl.class);

	private ServiceLocator<ISSNService> snLocator = null;
	private ScheduledExecutorService executor = null;
	private List<Subscription> refreshList = new ArrayList<Subscription>();

	private int threads = 3;
	private int timeToLive = 3600;

	public void setSnLocator(ServiceLocator<ISSNService> snLocator) {
		this.snLocator = snLocator;
	}

	public void setThreads(int threads) {
		this.threads = threads;
	}

	public void setTimeToLive(int timeToLive) {
		this.timeToLive = timeToLive;
	}

	public void init() {
		logger.debug("Creating thread pool (size: " + threads + ")");
		executor = Executors.newScheduledThreadPool(threads);

		// scheduling a thread to run every timeToLive seconds and
		// refresh all subscriptions. The initial delay is 0.8 of
		// timeToLive is to make sure that it will have enough time to finish
		// working before the first refreshed subscriptions need to be refreshed
		// again.
		logger.debug("Adding refresh task (period: " + timeToLive + " sec)");
		executor.scheduleAtFixedRate(new RefreshTask(),
				(long) (0.8 * timeToLive), timeToLive, TimeUnit.SECONDS);
	}

	public void shutdown() {
		logger.debug("Stopping thread pool");
		executor.shutdownNow();
		
		logger.debug("unsubscribing active subscriptions");
		synchronized (refreshList) {
			for (Subscription sub:refreshList) {
				try {
					snLocator.getService().unsubscribe(sub.getId());
				} catch (ISSNServiceException e) {
					logger.error("Error unsubscibing", e);
				}
			}
		}
	}

	@Override
	public void subscribe(Subscription subscription) {
		this.subscribe(subscription, true);
	}
	
	@Override
	public void subscribe(Subscription subscription, boolean refresh) {
		if (refresh)
			subscription.setTimeToLive(timeToLive);

		logger.debug("Subscribing with ttl " + subscription.getTimeToLive());
		logger.debug("Subscribing epr " + subscription.getEpr());
		logger.debug("Subscribing topic " + subscription.getTopic());
		
		executor.submit(new SubscriptionTask(subscription, refresh));
	}

	@Override
	public void unsubscribe(String subscriptionId) {
		// TODO check if subscription is new (null id) and?
		// either remove from pending list or ignore request

		try {
			snLocator.getService().unsubscribe(subscriptionId);
		} catch (ISSNServiceException e) {
			logger.error("Error unsubscibing", e);
		}

		synchronized (refreshList) {
			for (int i = 0; i < refreshList.size(); i++)
				if (refreshList.get(i).getId().equals(subscriptionId)) {
					refreshList.remove(i);

					break;
				}
		}
	}

	/**
	 * The task that submits a new subscription to the subscription service.
	 * Nothing to see here, move along...
	 * 
	 * @author <a href="mailto:antleb@di.uoa.gr">Antonis Lempesis</a>
	 *
	 */
	private class SubscriptionTask implements Runnable {
		private Subscription subscription = null;
		private boolean addToList = true;
		
		public SubscriptionTask(Subscription subscription, boolean addToList) {
			this.subscription = subscription;
			this.addToList = addToList;
		}

		public void run() {
			try {
				logger.debug("Adding new subscription : " + subscription);
				logger.debug("Epr: " + subscription.getEpr());
				logger.debug("Topic: " + subscription.getTopic());
				
				String subId = snLocator.getService().subscribe(
						subscription.getEpr(), subscription.getTopic(),
						subscription.getTimeToLive());
	
				logger.debug("Subscription id: " + subId);
				subscription.setId(subId);
	
				if (addToList) {
					logger.debug("Adding subscription " + subId + " to refresh list");
					synchronized (refreshList) {
						refreshList.add(0, subscription);
					}
				}
			} catch (Throwable t) {
				logger.error(t);
			}
		}

	}

	/**
	 * A task that refreshes the subscriptions (those that were created by this
	 * manager).
	 * The implementation locks the list of subscriptions which means
	 * that during the refresh process no new subscriptions will be submitted to
	 * the service and no subscriptions can be removed. This is generally not a 
	 * problem because the refresh period is large (typically around 1 hour or
	 * even more) and with the new subscription topic format only few new 
	 * subscriptions are needed.
	 * A possible drawback of this locking is that if the refresh process has not
	 * been completed in the 0.2*timeToLive period, a new thread will be started
	 * for the next period, but it will not be able to proceed...
	 *  
	 * @author <a href="mailto:antleb@di.uoa.gr">Antonis Lempesis</a>
	 *
	 */
	private class RefreshTask implements Runnable {
		public void run() {
			logger.debug("Refreshing subscriptions");
			
			synchronized (refreshList) {
				for (Subscription sub : refreshList) {
					logger.debug("Refreshing subscription with id " + sub.getId());
					try {
						snLocator.getService().renew(sub.getId(), timeToLive);
					} catch (ISSNServiceException e) {
						logger.error("Error renewing subscription", e);
					}
				}
			}
		}
	}
}
