package gr.uoa.di.driver.enabling.registry;

import org.apache.log4j.Logger;

import eu.dnetlib.api.enabling.ISRegistryService;
import eu.dnetlib.api.enabling.ISRegistryServiceException;
import eu.dnetlib.domain.SecureDriverResource;
import eu.dnetlib.domain.enabling.SecurityProfile;
import eu.dnetlib.domain.enabling.SecurityProfileSearchCriteria;
import gr.uoa.di.driver.enabling.ISLookUp;
import gr.uoa.di.driver.enabling.ISLookUpException;
import gr.uoa.di.driver.enabling.ISRegistry;
import gr.uoa.di.driver.enabling.ISRegistryException;
import gr.uoa.di.driver.util.ServiceLocator;

public class SecurityAwareRegistryImpl<R extends SecureDriverResource>
		implements ISRegistry<R> {
	private static Logger logger = Logger.getLogger(SecurityAwareRegistryImpl.class);
	
	private ServiceLocator<ISRegistryService> registryServiceLocator = null;
	private ISRegistry<R> registry = null;
	private ISRegistry<SecurityProfile> securityRegistry = null;
	private ISLookUp<SecurityProfile> securityLookUp = null;

	@Override
	public void delete(R object) throws ISRegistryException {
		securityRegistry.delete(object.getSecurityProfile());
		registry.delete(object);
	}

	@Override
	public R insertForValidation(R object) throws ISRegistryException {
		return registry.insertForValidation(object);
	}

	@Override
	public R save(R object) throws ISRegistryException {
		if (object.getResourceId() == null) {
			return saveNew(object);			
		} else {
			return update(object);
		}
	}
	
	private R update(R object) throws ISRegistryException {
		try {
			SecurityProfile secProfile = object.getSecurityProfile();
	
			// If the user has created a new object and has set the resourceId,
			// the resource will be updated but it's security profile will not
			// have been initialized...
			if (secProfile.getResourceId() == null) {
				SecurityProfile originalSecProfile = getOriginalSecProfile(object.getResourceId());
	
				secProfile.setResourceId(originalSecProfile.getResourceId());
				secProfile.setDateOfCreation(originalSecProfile.getDateOfCreation());
				secProfile.setResourceUri(originalSecProfile.getResourceUri());
				secProfile.setDriverResourceId(originalSecProfile.getDriverResourceId());
			}
			
			logger.debug("Updating profile");
			registry.save(object);
			
			logger.debug("Updating security profile");
			securityRegistry.save(secProfile);
			
			logger.debug("done");
			
			return object;
		} catch (ISLookUpException isle) {
			logger.error(isle);
			throw new ISRegistryException("Error updating profile", isle);
		}
	}
	
	private R saveNew(R object) throws ISRegistryException {
		try {
			logger.debug("Saving new profile");
			logger.debug("registering secure resource: \n");
			// registering resource
			object = registry.insertForValidation(object);
			
			// registering secure profile
			logger.debug("registering secure profile: \n");
			SecurityProfile secProfile = securityRegistry.save(object.getSecurityProfile()); 
			
			// binding profiles
			logger.debug("binding profiles (" + object.getResourceId() + ", " + secProfile.getResourceId() + ")");
			String profileId = registryServiceLocator.getService().registerSecureProfile(object.getResourceId(), secProfile.getResourceId());
			// assigning resource its new id
			// TODO change it when CNR changes implementation
			object.setResourceId(profileId);
			
			// assigning security profile its driver resource id (the foreign key)
			secProfile.setDriverResourceId(profileId);
			
			logger.debug("saving security profile (to save new resource id)");
			// saving security profile (to save new resource id)
			securityRegistry.save(secProfile);
			logger.debug("done");
			
			return object;
		} catch (ISRegistryServiceException e) {
			logger.error(e);
			
			try {
				if (object.getResourceId() != null)
					registry.delete(object);
				
				if (object.getSecurityProfile().getResourceId() != null)
					securityRegistry.delete(object.getSecurityProfile());
			} catch (ISRegistryException isre) {
				logger.error(isre);
			}
			
			throw new ISRegistryException("Error saving new profile", e);
		}

	}

	private SecurityProfile getOriginalSecProfile(String profileId) throws ISLookUpException {
		SecurityProfileSearchCriteria crit = new SecurityProfileSearchCriteria();
		
		crit.getDriverResourceIds().add(profileId);
		
		return securityLookUp.getUniqueResult(crit);
	}
	
	public void setRegistryServiceLocator(
			ServiceLocator<ISRegistryService> registryLocator) {
		this.registryServiceLocator = registryLocator;
	}

	public void setRegistry(ISRegistry<R> registry) {
		this.registry = registry;
	}

	public void setSecurityRegistry(ISRegistry<SecurityProfile> securityRegistry) {
		this.securityRegistry = securityRegistry;
	}

	public void setSecurityLookUp(ISLookUp<SecurityProfile> securityLookUp) {
		this.securityLookUp = securityLookUp;
	}
}
