package eu.dnetlib.uoamonitorservice.controllers;


import eu.dnetlib.uoaadmintoolslibrary.handlers.utils.RolesUtils;
import eu.dnetlib.uoamonitorservice.dao.*;
import eu.dnetlib.uoamonitorservice.entities.*;
import eu.dnetlib.uoamonitorservice.handlers.EntityNotFoundException;
import eu.dnetlib.uoaadmintoolslibrary.handlers.ForbiddenException;
import eu.dnetlib.uoamonitorservice.handlers.PathNotValidException;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.net.URLEncoder;
import java.util.*;

@RestController
@CrossOrigin(origins = "*")
public class IndicatorController {
    private final Logger log = Logger.getLogger(this.getClass());

    @Autowired
    private RolesUtils rolesUtils;

    @Autowired
    private StakeholderDAO stakeholderDAO;

    @Autowired
    private TopicDAO topicDAO;

    @Autowired
    private CategoryDAO categoryDAO;

    @Autowired
    private SubCategoryDAO subCategoryDAO;

    @Autowired
    private SectionDAO sectionDAO;

    @Autowired
    private IndicatorDAO indicatorDAO;

    @Autowired
    private SectionController sectionController;

    @Autowired
    private StakeholderController stakeholderController;

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/save-bulk", method = RequestMethod.POST)
    public Stakeholder saveBulkIndicators(@PathVariable("stakeholderId") String stakeholderId,
                                        @PathVariable("topicId") String topicId,
                                        @PathVariable("categoryId") String categoryId,
                                        @PathVariable("subcategoryId") String subcategoryId,
                                        @RequestBody List<Indicator> indicators) throws UnsupportedEncodingException {
        log.debug("save bulk indicators");
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        Date date = new Date();

        createSectionsAndSaveBulk(date, indicators, stakeholder, topicId, categoryId, subcategoryId);
//        createSectionAndSaveBulk(date, "number", "Numbers imported from file", number_indicators, stakeholder, topicId, categoryId, subcategoryId);

        return stakeholderController.setFullEntities(stakeholder, rolesUtils.getRoles());
    }

    private void createSectionsAndSaveBulk(Date date, List<Indicator> new_indicators,
                                          Stakeholder stakeholder, String topicId, String categoryId, String subcategoryId) throws UnsupportedEncodingException {
        Section chart_section = null;
        Section number_section = null;

        List<String> chart_indicators = null;
        List<String> number_indicators = null;

        for(Indicator indicator : new_indicators) {
            if(indicator.getType().equals("chart")) {
                if(chart_section == null) {
                    chart_section = createSection(chart_section, "chart", "Charts imported from file", date, stakeholder, topicId, categoryId, subcategoryId);
                    chart_indicators = chart_section.getIndicators();
                }
                saveIndicatorAndAddInSection(indicator, date, stakeholder, chart_section, chart_indicators);

            } else if(indicator.getType().equals("number")) {
                if(number_section == null) {
                    number_section = createSection(number_section, "number", "Numbers imported from file", date, stakeholder, topicId, categoryId, subcategoryId);
                    number_indicators = number_section.getIndicators();
                }
                saveIndicatorAndAddInSection(indicator, date, stakeholder, number_section, number_indicators);
            }
        }

        if(chart_section != null) {
            sectionDAO.save(chart_section);
        }
        if(number_section != null) {
            sectionDAO.save(number_section);
        }
    }

    private Section createSection(Section section, String type, String title, Date date,
                                  Stakeholder stakeholder, String topicId, String categoryId, String subcategoryId) {
        section = new Section<>();
        section.setType(type);
        section.setTitle(title);
        section.setStakeholderAlias(stakeholder.getAlias());
        section.setUpdateDate(date);
        section.setCreationDate(date);
        section.setIndicators(new ArrayList<>());
        sectionController.saveSection(stakeholder.getId(), topicId, categoryId, subcategoryId, "-1", section);

        return section;
    }

    private void saveIndicatorAndAddInSection(Indicator indicator, Date date, Stakeholder stakeholder, Section section, List<String> indicators) throws UnsupportedEncodingException {
        // indicator does not exist in DB
        indicator.setCreationDate(date);
        indicator.setUpdateDate(date);

        if (stakeholder.getDefaultId() == null) {   // this indicator belongs in default profile and it is new
            indicatorDAO.save(indicator);
            onSaveDefaultIndicator(indicator, section.getId());
        } else {    // this indicator belongs in a stakeholder's profile and it is new
            indicatorDAO.save(indicator);
        }

        indicators.add(indicator.getId());
        log.debug("Indicator saved!");
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/save", method = RequestMethod.POST)
    public Indicator saveIndicator(@PathVariable("stakeholderId") String stakeholderId,
                                   @PathVariable("topicId") String topicId,
                                   @PathVariable("categoryId") String categoryId,
                                   @PathVariable("subcategoryId") String subcategoryId,
                                   @PathVariable("sectionId") String sectionId,
                                   @RequestBody Indicator indicator) throws UnsupportedEncodingException {
        log.debug("save indicator");
        log.debug("Name: "+indicator.getName() + " - Id: "+indicator.getId() + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId);

        Section<String> section = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator.getType());

        Date date = new Date();
        indicator.setUpdateDate(date);

        Indicator oldIndicator = null;
        if(indicator.getId() != null) {
            oldIndicator = indicatorDAO.findById(indicator.getId());
            if(oldIndicator == null) {
                // EXCEPTION - Indicator not found
                throw new EntityNotFoundException("save indicator: Indicator with id: " + indicator.getId() + " not found");
            }
        } else { // indicator does not exist in DB
            indicator.setCreationDate(date);
        }

        String indicatorId = indicator.getId();

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
        // this indicator belongs in default profile and it is new or it is updated
        if(stakeholder.getDefaultId() == null) {
            if(indicatorId == null) {
                indicatorDAO.save(indicator);
                onSaveDefaultIndicator(indicator, sectionId);
            }
            else {
                onUpdateDefaultIndicator(indicator, stakeholder, oldIndicator);
                indicatorDAO.save(indicator);
            }
        } else {
            indicatorDAO.save(indicator);
        }

        List<String> indicators = section.getIndicators();

        int index = indicators.indexOf(indicator.getId());
        if (index == -1) {
            indicators.add(indicator.getId());
            sectionDAO.save(section);
            log.debug("Indicator saved!");
        }

        return indicator;
    }

    public void onSaveDefaultIndicator(Indicator indicator, String defaultSectionId) throws UnsupportedEncodingException {
        log.debug("On save default indicator");

        // new indicator in default profile - add it on profiles of the same type
        List<Section> sections = sectionDAO.findByDefaultId(defaultSectionId);

       for (Section section : sections) {
            Indicator indicatorNew = new Indicator();
            indicatorNew.copyFromDefault(indicator);
            for (IndicatorPath indicatorPath : indicatorNew.getIndicatorPaths()) {
                Stakeholder stakeholder = stakeholderDAO.findByAlias(section.getStakeholderAlias());
                parameterMapping(indicatorPath, stakeholder);
            }

            indicatorDAO.save(indicatorNew);

            List<String> indicators = section.getIndicators();
            indicators.add(indicatorNew.getId());

            sectionDAO.save(section);
        }
    }

    public void onUpdateDefaultIndicator(Indicator indicator, Stakeholder stakeholder, Indicator oldIndicator) throws UnsupportedEncodingException {
        log.debug("On update default indicator");

        // indicator already exists - check if changed and update all indicators based on it

        boolean changed;
        List<Indicator> indicators = indicatorDAO.findByDefaultId(indicator.getId());

        for(Indicator indicatorBasedOnDefault : indicators) {
            changed = false;

//            if(indicator.getName() != null && !indicator.getName().equals(indicatorBasedOnDefault.getName())
//                    && (oldIndicator.getName() == null || oldIndicator.getName().equals(indicatorBasedOnDefault.getName()))) {
            if((
                    (indicator.getName() == null && oldIndicator.getName() != null)
                            ||
                            (indicator.getName() != null && !indicator.getName().equals(indicatorBasedOnDefault.getName()))
            ) && (
                    (oldIndicator.getName() == null && indicatorBasedOnDefault.getName() == null)
                            ||
                            (oldIndicator.getName() != null && oldIndicator.getName().equals(indicatorBasedOnDefault.getName()))
            )) {
                indicatorBasedOnDefault.setName(indicator.getName());
                changed = true;
            }

            if(indicator.getDescription() != null && !indicator.getDescription().equals(indicatorBasedOnDefault.getDescription())
                || indicator.getDescription() == null && indicatorBasedOnDefault.getDescription() != null) {

                indicatorBasedOnDefault.setDescription(indicator.getDescription());
                changed = true;
            }

//            if(indicator.getAdditionalDescription() != null && !indicator.getAdditionalDescription().equals(indicatorBasedOnDefault.getAdditionalDescription())
//                    && (oldIndicator.getAdditionalDescription() == null || oldIndicator.getAdditionalDescription().equals(indicatorBasedOnDefault.getAdditionalDescription()))) {
            if((
                    (indicator.getAdditionalDescription() == null && oldIndicator.getAdditionalDescription() != null)
                            ||
                            (indicator.getAdditionalDescription() != null && !indicator.getAdditionalDescription().equals(indicatorBasedOnDefault.getAdditionalDescription()))
            ) && (
                    (oldIndicator.getAdditionalDescription() == null && indicatorBasedOnDefault.getAdditionalDescription() == null)
                            ||
                            (oldIndicator.getAdditionalDescription() != null && oldIndicator.getAdditionalDescription().equals(indicatorBasedOnDefault.getAdditionalDescription()))
            )) {
                indicatorBasedOnDefault.setAdditionalDescription(indicator.getAdditionalDescription());
                changed = true;
            }

            int i = 0;
            List<IndicatorPath> indicatorPaths = indicatorBasedOnDefault.getIndicatorPaths();
            if(indicatorPaths == null && indicator.getIndicatorPaths() != null) {
                indicatorPaths = new ArrayList<>();
            }

            for (IndicatorPath indicatorPath : indicator.getIndicatorPaths()) {
                IndicatorPath indicatorPathBasedOnDefault = null;
                if(i < indicatorPaths.size()) {
                    indicatorPathBasedOnDefault = indicatorPaths.get(i);
                }

                if(indicatorPathBasedOnDefault == null) {
                    // Add new indicator path in existing indicators
                    IndicatorPath indicatorPathNew = new IndicatorPath(indicatorPath);
                    parameterMapping(indicatorPathNew, stakeholder);
                    indicatorPaths.add(indicatorPathNew);
                    changed = true;
                } else {
                    IndicatorPath oldIndicatorPath = oldIndicator.getIndicatorPaths().get(i);

                    // Check if there are changes in indicator path and update existing indicators if needed
                    log.debug("update indicator path: "+i + " (indicator id: "+indicatorBasedOnDefault.getId()+")");

//                    if(indicatorPath.getType() != null
//                            && !indicatorPath.getType().equals(indicatorPathBasedOnDefault.getType())
//                            && (oldIndicatorPath.getType().equals(indicatorPathBasedOnDefault.getType()))) {
                    if((
                            (indicatorPath.getType() == null && oldIndicatorPath.getType() != null)
                                    ||
                                    (indicatorPath.getType() != null && !indicatorPath.getType().equals(indicatorPathBasedOnDefault.getType()))
                    ) && (
                            (oldIndicatorPath.getType() == null && indicatorPathBasedOnDefault.getType() == null)
                                    ||
                                    (oldIndicatorPath.getType() != null && oldIndicatorPath.getType().equals(indicatorPathBasedOnDefault.getType()))
                    )) {
                        indicatorPathBasedOnDefault.setType(indicatorPath.getType());
                        changed = true; // parameter "type" needs to be changed as well
                    }
                    log.debug("After type check: "+changed);

//                    if(indicatorPath.getSource() != null
//                            && !indicatorPath.getSource().equals(indicatorPathBasedOnDefault.getSource())
//                            && (oldIndicatorPath.getSource().equals(indicatorPathBasedOnDefault.getSource()))) {
                    if((
                            (indicatorPath.getSource() == null && oldIndicatorPath.getSource() != null)
                                    ||
                                    (indicatorPath.getSource() != null && !indicatorPath.getSource().equals(indicatorPathBasedOnDefault.getSource()))
                    ) && (
                            (oldIndicatorPath.getSource() == null && indicatorPathBasedOnDefault.getSource() == null)
                                    ||
                                    (oldIndicatorPath.getSource() != null && oldIndicatorPath.getSource().equals(indicatorPathBasedOnDefault.getSource()))
                    )) {
                        indicatorPathBasedOnDefault.setSource(indicatorPath.getSource());
                        changed = true;
                    }
                    log.debug("After source check: "+changed);

//                    if(indicatorPath.getUrl() != null
//                            && !indicatorPath.getUrl().equals(indicatorPathBasedOnDefault.getUrl())
//                            && (oldIndicatorPath.getUrl().equals(indicatorPathBasedOnDefault.getUrl()))) {
                    if((
                            (indicatorPath.getUrl() == null && oldIndicatorPath.getUrl() != null)
                                    ||
                                    (indicatorPath.getUrl() != null && !indicatorPath.getUrl().equals(indicatorPathBasedOnDefault.getUrl()))
                    ) && (
                            (oldIndicatorPath.getUrl() == null && indicatorPathBasedOnDefault.getUrl() == null)
                                    ||
                                    (oldIndicatorPath.getUrl() != null && oldIndicatorPath.getUrl().equals(indicatorPathBasedOnDefault.getUrl()))
                    )) {
                        indicatorPathBasedOnDefault.setUrl(indicatorPath.getUrl());
                        changed = true;
                    }
                    log.debug("After url check: "+changed);

                    if((
                            (indicatorPath.getChartObject() == null && oldIndicatorPath.getChartObject() != null)
                            ||
                            (indicatorPath.getChartObject() != null && !indicatorPath.getChartObject().equals(indicatorPathBasedOnDefault.getChartObject()))
                        ) && (
                            (oldIndicatorPath.getChartObject() == null && indicatorPathBasedOnDefault.getChartObject() == null)
                            ||
                            (oldIndicatorPath.getChartObject() != null && oldIndicatorPath.getChartObject().equals(indicatorPathBasedOnDefault.getChartObject()))
                    )) {

                        indicatorPathBasedOnDefault.setChartObject(indicatorPath.getChartObject());
                        changed = true;
                    }
                    log.debug("After chartObject check: "+changed);

                    if(indicatorPath.getParameters() != null) {
                        if (indicatorPathBasedOnDefault.getParameters() == null) {
                            indicatorPathBasedOnDefault.setParameters(new HashMap<>());
                        }
                        //if (indicatorPath.getParameters().size() != indicatorPathBasedOnDefault.getParameters().size()) {
                            //log.debug("Different number of parameters");
                        for (Map.Entry<String, String> parameter : indicatorPath.getParameters().entrySet()) {
                            log.debug("\nindicatorPath: parameter.getKey(): "+parameter.getKey()+" - value: "+parameter.getValue()
                                    +"\nindicatorPathBasedOnDefault:parameters:key: "+  indicatorPathBasedOnDefault.getParameters().get(parameter.getKey())
                                    +"\noldIndicatorPath:parameters:key: "+  (oldIndicatorPath.getParameters() == null ? "null" : oldIndicatorPath.getParameters().get(parameter.getKey())));
                            if (!indicatorPathBasedOnDefault.getParameters().containsKey(parameter.getKey())
                                    || (oldIndicatorPath.getParameters() == null || oldIndicatorPath.getParameters().get(parameter.getKey()) == null
                                        || (oldIndicatorPath.getParameters().get(parameter.getKey()).equals(indicatorPathBasedOnDefault.getParameters().get(parameter.getKey()))
                                            && !parameter.getValue().equals(indicatorPathBasedOnDefault.getParameters().get(parameter.getKey()))))
                            ) {
                                indicatorPathBasedOnDefault.getParameters().put(parameter.getKey(), parameter.getValue());
                                changed = true;
                            }
//                            else if(parameter.getKey().equals("type")) {
//                                indicatorPathBasedOnDefault.getParameters().put(parameter.getKey(), parameter.getValue());
//                                changed = true;
//                            }
                        }

                        // When deleting indicator path parameters in a default profile, delete them also from all children profiles
                        if(oldIndicatorPath.getParameters() != null && indicatorPath.getParameters().size() < oldIndicatorPath.getParameters().size()) {
                            for (Map.Entry<String, String> parameter : oldIndicatorPath.getParameters().entrySet()) {
                                if(!indicatorPath.getParameters().containsKey(parameter.getKey())) {
                                    indicatorPathBasedOnDefault.getParameters().remove(parameter.getKey());
                                }
                            }
                        }
                        parameterMapping(indicatorPathBasedOnDefault, stakeholder);
                        //}
                    }
                    log.debug("After parameters check: " + changed);

                    if(indicatorPath.getJsonPath() != null) {
                        boolean jsonPathChanged = false;
                        boolean breaked = false;

                        int oldJsonPathSize = 0;
                        if(oldIndicatorPath.getJsonPath() != null) {
                            oldJsonPathSize = oldIndicatorPath.getJsonPath().size();
                        }
                        int basedOnDefaultJsonPathSize = 0;
                        if(indicatorPathBasedOnDefault.getJsonPath() != null) {
                            basedOnDefaultJsonPathSize = indicatorPathBasedOnDefault.getJsonPath().size();
                        }
                        log.debug("old: "+oldJsonPathSize+" - based on default: "+basedOnDefaultJsonPathSize+" - new: "+indicatorPath.getJsonPath().size());
                        if(oldJsonPathSize == basedOnDefaultJsonPathSize) {
                            if(indicatorPathBasedOnDefault.getJsonPath() == null && indicatorPath.getJsonPath().size() > 0) {
                                indicatorPathBasedOnDefault.setJsonPath(new ArrayList<>());
                            }

                            int basedOnDefaultIndex = 0;
                            int oldIndex = 0;

                            Iterator<String> jsonStringBasedOnDefaultIterator = indicatorPathBasedOnDefault.getJsonPath().iterator();
                            while (jsonStringBasedOnDefaultIterator.hasNext()) {
                                String jsonStringBasedOnDefault = jsonStringBasedOnDefaultIterator.next();
                                if(oldIndicatorPath.getJsonPath().get(oldIndex).equals(jsonStringBasedOnDefault)) {
                                    if(basedOnDefaultIndex >= indicatorPath.getJsonPath().size()) { // string deleted
                                        jsonStringBasedOnDefaultIterator.remove();
                                        jsonPathChanged = true;
                                    } else {    // check if string changed
                                        if(!indicatorPath.getJsonPath().get(basedOnDefaultIndex).equals(jsonStringBasedOnDefault)) {
                                            indicatorPathBasedOnDefault.getJsonPath().set(basedOnDefaultIndex, indicatorPath.getJsonPath().get(basedOnDefaultIndex));
                                            jsonPathChanged = true;
                                        }
                                        basedOnDefaultIndex++;
                                    }
                                    oldIndex++;
                                } else {
                                    breaked = true;
                                    jsonPathChanged = false;
                                    log.debug("not the same: "+oldIndex);
                                    break;
                                }
                            }

                            int index=0;
                            if(!breaked && indicatorPath.getJsonPath().size() > indicatorPathBasedOnDefault.getJsonPath().size()) { // strings added
                                jsonPathChanged = true;
                                for(index=indicatorPathBasedOnDefault.getJsonPath().size(); index < indicatorPath.getJsonPath().size(); index++) {
                                    indicatorPathBasedOnDefault.getJsonPath().add(indicatorPath.getJsonPath().get(index));
                                }
                            }

                            if(jsonPathChanged) {
                                changed = true;
                            }
                        }
                        // TODO when deleting indicator path json path strings... --> is this done? (line 327)
                    }
                    log.debug("After jsonPath check: " + changed);
                }
                i++;
            }
            // TODO when deleting indicator paths...

            if(!changed) {
//                break;
                continue;
            }

            indicatorBasedOnDefault.setUpdateDate(indicator.getUpdateDate());
            indicatorDAO.save(indicatorBasedOnDefault);
        }
    }

    public void parameterMapping(IndicatorPath indicatorPath, Stakeholder stakeholder) throws UnsupportedEncodingException {
        if (indicatorPath.getParameters() != null) {
            if (indicatorPath.getParameters().containsKey("index_name")) {
                indicatorPath.getParameters().put("index_name", stakeholder.getIndex_name());
            } else if (indicatorPath.getParameters().containsKey("index_shortName")) {
                indicatorPath.getParameters().put("index_shortName", stakeholder.getIndex_name().toLowerCase());
            } else if (indicatorPath.getParameters().containsKey("index_id")) {
                indicatorPath.getParameters().put("index_id", stakeholder.getIndex_id());
            }
        }

//        // url encoding for number indicators
//        String url = indicatorPath.getUrl();
//        String encoded_index_id = urlEncode(URLEncoder.encode(stakeholder.getIndex_id(), "UTF-8"));
//        url = url.replace("index_id", encoded_index_id);
//        String encoded_index_name = urlEncode(URLEncoder.encode(stakeholder.getIndex_name(), "UTF-8"));
//        url = url.replace("index_name", encoded_index_name);
//        String encoded_index_shortName = urlEncode(URLEncoder.encode(stakeholder.getIndex_shortName(), "UTF-8"));
//        url = url.replace("index_shortName", encoded_index_shortName);
//        indicatorPath.setUrl(url);
    }

    public String urlEncode(String encodedIndicatorPathField) {
        String indicatorPathField = "";

        for( int i=0; i<encodedIndicatorPathField.length(); i++ ){
            String character = encodedIndicatorPathField.substring(i, i+1);

            if(character.equals("+")) {
                indicatorPathField = indicatorPathField.concat("%20");
            } else if(character.equals("%")) {
                //grab the hex in pairs
                String output = encodedIndicatorPathField.substring(i+1, (i + 3));

                if(output.equals("7E") || output.equals("27") || output.equals("28") || output.equals("29") || output.equals("21")) {
                    //convert hex to decimal
                    int decimal = Integer.parseInt(output, 16);
                    //convert the decimal to character
                    StringBuilder sb = new StringBuilder();
                    sb.append((char) decimal);

                    indicatorPathField = indicatorPathField.concat(sb.toString());
                } else {
                    indicatorPathField = indicatorPathField.concat(character + output);
                }

                i += 2;
            } else {
                indicatorPathField = indicatorPathField.concat(character);
            }
        }

        return indicatorPathField;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/{indicatorId}/delete", method = RequestMethod.DELETE)
    public boolean deleteIndicator(@PathVariable("stakeholderId") String stakeholderId,
                                   @PathVariable("topicId") String topicId,
                                   @PathVariable("categoryId") String categoryId,
                                   @PathVariable("subcategoryId") String subcategoryId,
                                   @PathVariable("sectionId") String sectionId,
                                   @PathVariable("indicatorId") String indicatorId,
                                   @RequestParam(required = false) String children) {
        log.debug("delete indicator");
        log.debug("Id: "+indicatorId + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId);

        Indicator indicator = indicatorDAO.findById(indicatorId);
        if(indicator != null) {
            Section<String> section = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator.getType());

            Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
            List<String> roles = rolesUtils.getRoles();
            if(indicator.getDefaultId() != null && !rolesUtils.hasCreateAndDeleteAuthority(roles, stakeholder.getType())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Delete indicator: You are not authorized to delete a default Indicator in stakeholder with id: "+stakeholderId);
            }

            List<String> indicators = section.getIndicators();

            int index = indicators.indexOf(indicatorId);
            if (index != -1) {

                // this indicator belongs in default profile
                if(section.getDefaultId() == null && children != null) {
                    onDeleteDefaultIndicator(indicatorId, sectionId, children);
                }


                indicators.remove(index);
                sectionDAO.save(section);

                indicatorDAO.delete(indicatorId);
                log.debug("Indicator deleted!");
            } else {
                // EXCEPTION - Indicator not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias(); -> Section: section.getTitle();
                throw new PathNotValidException("Delete indicator: Indicator with id: "+indicatorId+" not found in Sectiom: "+sectionId);
            }
        } else {
            // EXCEPTION - Indicator not found
            throw new EntityNotFoundException("Delete indicator: Indicator with id: "+indicatorId+" not found");
        }
        return true;
    }

    public boolean onDeleteDefaultIndicator(String defaultIndicatorId, String defaultSectionId, String children) {
        if(children.equals("delete")) {
//            // 1st way
//            List<Section> sections = sectionDAO.findByDefaultId(defaultSectionId);
//
//            for(Section section : sections) {
//                List<String> indicators = section.getIndicators();
//
//                Iterator<String> indicatorsIterator = indicators.iterator();
//                while(indicatorsIterator.hasNext()) {
//                    String indicatorId = indicatorsIterator.next();
//
//                    Indicator indicator = indicatorDAO.findById(indicatorId);
//                    if (indicator.getDefaultId().equals(defaultIndicatorId)) {
//                        indicatorsIterator.remove();
//                        sectionDAO.save(section);
//
//                        indicatorDAO.delete(indicatorId);
//                        log.debug("Indicator deleted!");
//
//                        break;
//                    }
//                }
//            }

            // 2nd way
            List<Section> sections = sectionDAO.findByDefaultId(defaultSectionId);
            List<Indicator> indicators = indicatorDAO.findByDefaultId(defaultIndicatorId);

            for(Section section : sections) {
                Iterator<Indicator> indicatorsIterator = indicators.iterator();
                while(indicatorsIterator.hasNext()) {
                    String indicatorId = indicatorsIterator.next().getId();
                    if(section.getIndicators().contains(indicatorId)) {
                        indicatorsIterator.remove();

                        section.getIndicators().remove(indicatorId);
                        sectionDAO.save(section);

                        indicatorDAO.delete(indicatorId);
                        log.debug("Indicator with id: "+indicatorId+" deleted!");

                        break;
                    }
                }
            }

//            // 3rd way - parentId
//            List<Indicator> indicators = indicatorDAO.findByDefaultId(defaultIndicatorId);
//            for(Indicator indicator : indicators) {
//                Section section = sectionDAO.findById(indicator.getParent());
//                List<String> sectionIndicators = section.getIndicators();
//
//                sectionIndicators.remove(indicator.getId());
//                sectionDAO.save(section);
//
//                indicatorDAO.delete(indicator.getId());
//                log.debug("Indicator deleted!");
//            }
        } else if(children.equals("disconnect")) {
            List<Indicator> indicators = indicatorDAO.findByDefaultId(defaultIndicatorId);
            for(Indicator indicator : indicators) {
                indicator.setDefaultId(null);
                indicatorDAO.save(indicator);
                log.debug("DefaultId for Indicator with id: "+indicator.getId()+" empty!");
            }
        }
        return true;
    }

//    @RequestMapping(value = "/{stakeholderId}/charts/delete", method = RequestMethod.DELETE)
//    public boolean deleteAllChartIndicators(@PathVariable("stakeholderId") String stakeholderId) {
//        log.debug("delete all chart indicators of stakeholder");
//        log.debug("Stakeholder: "+stakeholderId);
//
//        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
//        if(stakeholder != null) {
//
//            for(String topicId : stakeholder.getTopics()) {
//                Topic<String> topic = topicDAO.findById(topicId);
//                if(topic != null) {
//                    for(String categoryId : topic.getCategories()) {
//                        Category<String> category = categoryDAO.findById(categoryId);
//                        if(category != null) {
//                            for(String subcategoryId : category.getSubCategories()) {
//                                SubCategory<String> subcategory = subCategoryDAO.findById(subcategoryId);
//                                if(subcategory != null) {
//
//                                    for(String sectionId : subcategory.getCharts()) {
//                                        Section<String> section = sectionDAO.findById(sectionId);
//                                        if (section != null) {
//
//                                            List<String> indicators = section.getIndicators();
//                                            Iterator<String> indicatorsIterator = section.getIndicators().iterator();
//
//                                            while (indicatorsIterator.hasNext()) {
//                                                String indicatorId = indicatorsIterator.next();
//                                                Indicator indicator = indicatorDAO.findById(indicatorId);
//                                                if (indicator != null) {
//                                                    int index = indicators.indexOf(indicatorId);
//                                                    if (index != -1) {
//                                                        indicatorsIterator.remove();
//                                                        //indicators.remove(index);
//
//                                                        indicatorDAO.delete(indicatorId);
//                                                        log.debug("Indicator deleted!");
//                                                    } else {
//                                                        // EXCEPTION - Indicator not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias(); -> Section: section.getTitle();
//                                                        throw new PathNotValidException("Delete indicator: Indicator with id: " + indicatorId + " not found in Section: " + sectionId);
//                                                    }
//                                                } else {
//                                                    // EXCEPTION - Indicator not found
//                                                    throw new EntityNotFoundException("Delete indicator: Indicator with id: " + indicatorId + " not found");
//                                                }
//                                            }
//                                            sectionDAO.save(section);
//                                        } else {
//                                            // EXCEPTION - Section not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias();
//                                            throw new PathNotValidException("Delete indicator: Section with id: " + sectionId + " not found in SubCategory: " + subcategoryId);
//                                        }
//                                    }
//                                } else {
//                                    // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
//                                    throw new PathNotValidException("Delete indicator: SubCategory with id: "+subcategoryId+" not found in Category: "+categoryId);
//                                }
//                            }
//                        } else {
//                            // EXCEPTION - Category not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias();
//                            throw new PathNotValidException("Delete indicator: Category with id: "+categoryId+" not found in Topic: "+topicId);
//                        }
//                    }
//                } else {
//                    // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
//                    throw new PathNotValidException("Delete indicator: Topic with id: "+topicId+" not found in Stakeholder: "+stakeholderId);
//                }
//            }
//        } else {
//            // EXCEPTION - Stakeholder not found
//            throw new EntityNotFoundException("Delete indicator: Stakeholder with id: "+stakeholderId+" not found");
//        }
//        return true;
//    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/{type}/reorder", method = RequestMethod.POST)
    public List<Indicator> reorderIndicators(@PathVariable("stakeholderId") String stakeholderId,
                                             @PathVariable("topicId") String topicId,
                                             @PathVariable("categoryId") String categoryId,
                                             @PathVariable("subcategoryId") String subcategoryId,
                                             @PathVariable("sectionId") String sectionId,
                                             @PathVariable("type") String type,
                                             @RequestBody ReorderEvent reorderEvent) {
        log.debug("reorder indicators of type: "+type);
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId);

        List<String> indicators = reorderEvent.getIds();
        String actionType = reorderEvent.getAction();
        String targetId = reorderEvent.getTarget();

        Section<String> section = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId, sectionId, type);

        List<String> oldIndicators = section.getIndicators();
        for (String indicatorId : oldIndicators) {
            if ((!actionType.equals("removed") || !targetId.equals(indicatorId)) && !indicators.contains(indicatorId)) {
                indicators.add(indicatorId);
            }
        }
        section.setIndicators(indicators);

        List<Indicator> indicatorsFull = new ArrayList<>();
        for(String indicatorId : indicators) {
            Indicator indicator = indicatorDAO.findById(indicatorId);
            if(indicator == null) {
                // EXCEPTION - Indicator not found
                throw new EntityNotFoundException("Reorder indicators: Indicator with id: " + indicatorId + " not found");
            }
            indicatorsFull.add(indicator);
        }

        sectionDAO.save(section);
        log.debug("Indicators reordered!");

        return indicatorsFull;
    }

//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/{indicatorId}/toggle-status", method = RequestMethod.POST)
//    public Boolean toggleIndicatorStatus(@PathVariable("stakeholderId") String stakeholderId,
//                                         @PathVariable("topicId") String topicId,
//                                         @PathVariable("categoryId") String categoryId,
//                                         @PathVariable("subcategoryId") String subcategoryId,
//                                         @PathVariable("sectionId") String sectionId,
//                                         @PathVariable("indicatorId") String indicatorId) {
//        log.debug("toggle indicator status (isActive)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId+ " - Indicator: "+indicatorId);
//
//        Indicator indicator = indicatorDAO.findById(indicatorId);
//        if (indicator == null) {
//            // EXCEPTION - Indicator not found
//            throw new EntityNotFoundException("Toggle indicator status: Indicator with id: "+indicatorId+" not found");
//        }
//        indicator.setIsActive(!indicator.getIsActive());
//
//        this.toggleIndicator(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator);
//
//        return indicator.getIsActive();
//    }
//
//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/{indicatorId}/toggle-access", method = RequestMethod.POST)
//    public Boolean toggleIndicatorAccess(@PathVariable("stakeholderId") String stakeholderId,
//                                         @PathVariable("topicId") String topicId,
//                                         @PathVariable("categoryId") String categoryId,
//                                         @PathVariable("subcategoryId") String subcategoryId,
//                                         @PathVariable("sectionId") String sectionId,
//                                         @PathVariable("indicatorId") String indicatorId) {
//        log.debug("toggle indicator access (isPublic)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId+ " - Indicator: "+indicatorId);
//
//        Indicator indicator = indicatorDAO.findById(indicatorId);
//        if (indicator == null) {
//            // EXCEPTION - Indicator not found
//            throw new EntityNotFoundException("Toggle indicator access: Indicator with id: "+indicatorId+" not found");
//        }
//        indicator.setIsPublic(!indicator.getIsPublic());
//
//        this.toggleIndicator(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator);
//
//        return indicator.getIsPublic();
//    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/{indicatorId}/change-visibility", method = RequestMethod.POST)
    public Visibility changeTopicVisibility(@PathVariable("stakeholderId") String stakeholderId,
                                            @PathVariable("topicId") String topicId,
                                            @PathVariable("categoryId") String categoryId,
                                            @PathVariable("subcategoryId") String subcategoryId,
                                            @PathVariable("sectionId") String sectionId,
                                         @PathVariable("indicatorId") String indicatorId,
                                            @RequestParam("visibility") Visibility visibility) {
        log.debug("change indicator visibility: "+visibility);
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId + " - Section: "+sectionId+ " - Indicator: "+indicatorId);

        Indicator indicator = indicatorDAO.findById(indicatorId);
        if (indicator == null) {
            // EXCEPTION - Indicator not found
            throw new EntityNotFoundException("Change indicator visibility: Indicator with id: "+indicatorId+" not found");
        }
        indicator.setVisibility(visibility);

        this.toggleIndicator(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator);

        return indicator.getVisibility();
    }

    public void toggleIndicator(String stakeholderId, String topicId, String categoryId, String subcategoryId, String sectionId, Indicator indicator) {
        Section<String> section = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId, sectionId, indicator.getType());
        List<String> indicators = section.getIndicators();

        if(indicators.contains(indicator.getId())) {
            indicatorDAO.save(indicator);
            log.debug("Indicator toggled!");
        } else {
            // EXCEPTION - Indicator not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subCategory.getAlias(); -> Section: section.getTitle();
            throw new PathNotValidException("Toggle indicators: Indicator with id: "+indicator.getId()+" not found in Section: "+sectionId);
        }

    }

    private Section checkForExceptions(String stakeholderId, String topicId, String categoryId, String subcategoryId, String sectionId, String indicatorType) {

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder == null) {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Save indicator: Stakeholder with id: " + stakeholderId + " not found");
        }

        List<String> roles = rolesUtils.getRoles();
        if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
            // EXCEPTION - Access denied
            throw new ForbiddenException("CheckForExceptions Indicator: You are not authorized to update stakeholder with id: "+stakeholderId);
        }

        Topic<String> topic = topicDAO.findById(topicId);
        if(topic == null) {
            // EXCEPTION - Topic not found
            throw new EntityNotFoundException("Save indicator: Topic with id: "+topicId+" not found");
        }

        if(!stakeholder.getTopics().contains(topicId)) {
            // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
            throw new PathNotValidException("Save indicator: Topic with id: " + topicId + " not found in Stakeholder: " + stakeholderId);
        }

        Category<String> category = categoryDAO.findById(categoryId);
        if(category == null) {
            // EXCEPTION - Category not found
            throw new EntityNotFoundException("Save indicator: Category with id: "+categoryId+" not found");
        }

        if(!topic.getCategories().contains(categoryId)) {
            // EXCEPTION - Category not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias();
            throw new PathNotValidException("Save indicator: Category with id: "+categoryId+" not found in Topic: "+topicId);
        }

        SubCategory<String> subcategory = subCategoryDAO.findById(subcategoryId);
        if(subcategory == null) {
            // EXCEPTION - SubCategory not found
            throw new EntityNotFoundException("Save indicator: SubCategory with id: "+subcategoryId+" not found");
        }

        if (!category.getSubCategories().contains(subcategoryId)) {
            // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
            throw new PathNotValidException("Save indicator: SubCategory with id: "+subcategoryId+" not found in Category: "+categoryId);
        }

        Section<String> section = sectionDAO.findById(sectionId);
        if(section == null) {
            // EXCEPTION - Section not found
            throw new EntityNotFoundException("Save indicator: Section with id: "+sectionId+" not found");
        }

        if(indicatorType.equals("chart")) {
            if (!subcategory.getCharts().contains(sectionId)) {
                // EXCEPTION - Section not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias();
                throw new PathNotValidException("Save indicator: SubCategory with id: " + subcategoryId + " not found in Category: " + categoryId);
            }
        } else if(indicatorType.equals("number")) {
            if (!subcategory.getNumbers().contains(sectionId)) {
                // EXCEPTION - Section not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias();
                throw new PathNotValidException("Save indicator: SubCategory with id: " + subcategoryId + " not found in Category: " + categoryId);
            }
        }

        return  section;
    }

    public void deleteTree(Section section) {
        List<String> indicators = section.getIndicators();
        for(String indicatorId : indicators) {
            indicatorDAO.delete(indicatorId);
        }
    }

    public void disConnectTree(Section section) {
        List<String> indicators = section.getIndicators();
        for(String indicatorId : indicators) {
            Indicator indicator = indicatorDAO.findById(indicatorId);
            indicator.setDefaultId(null);
            indicatorDAO.save(indicator);
        }
    }
}
