package eu.dnetlib.uoamonitorservice.controllers;

import eu.dnetlib.uoaadmintoolslibrary.handlers.utils.RolesUtils;
import eu.dnetlib.uoamonitorservice.dao.*;
import eu.dnetlib.uoamonitorservice.entities.*;
import eu.dnetlib.uoamonitorservice.handlers.EntityNotFoundException;
import eu.dnetlib.uoaadmintoolslibrary.handlers.ForbiddenException;
import eu.dnetlib.uoamonitorservice.handlers.PathNotValidException;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

@RestController
@CrossOrigin(origins = "*")
public class SectionController {
    private final Logger log = Logger.getLogger(this.getClass());

    @Autowired
    private RolesUtils rolesUtils;

    @Autowired
    private StakeholderDAO stakeholderDAO;

    @Autowired
    private TopicDAO topicDAO;

    @Autowired
    private CategoryDAO categoryDAO;

    @Autowired
    private SubCategoryDAO subCategoryDAO;

    @Autowired
    private SectionDAO sectionDAO;

    @Autowired
    private IndicatorDAO indicatorDAO;

    @Autowired
    private IndicatorController indicatorController;

    public Section<Indicator> buildSection(Section<Indicator> sectionFull) {
        Section<String> section = new Section<>(sectionFull);

        List<String> indicators = new ArrayList<>();
        List<Indicator> indicatorsFull = new ArrayList<>();
        for(Indicator chart : sectionFull.getIndicators()) {
            Indicator chartSaved = indicatorDAO.save(chart);
            chart.setId(chartSaved.getId());
            indicatorsFull.add(chart);
            indicators.add(chartSaved.getId());
        }
        sectionFull.setIndicators(indicatorsFull);
        section.setIndicators(indicators);

        Date date = new Date();
        section.setCreationDate(date);
        section.setUpdateDate(date);

        sectionFull.setCreationDate(date);
        sectionFull.setUpdateDate(date);

        sectionDAO.save(section);

        sectionFull.setId(section.getId());
        return sectionFull;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/save/{index}", method = RequestMethod.POST)
    public Section saveSection(@PathVariable("stakeholderId") String stakeholderId,
                               @PathVariable("topicId") String topicId,
                               @PathVariable("categoryId") String categoryId,
                               @PathVariable("subcategoryId") String subcategoryId,
                               @PathVariable("index") String index,
                               @RequestBody Section<Indicator> sectionFull) {
        log.debug("save section");
        log.debug("Name: "+sectionFull.getTitle() + " - Id: "+sectionFull.getId() + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);

        SubCategory<String> subCategory = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId);

        Section<String> section = new Section<>(sectionFull);

        Date date = new Date();
        section.setUpdateDate(date);
        sectionFull.setUpdateDate(date);

        List<String> indicators = new ArrayList<>();

        Section<String> oldSection = null;
        if(sectionFull.getId() != null) {
            oldSection = sectionDAO.findById(sectionFull.getId());
            if(oldSection == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("save section: Section with id: " + sectionFull.getId() + " not found");
            }

            for(String indicatorId : oldSection.getIndicators()) {
                Indicator indicator = indicatorDAO.findById(indicatorId);
                if (indicator == null) {
                    // EXCEPTION - Indicator not found
                    throw new EntityNotFoundException("Save section: Indicator with id: "+indicatorId+" not found (indicator exists in section: "+section.getId()+")");
                }
                indicators.add(indicator.getId());
            }
        } else { // section does not exist in DB
            section.setCreationDate(date);
            sectionFull.setCreationDate(date);

            for(Indicator indicator : sectionFull.getIndicators()) {
                indicators.add(indicator.getId());
            }
        }

        String sectionId = sectionFull.getId();

        section.setIndicators(indicators);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
        // this section belongs in default profile and it is new or it is updated
        if(stakeholder.getDefaultId() == null) {
            if(sectionId == null) {
                sectionDAO.save(section);
                onSaveDefaultSection(section, topicId, categoryId, subcategoryId, stakeholder);
            }
            else {
                onUpdateDefaultSection(section, stakeholder, oldSection);
                sectionDAO.save(section);
            }
        } else {
            sectionDAO.save(section);
        }

        List<String> sections = null;
        if(sectionFull.getType().equals("chart")) {
            sections = subCategory.getCharts();
        } else if(sectionFull.getType().equals("number")) {
            sections = subCategory.getNumbers();
        }

        int existing_index = sections.indexOf(section.getId());
        if (existing_index == -1) {
            if(Integer.parseInt(index) != -1) {
                sections.add(Integer.parseInt(index), section.getId());
            } else {
                sections.add(section.getId());
            }
            subCategoryDAO.save(subCategory);
            log.debug("Section saved!");

            sectionFull.setId(section.getId());
        }

        return sectionFull;
    }

    public void onSaveDefaultSection(Section section,
                                     String defaultTopicId, String defaultCategoryId,
                                     String defaultSubcategoryId, Stakeholder defaultStakeholder) {
        log.debug("On save default section");


        // new section in default profile - add it on profiles of the same type
        List<SubCategory> subCategories = subCategoryDAO.findByDefaultId(defaultSubcategoryId);

        for (SubCategory subCategory : subCategories) {
            Category parentCategory = categoryDAO.findBySubCategoriesContaining(subCategory.getId());
            Topic parentTopic = topicDAO.findByCategoriesContaining(parentCategory.getId());
            Stakeholder parentStakeholder = stakeholderDAO.findByTopicsContaining(parentTopic.getId());

            Section sectionNew = new Section();
            sectionNew.copyFromDefault(section);

            sectionNew.setStakeholderAlias(parentStakeholder.getAlias());

            sectionDAO.save(sectionNew);

            List<String> sections = null;
            if (section.getType().equals("chart")) {
                sections = subCategory.getCharts();
            } else if (section.getType().equals("number")) {
                sections = subCategory.getNumbers();
            }
            sections.add(sectionNew.getId());

            subCategoryDAO.save(subCategory);
        }
    }

    public void onUpdateDefaultSection(Section section, Stakeholder stakeholder, Section oldSection) {
        log.debug("On update default section");

        // section already exists - check if changed and update all sections based on it

        boolean changed = false;
        List<Section> sections = sectionDAO.findByDefaultId(section.getId());

        for(Section sectionBasedOnDefault : sections) {
            if(section.getTitle() != null && !section.getTitle().equals(sectionBasedOnDefault.getTitle())
                    && (oldSection.getTitle() == null || oldSection.getTitle().equals(sectionBasedOnDefault.getTitle()))) {

                sectionBasedOnDefault.setTitle(section.getTitle());
                changed = true;
            }

            if(!changed) {
//                break;
                continue;
            }

//            sectionBasedOnDefault.setTitle(section.getTitle());
            sectionBasedOnDefault.setUpdateDate(section.getUpdateDate());
            sectionDAO.save(sectionBasedOnDefault);
        }
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/delete", method = RequestMethod.DELETE)
    public boolean deleteSection(@PathVariable("stakeholderId") String stakeholderId,
                                 @PathVariable("topicId") String topicId,
                                 @PathVariable("categoryId") String categoryId,
                                 @PathVariable("subcategoryId") String subcategoryId,
                                 @PathVariable("sectionId") String sectionId,
                                 @RequestParam(required = false) String children) {
        log.debug("delete section");
        log.debug("Id: "+sectionId + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);

        Section section = sectionDAO.findById(sectionId);
        if(section != null) {
            SubCategory<String> subCategory = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId);

            Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
            List<String> roles = rolesUtils.getRoles();
            if(section.getDefaultId() != null && !rolesUtils.hasCreateAndDeleteAuthority(roles, stakeholder.getType())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Delete section: You are not authorized to delete a default Section in stakeholder with id: "+stakeholderId);
            }

            String type = "";
            List<String> sections = null;
            if (section.getType().equals("chart")) {
                sections = subCategory.getCharts();
                type = "chart";
            } else if (section.getType().equals("number")) {
                sections = subCategory.getNumbers();
                type = "number";
            }

            int index = sections.indexOf(sectionId);
            if (index != -1) {
                // this section belongs in default profile
                if(subCategory.getDefaultId() == null && children != null) {
                    onDeleteDefaultSection(sectionId, subcategoryId, children, type);
                }

                indicatorController.deleteTree(section);

                sections.remove(index);
                subCategoryDAO.save(subCategory);

                sectionDAO.delete(sectionId);
                log.debug("Section deleted!");
            } else {
                // EXCEPTION - Section not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subcategory.getAlias();
                throw new PathNotValidException("Delete section: Section with id: "+sectionId+" not found in SubCategory: "+subcategoryId);
            }
        } else {
            // EXCEPTION - Section not found
            throw new EntityNotFoundException("Delete section: Section with id: "+sectionId+" not found");
        }
        return true;
    }

    public boolean onDeleteDefaultSection(String defaultSectionId, String defaultSubCategoryId, String children, String type) {
        if(children.equals("delete")) {
            List<SubCategory> subCategories = subCategoryDAO.findByDefaultId(defaultSubCategoryId);
            List<Section> sections = sectionDAO.findByDefaultId(defaultSectionId);

            for(SubCategory subCategory : subCategories) {
                Iterator<Section> sectionsIterator = sections.iterator();
                while(sectionsIterator.hasNext()) {
                    Section section = sectionsIterator.next();

                    String sectionId = section.getId();
                    List<String> subCategorySections = null;
                    if(type.equals("chart")) {
                        subCategorySections = subCategory.getCharts();
                    } else if(type.equals("number")) {
                        subCategorySections = subCategory.getNumbers();
                    }
                    if(subCategorySections != null && subCategorySections.contains(sectionId)) {
                        sectionsIterator.remove();

                        subCategorySections.remove(sectionId);
                        subCategoryDAO.save(subCategory);

                        indicatorController.deleteTree(section);

                        sectionDAO.delete(sectionId);
                        log.debug("Section with id: "+sectionId+" deleted!");

                        break;
                    }
                }
            }
        } else if(children.equals("disconnect")) {
            List<Section> sections = sectionDAO.findByDefaultId(defaultSectionId);
            for(Section section : sections) {
                indicatorController.disConnectTree(section);

                section.setDefaultId(null);
                sectionDAO.save(section);

                log.debug("DefaultId for Section with id: "+section.getId()+" empty!");
            }
        }
        return true;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{type}/reorder", method = RequestMethod.POST)
    public List<Section> reorderSections(@PathVariable("stakeholderId") String stakeholderId,
                                         @PathVariable("topicId") String topicId,
                                         @PathVariable("categoryId") String categoryId,
                                         @PathVariable("subcategoryId") String subcategoryId,
                                         @PathVariable("type") String type,
                                         @RequestBody List<String> sections) {
        log.debug("reorder sections of type: "+type);
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);

        SubCategory<String> subCategory = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId);

        if (type.equals("chart")) {
            List<String> oldSections = subCategory.getCharts();
            for (String sectionId : oldSections) {
                if (!sections.contains(sectionId)) {
                    sections.add(sectionId);
                }
            }
            subCategory.setCharts(sections);
        } else if (type.equals("number")) {
            List<String> oldSections = subCategory.getNumbers();
            for (String sectionId : oldSections) {
                if (!sections.contains(sectionId)) {
                    sections.add(sectionId);
                }
            }
            subCategory.setNumbers(sections);
        }

        List<Section> sectionsFull = new ArrayList<>();
        for(String sectionId : sections) {
            Section section = sectionDAO.findById(sectionId);
            if(section == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("Reorder sections: Section with id: " + sectionId + " not found");
            }
            sectionsFull.add(section);
        }

        subCategoryDAO.save(subCategory);
        log.debug("Sections reordered!");

        return sectionsFull;
    }

//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/toggle-status", method = RequestMethod.POST)
//    public Boolean toggleSectionStatus(@PathVariable("stakeholderId") String stakeholderId,
//                                       @PathVariable("topicId") String topicId,
//                                       @PathVariable("categoryId") String categoryId,
//                                       @PathVariable("subcategoryId") String subcategoryId,
//                                       @PathVariable("sectionId") String sectionId) {
//        log.debug("toggle section status (isActive)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId+ " - Section: "+sectionId);
//
//        Section section = sectionDAO.findById(sectionId);
//        if (section == null) {
//            // EXCEPTION - Section not found
//            throw new EntityNotFoundException("Toggle section status: Section with id: "+sectionId+" not found");
//        }
//        section.setIsActive(!section.getIsActive());
//
//        this.toggleSection(stakeholderId, topicId, categoryId, subcategoryId, section);
//
//        return section.getIsActive();
//    }

//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/{sectionId}/toggle-access", method = RequestMethod.POST)
//    public Boolean toggleSectionAccess(@PathVariable("stakeholderId") String stakeholderId,
//                                         @PathVariable("topicId") String topicId,
//                                         @PathVariable("categoryId") String categoryId,
//                                         @PathVariable("subcategoryId") String subcategoryId,
//                                         @PathVariable("sectionId") String sectionId) {
//        log.debug("toggle section access (isPublic)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);
//
//        Section section = sectionDAO.findById(sectionId);
//        if (section == null) {
//            // EXCEPTION - Section not found
//            throw new EntityNotFoundException("Toggle section access: Section with id: "+sectionId+" not found");
//        }
//        section.setIsPublic(!section.getIsPublic());
//
//        this.toggleSection(stakeholderId, topicId, categoryId, subcategoryId);
//
//        return section.getIsPublic();
//    }


    public void toggleSection(String stakeholderId, String topicId, String categoryId, String subcategoryId, Section section) {
        SubCategory<String> subCategory = checkForExceptions(stakeholderId, topicId, categoryId, subcategoryId);

        List<String> sections = null;
        if (section.getType().equals("chart")) {
            sections = subCategory.getCharts();
        } else if (section.getType().equals("number")) {
            sections = subCategory.getNumbers();
        }

        if(sections.contains(section.getId())) {
            sectionDAO.save(section);
            log.debug("Section toggled!");
        } else {
            // EXCEPTION - Section not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias(); -> SubCategory: subCategory.getAlias();
            throw new PathNotValidException("Toggle section: Section with id: "+section.getId()+" not found in SubCategory: "+subcategoryId);
        }

    }

    private SubCategory checkForExceptions(String stakeholderId, String topicId, String categoryId, String subcategoryId) {

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder == null) {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Save indicator: Stakeholder with id: " + stakeholderId + " not found");
        }

        List<String> roles = rolesUtils.getRoles();
        if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
            // EXCEPTION - Access denied
            throw new ForbiddenException("CheckForExceptions Section: You are not authorized to update stakeholder with id: "+stakeholderId);
        }

        Topic<String> topic = topicDAO.findById(topicId);
        if(topic == null) {
            // EXCEPTION - Topic not found
            throw new EntityNotFoundException("Save indicator: Topic with id: "+topicId+" not found");
        }

        if(!stakeholder.getTopics().contains(topicId)) {
            // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
            throw new PathNotValidException("Save indicator: Topic with id: " + topicId + " not found in Stakeholder: " + stakeholderId);
        }

        Category<String> category = categoryDAO.findById(categoryId);
        if(category == null) {
            // EXCEPTION - Category not found
            throw new EntityNotFoundException("Save indicator: Category with id: "+categoryId+" not found");
        }

        if(!topic.getCategories().contains(categoryId)) {
            // EXCEPTION - Category not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias();
            throw new PathNotValidException("Save indicator: Category with id: "+categoryId+" not found in Topic: "+topicId);
        }

        SubCategory<String> subcategory = subCategoryDAO.findById(subcategoryId);
        if(subcategory == null) {
            // EXCEPTION - SubCategory not found
            throw new EntityNotFoundException("Save indicator: SubCategory with id: "+subcategoryId+" not found");
        }

        if (!category.getSubCategories().contains(subcategoryId)) {
            // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
            throw new PathNotValidException("Save indicator: SubCategory with id: "+subcategoryId+" not found in Category: "+categoryId);
        }

        return  subcategory;
    }

    public void deleteTree(SubCategory subCategory) {
        List<String> sections = subCategory.getCharts();
        for(String sectionId : sections) {
            Section section = sectionDAO.findById(sectionId);
            if (section == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("Section delete tree: Chart Section with id: "+sectionId+" not found (section exists in subCategory: "+subCategory.getId()+")");
            }

            indicatorController.deleteTree(section);

            sectionDAO.delete(sectionId);
        }

        sections = subCategory.getNumbers();
        for(String sectionId : sections) {
            Section section = sectionDAO.findById(sectionId);
            if (section == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("Section delete tree: Number Section with id: "+sectionId+" not found (section exists in subCategory: "+subCategory.getId()+")");
            }

            indicatorController.deleteTree(section);

            sectionDAO.delete(sectionId);
        }
    }

    public void disConnectTree(SubCategory subCategory) {
        List<String> sections = subCategory.getCharts();
        for(String sectionId : sections) {
            Section section = sectionDAO.findById(sectionId);
            if (section == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("Section disconnect tree: Chart Section with id: "+sectionId+" not found (section exists in subCategory: "+subCategory.getId()+")");
            }

            indicatorController.disConnectTree(section);

            section.setDefaultId(null);
            sectionDAO.save(section);
        }

        sections = subCategory.getNumbers();
        for(String sectionId : sections) {
            Section section = sectionDAO.findById(sectionId);
            if (section == null) {
                // EXCEPTION - Section not found
                throw new EntityNotFoundException("Section disconnect tree: Number Section with id: "+sectionId+" not found (section exists in subCategory: "+subCategory.getId()+")");
            }

            indicatorController.disConnectTree(section);

            section.setDefaultId(null);
            sectionDAO.save(section);
        }
    }
}
