package eu.dnetlib.uoamonitorservice.controllers;

import eu.dnetlib.uoaadmintoolslibrary.handlers.utils.RolesUtils;
import eu.dnetlib.uoamonitorservice.dao.*;
import eu.dnetlib.uoamonitorservice.entities.*;
import eu.dnetlib.uoamonitorservice.handlers.EntityNotFoundException;
import eu.dnetlib.uoaadmintoolslibrary.handlers.ForbiddenException;
import eu.dnetlib.uoamonitorservice.handlers.PathNotValidException;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

@RestController
@CrossOrigin(origins = "*")
public class SubCategoryController {
    private final Logger log = Logger.getLogger(this.getClass());

    @Autowired
    private RolesUtils rolesUtils;

    @Autowired
    private StakeholderDAO stakeholderDAO;

    @Autowired
    private TopicDAO topicDAO;

    @Autowired
    private CategoryDAO categoryDAO;

    @Autowired
    private SubCategoryDAO subCategoryDAO;

    @Autowired
    private SectionDAO sectionDAO;

    @Autowired
    private IndicatorDAO indicatorDAO;

    @Autowired
    private SectionController sectionController;

    public SubCategory<Section<Indicator>> buildSubCategory(SubCategory<Section<Indicator>> subcategoryFull) {
        SubCategory<String> subCategory = new SubCategory<>(subcategoryFull);

        List<String> sectionCharts = new ArrayList<>();
        List<Section<Indicator>> sectionChartsFull = new ArrayList<>();

        for(Section section : subcategoryFull.getCharts()) {
            Section<Indicator> sectionFull = sectionController.buildSection(section);
            sectionChartsFull.add(sectionFull);
            sectionCharts.add(sectionFull.getId());
        }
        subcategoryFull.setCharts(sectionChartsFull);
        subCategory.setCharts(sectionCharts);

        List<String> sectionNumbers = new ArrayList<>();
        List<Section<Indicator>> sectionNumbersFull = new ArrayList<>();

        for(Section section : subcategoryFull.getNumbers()) {
            Section<Indicator> sectionFull = sectionController.buildSection(section);
            sectionNumbersFull.add(sectionFull);
            sectionNumbers.add(sectionFull.getId());
        }
        subcategoryFull.setNumbers(sectionNumbersFull);
        subCategory.setNumbers(sectionNumbers);

        Date date = new Date();
        subCategory.setCreationDate(date);
        subCategory.setUpdateDate(date);

        subcategoryFull.setCreationDate(date);
        subcategoryFull.setUpdateDate(date);


        subCategoryDAO.save(subCategory);

        subcategoryFull.setId(subCategory.getId());
        return subcategoryFull;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/save", method = RequestMethod.POST)
    public SubCategory<Section<Indicator>> saveSubCategory(@PathVariable("stakeholderId") String stakeholderId,
                                                           @PathVariable("topicId") String topicId,
                                                           @PathVariable("categoryId") String categoryId,
                                                           @RequestBody SubCategory<Section<Indicator>> subcategoryFull) {
        log.debug("save subcategory");
        log.debug("Alias: "+subcategoryFull.getAlias() + " - Id: "+subcategoryFull.getId() + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId);

        Category category = checkForExceptions(stakeholderId, topicId, categoryId);

        SubCategory<String> subCategory = new SubCategory<>(subcategoryFull);

        Date date = new Date();
        subCategory.setUpdateDate(date);
        subcategoryFull.setUpdateDate(date);

        List<String> chartSections = new ArrayList<>();
        List<String> numberSections = new ArrayList<>();

        SubCategory<String> oldSubcategory = null;
        if(subcategoryFull.getId() != null) {
            oldSubcategory = subCategoryDAO.findById(subcategoryFull.getId());
            if(oldSubcategory == null) {
                // EXCEPTION - SubCategory not found
                throw new EntityNotFoundException("save subcategory: SubCategory with id: " + subcategoryFull.getId() + " not found");
            }

            for(String chartSectionId : oldSubcategory.getCharts()) {
                Section section = sectionDAO.findById(chartSectionId);
                if (section == null) {
                    // EXCEPTION - Section not found
                    throw new EntityNotFoundException("Save subcategory: Chart section with id: "+chartSectionId+" not found (section exists in subcategory: "+subCategory.getId()+")");
                }
                chartSections.add(section.getId());
            }

            for(String numberSectionId : oldSubcategory.getNumbers()) {
                Section section = sectionDAO.findById(numberSectionId);
                if (section == null) {
                    // EXCEPTION - Section not found
                    throw new EntityNotFoundException("Save subcategory: Number section with id: "+numberSectionId+" not found (section exists in subcategory: "+subCategory.getId()+")");
                }
                numberSections.add(section.getId());
            }
        } else { // subcategory does not exist in DB
            subCategory.setCreationDate(date);
            subcategoryFull.setCreationDate(date);

            for(Section chartSection : subcategoryFull.getCharts()) {
                chartSections.add(chartSection.getId());
            }

            for(Section numberSection : subcategoryFull.getNumbers()) {
                numberSections.add(numberSection.getId());
            }
        }

//        List<String> charts = new ArrayList<>();
//        for(Indicator chart : subcategoryFull.getCharts()) {
//            charts.add(chart.getId());
//        }
//        subCategory.setCharts(charts);
//
//        List<String> numbers = new ArrayList<>();
//        for(Indicator numbr : subcategoryFull.getNumbers()) {
//            numbers.add(numbr.getId());
//        }
//        subCategory.setNumbers(numbers);


        subCategory.setCharts(chartSections);
        subCategory.setNumbers(numberSections);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
        if(stakeholder.getDefaultId() == null) {
            if(subcategoryFull.getId() == null) {
                subCategoryDAO.save(subCategory);
                onSaveDefaultSubCategory(subCategory, categoryId);
            } else {
                onUpdateDefaultSubCategory(subCategory, oldSubcategory);
                subCategoryDAO.save(subCategory);
            }
        } else {
            subCategoryDAO.save(subCategory);
        }

        List<String> subcategories = category.getSubCategories();
        int index = subcategories.indexOf(subCategory.getId());
        if(index == -1) {
            subcategories.add(subCategory.getId());
            categoryDAO.save(category);
            log.debug("Subcategory saved!");

            subcategoryFull.setId(subCategory.getId());
        }

        chartSections = null;
        numberSections = null;
        subCategory = null;

        return subcategoryFull;
    }

    public void onSaveDefaultSubCategory(SubCategory subCategory, String categoryId) {
        log.debug("On save default subCategory");

        List<Category> categories = categoryDAO.findByDefaultId(categoryId);
        for(Category category : categories) {
            SubCategory subCategoryNew = new SubCategory();
            subCategoryNew.copyFromDefault(subCategory);

            subCategoryDAO.save(subCategoryNew);

            List<String> subCategories = category.getSubCategories();
            subCategories.add(subCategoryNew.getId());

            categoryDAO.save(category);
        }
    }

    public void onUpdateDefaultSubCategory(SubCategory subCategory, SubCategory oldSubcategory) {
        log.debug("On update default subCategory");

        List<SubCategory> subCategories = subCategoryDAO.findByDefaultId(subCategory.getId());
        boolean changed = false;
        for(SubCategory subCategoryBasedOnDefault : subCategories) {
            if(subCategory.getName() != null && !subCategory.getName().equals(subCategoryBasedOnDefault.getName())
                    && (oldSubcategory.getName() == null || oldSubcategory.getName().equals(subCategoryBasedOnDefault.getName()))) {

                subCategoryBasedOnDefault.setName(subCategory.getName());
                subCategoryBasedOnDefault.setAlias(subCategory.getAlias());
                changed = true;
            }
            if(subCategory.getDescription() != null && !subCategory.getDescription().equals(subCategoryBasedOnDefault.getDescription())
                    && (oldSubcategory.getDescription() == null || oldSubcategory.getDescription().equals(subCategoryBasedOnDefault.getDescription()))) {

                subCategoryBasedOnDefault.setDescription(subCategory.getDescription());
                changed = true;
            }

            if(!changed) {
//                break;
                continue;
            }

//            subCategoryBasedOnDefault.setName(subCategory.getName());
//            subCategoryBasedOnDefault.setDescription(subCategory.getDescription());
            subCategoryBasedOnDefault.setUpdateDate(subCategory.getUpdateDate());
            subCategoryDAO.save(subCategoryBasedOnDefault);
        }
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/delete", method = RequestMethod.DELETE)
    public boolean deleteSubCategory(@PathVariable("stakeholderId") String stakeholderId,
                                     @PathVariable("topicId") String topicId,
                                     @PathVariable("categoryId") String categoryId,
                                     @PathVariable("subcategoryId") String subcategoryId,
                                     @RequestParam(required = false) String children) {
        log.debug("delete subcategory");
        log.debug("Id: "+subcategoryId + " - Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId);

        Category category = checkForExceptions(stakeholderId, topicId, categoryId);

        SubCategory<String> subcategory = subCategoryDAO.findById(subcategoryId);
        if(subcategory != null) {

            Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);
            List<String> roles = rolesUtils.getRoles();
            if(subcategory.getDefaultId() != null && !rolesUtils.hasCreateAndDeleteAuthority(roles, stakeholder.getType())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Delete subcategory: You are not authorized to delete a default SubCategory in stakeholder with id: "+stakeholderId);
            }

            List<String> subcategories = category.getSubCategories();
            int index = subcategories.indexOf(subcategoryId);
            if(index != -1) {
                // this subCategory belongs in default profile
                if(category.getDefaultId() == null && children != null) {
                    onDeleteDefaultSubCategory(subcategoryId, categoryId, children);
                }

//                for(String chartSectionId : subcategory.getCharts()) {
//                    Section<String> chartSection = sectionDAO.findById(chartSectionId);
//                    if (chartSection == null) {
//                        // EXCEPTION - Section not found
//                        throw new EntityNotFoundException("Delete SubCategory: Section with id: "+chartSectionId+" not found (section exists in subcategory: "+subcategoryId+")");
//                    }
//
//                    for (String chartId : chartSection.getIndicators()) {
//                        indicatorDAO.delete(chartId);
//                    }
//                    subcategory.setCharts(null);
//                    sectionDAO.delete(chartSectionId);
//                }
//
//                for(String numberSectionId : subcategory.getNumbers()) {
//                    Section<String> numberSection = sectionDAO.findById(numberSectionId);
//                    if (numberSection == null) {
//                        // EXCEPTION - Section not found
//                        throw new EntityNotFoundException("Delete SubCategory: Section with id: "+numberSectionId+" not found (section exists in subcategory: "+subcategoryId+")");
//                    }
//
//                    for (String numberId : numberSection.getIndicators()) {
//                        indicatorDAO.delete(numberId);
//                    }
//                    subcategory.setNumbers(null);
//                    sectionDAO.delete(numberSectionId);
//                }

                sectionController.deleteTree(subcategory);

                subcategory.setCharts(null);
                subcategory.setNumbers(null);

                subcategories.remove(index);
                categoryDAO.save(category);

                subCategoryDAO.delete(subcategoryId);
                log.debug("Subcategory deleted!");
            } else {
                // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
                throw new PathNotValidException("Delete subcategory: Subcategory with id: "+subcategoryId+" not found in Category: "+categoryId);
            }

        } else {
            // EXCEPTION - SubCategory not found
            throw new EntityNotFoundException("Delete subcategory: SubCategory with id: "+subcategoryId+" not found");
        }
        return true;
    }

    public boolean onDeleteDefaultSubCategory(String defaultSubCategoryId, String defaultCategoryId, String children) {
        if(children.equals("delete")) {
            List<Category> categories = categoryDAO.findByDefaultId(defaultCategoryId);
            List<SubCategory> subCategories = subCategoryDAO.findByDefaultId(defaultSubCategoryId);

            for(Category category : categories) {
                Iterator<SubCategory> subCategoriesIterator = subCategories.iterator();
                while(subCategoriesIterator.hasNext()) {
                    SubCategory subCategory = subCategoriesIterator.next();

                    String subCategoryId = subCategory.getId();

                    if(category.getSubCategories() != null && category.getSubCategories().contains(subCategoryId)) {
                        subCategoriesIterator.remove();

                        category.getSubCategories().remove(subCategoryId);
                        categoryDAO.save(category);

                        sectionController.deleteTree(subCategory);

                        subCategoryDAO.delete(subCategoryId);
                        log.debug("SubCategory with id: "+subCategoryId+" deleted!");

                        break;
                    }
                }
            }
        } else if(children.equals("disconnect")) {
            List<SubCategory> subCategories = subCategoryDAO.findByDefaultId(defaultSubCategoryId);
            for(SubCategory subCategory : subCategories) {
                sectionController.disConnectTree(subCategory);

                subCategory.setDefaultId(null);
                subCategoryDAO.save(subCategory);

                log.debug("DefaultId for SubCategory with id: "+subCategory.getId()+" empty!");
            }
        }
        return true;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/reorder", method = RequestMethod.POST)
    public List<SubCategory> reorderSubCategories(@PathVariable("stakeholderId") String stakeholderId,
                                            @PathVariable("topicId") String topicId,
                                            @PathVariable("categoryId") String categoryId,
                                            @RequestBody List<String> subCategories) {
        log.debug("reorder subCategories");
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId);

        Category<String> category = checkForExceptions(stakeholderId, topicId, categoryId);

        List<String> oldSubcategories = category.getSubCategories();
        for (String subcategoryId : oldSubcategories) {
            if (!subCategories.contains(subcategoryId)) {
                subCategories.add(subcategoryId);
            }
        }
        category.setSubCategories(subCategories);

        List<SubCategory> subCategoriesFull = new ArrayList<>();
        for(String subCategoryId : subCategories) {
            SubCategory subCategory = subCategoryDAO.findById(subCategoryId);
            if(subCategory == null) {
                // EXCEPTION - SubCategory not found
                throw new EntityNotFoundException("Reorder subCategories: subCategory with id: " + subCategoryId + " not found");
            }
            subCategoriesFull.add(subCategory);
        }

        categoryDAO.save(category);
        log.debug("SubCategories reordered!");

        return subCategoriesFull;
    }

//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/toggle-status", method = RequestMethod.POST)
//    public Boolean toggleSubCategoryStatus(@PathVariable("stakeholderId") String stakeholderId,
//                                           @PathVariable("topicId") String topicId,
//                                           @PathVariable("categoryId") String categoryId,
//                                           @PathVariable("subcategoryId") String subcategoryId) {
//        log.debug("toggle subCategory status (isActive)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);
//
//        SubCategory subCategory = subCategoryDAO.findById(subcategoryId);
//        if (subCategory == null) {
//            // EXCEPTION - SubCategory not found
//            throw new EntityNotFoundException("Toggle subCategory status: SubCategory with id: "+subcategoryId+" not found");
//        }
//        subCategory.setIsActive(!subCategory.getIsActive());
//
//        this.toggleSubCategory(stakeholderId, topicId, categoryId, subCategory);
//
//        return subCategory.getIsActive();
//    }
//
//    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/toggle-access", method = RequestMethod.POST)
//    public Boolean toggleSubCategoryAccess(@PathVariable("stakeholderId") String stakeholderId,
//                                           @PathVariable("topicId") String topicId,
//                                           @PathVariable("categoryId") String categoryId,
//                                           @PathVariable("subcategoryId") String subcategoryId) {
//        log.debug("toggle subCategory access (isPublic)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);
//
//        SubCategory subCategory = subCategoryDAO.findById(subcategoryId);
//        if (subCategory == null) {
//            // EXCEPTION - SubCategory not found
//            throw new EntityNotFoundException("Toggle subCategory access: SubCategory with id: "+subcategoryId+" not found");
//        }
//        subCategory.setIsPublic(!subCategory.getIsPublic());
//
//        this.toggleSubCategory(stakeholderId, topicId, categoryId, subCategory);
//
//        return subCategory.getIsPublic();
//    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/{categoryId}/{subcategoryId}/change-visibility", method = RequestMethod.POST)
    public Visibility changeTopicVisibility(@PathVariable("stakeholderId") String stakeholderId,
                                            @PathVariable("topicId") String topicId,
                                            @PathVariable("categoryId") String categoryId,
                                            @PathVariable("subcategoryId") String subcategoryId,
                                            @RequestParam("visibility") Visibility visibility) {
        log.debug("change subCategory visibility: "+visibility);
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId + " - Category: "+categoryId+ " - SubCategory: "+subcategoryId);

        SubCategory subCategory = subCategoryDAO.findById(subcategoryId);
        if (subCategory == null) {
            // EXCEPTION - SubCategory not found
            throw new EntityNotFoundException("Change subCategory visibility: SubCategory with id: "+subcategoryId+" not found");
        }
        subCategory.setVisibility(visibility);

        this.toggleSubCategory(stakeholderId, topicId, categoryId, subCategory);

        return subCategory.getVisibility();
    }

    public void toggleSubCategory(String stakeholderId, String topicId, String categoryId, SubCategory subcategory) {
        Category category = checkForExceptions(stakeholderId, topicId, categoryId);

        if (category.getSubCategories().contains(subcategory.getId())) {
            subCategoryDAO.save(subcategory);
            log.debug("SubCategory toggled!");
        } else {
            // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
            throw new PathNotValidException("Toggle subCategory: SubCategory with id: "+subcategory.getId()+" not found in Category: "+categoryId);
        }
    }


    private Category checkForExceptions(String stakeholderId, String topicId, String categoryId) {

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder == null) {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Save indicator: Stakeholder with id: " + stakeholderId + " not found");
        }

        List<String> roles = rolesUtils.getRoles();
        if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
            // EXCEPTION - Access denied
            throw new ForbiddenException("CheckForExceptions SubCategory: You are not authorized to update stakeholder with id: "+stakeholderId);
        }

        Topic<String> topic = topicDAO.findById(topicId);
        if(topic == null) {
            // EXCEPTION - Topic not found
            throw new EntityNotFoundException("Save indicator: Topic with id: "+topicId+" not found");
        }

        if(!stakeholder.getTopics().contains(topicId)) {
            // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
            throw new PathNotValidException("Save indicator: Topic with id: " + topicId + " not found in Stakeholder: " + stakeholderId);
        }

        Category<String> category = categoryDAO.findById(categoryId);
        if(category == null) {
            // EXCEPTION - Category not found
            throw new EntityNotFoundException("Save indicator: Category with id: "+categoryId+" not found");
        }

        if(!topic.getCategories().contains(categoryId)) {
            // EXCEPTION - Category not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias();
            throw new PathNotValidException("Save indicator: Category with id: "+categoryId+" not found in Topic: "+topicId);
        }

//        SubCategory<String> subcategory = subCategoryDAO.findById(subcategoryId);
//        if(subcategory == null) {
//            // EXCEPTION - SubCategory not found
//            throw new EntityNotFoundException("Save indicator: SubCategory with id: "+subcategoryId+" not found");
//        }
//
//        if (!category.getSubCategories().contains(subcategoryId)) {
//            // EXCEPTION - SubCategory not found in Stakeholder: stakeholder.getAlias(); -> Topic: topic.getAlias(); -> Category: category.getAlias();
//            throw new PathNotValidException("Save indicator: SubCategory with id: "+subcategoryId+" not found in Category: "+categoryId);
//        }

        return  category;
    }

    public void deleteTree(Category category) {
        List<String> subCategories = category.getSubCategories();
        for(String subCategoryId : subCategories) {
            SubCategory subCategory = subCategoryDAO.findById(subCategoryId);
            if (subCategory == null) {
                // EXCEPTION - SubCategory not found
                throw new EntityNotFoundException("SubCategory delete tree: SubCategory with id: "+subCategoryId+" not found (subCategory exists in category: "+category.getId()+")");
            }

            sectionController.deleteTree(subCategory);

            subCategoryDAO.delete(subCategoryId);
        }
    }

    public void disConnectTree(Category category) {
        List<String> subCategories = category.getSubCategories();
        for(String subCategoryId : subCategories) {
            SubCategory subCategory = subCategoryDAO.findById(subCategoryId);
            if (subCategory == null) {
                // EXCEPTION - SubCategory not found
                throw new EntityNotFoundException("SubCategory disconnect tree: SubCategory with id: "+subCategoryId+" not found (subCategory exists in category: "+category.getId()+")");
            }

            sectionController.disConnectTree(subCategory);

            subCategory.setDefaultId(null);
            subCategoryDAO.save(subCategory);
        }
    }
}
