package eu.dnetlib.uoamonitorservice.controllers;

import eu.dnetlib.uoaadmintoolslibrary.handlers.utils.RolesUtils;
import eu.dnetlib.uoamonitorservice.dao.*;
import eu.dnetlib.uoamonitorservice.entities.*;
import eu.dnetlib.uoamonitorservice.handlers.EntityNotFoundException;
import eu.dnetlib.uoaadmintoolslibrary.handlers.ForbiddenException;
import eu.dnetlib.uoamonitorservice.handlers.PathNotValidException;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

@RestController
@CrossOrigin(origins = "*")
public class TopicController {
    private final Logger log = Logger.getLogger(this.getClass());

    @Autowired
    private RolesUtils rolesUtils;

    @Autowired
    private StakeholderDAO stakeholderDAO;

    @Autowired
    private TopicDAO topicDAO;

    @Autowired
    private CategoryController categoryController;

    @Autowired
    private CategoryDAO categoryDAO;

    public Topic<Category> buildTopic(Topic<Category> topicFull) {
        Topic<String> topic = new Topic<>(topicFull);

        List<String> categories = new ArrayList<>();
        List<Category> categoriesFull = new ArrayList<>();
        for(Category<SubCategory> category : topicFull.getCategories()) {
            Category<SubCategory> categoryFull = categoryController.buildCategory(category);
            categoriesFull.add(categoryFull);
            categories.add(categoryFull.getId());
        }
        topicFull.setCategories(categoriesFull);
        topic.setCategories(categories);

        Date date = new Date();
        topic.setCreationDate(date);
        topic.setUpdateDate(date);

        topicFull.setCreationDate(date);
        topicFull.setUpdateDate(date);

        topicDAO.save(topic);

        topicFull.setId(topic.getId());
        return topicFull;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/save", method = RequestMethod.POST)
    public Topic<Category> saveTopic(@PathVariable("stakeholderId") String stakeholderId,
                                     @RequestBody Topic<Category> topicFull) {
        log.debug("save topic");
        log.debug("Alias: "+topicFull.getAlias() + " - Id: "+topicFull.getId()+ " - Stakeholder: "+stakeholderId);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder != null) {
            List<String> roles = rolesUtils.getRoles();
            if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Save Topic: You are not authorized to update stakeholder with id: "+stakeholderId);
            }

            Topic<String> topic = new Topic<>(topicFull);
            Date date = new Date();
            topic.setUpdateDate(date);
            topicFull.setUpdateDate(date);

            List<String> categories = new ArrayList<>();

            Topic<String> oldTopic = null;
            if(topicFull.getId() != null) {
                oldTopic = topicDAO.findById(topicFull.getId());
                if(oldTopic == null) {
                    // EXCEPTION - Topic not found
                    throw new EntityNotFoundException("save topic: Topic with id: "+topicFull.getId()+" not found");
                }
                for(String categoryId : oldTopic.getCategories()) {
                    Category category = categoryDAO.findById(categoryId);
                    if (category == null) {
                        // EXCEPTION - Category not found
                        throw new EntityNotFoundException("Save topic: Category with id: "+categoryId+" not found (category exists in topic: "+topic.getId()+")");
                    }
                    categories.add(category.getId());
                }
            } else { // topic does not exist in DB
                topic.setCreationDate(date);
                topicFull.setCreationDate(date);

                for(Category category : topicFull.getCategories()) {
                    categories.add(category.getId());
                }
            }

            topic.setCategories(categories);

            if(stakeholder.getDefaultId() == null) {
                if(topicFull.getId() == null) {
                    topicDAO.save(topic);
                    onSaveDefaultTopic(topic, stakeholderId);
                } else {
                    onUpdateDefaultTopic(topic, oldTopic);
                    topicDAO.save(topic);
                }
            } else {
                topicDAO.save(topic);
            }

            List<String> topics = stakeholder.getTopics();
            int index = topics.indexOf(topic.getId());
            if(index == -1) {
                topics.add(topic.getId());
                stakeholderDAO.save(stakeholder);
                log.debug("Topic saved!");

                topicFull.setId(topic.getId());
            }

            categories = null;
            topic = null;
        } else {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Save topic: Stakeholder with id: "+stakeholderId+" not found");
        }
        return topicFull;
    }

    public void onSaveDefaultTopic(Topic topic, String stakeholderId) {
        log.debug("On save default topic");

        List<Stakeholder> stakeholders = stakeholderDAO.findByDefaultId(stakeholderId);
        for(Stakeholder _stakeholder : stakeholders) {
            Topic topicNew = new Topic();
            topicNew.copyFromDefault(topic);

            topicDAO.save(topicNew);

            List<String> topics = _stakeholder.getTopics();
            topics.add(topicNew.getId());

            stakeholderDAO.save(_stakeholder);
        }
    }

    public void onUpdateDefaultTopic(Topic topic, Topic oldTopic) {
        log.debug("On update default topic");

        List<Topic> topics = topicDAO.findByDefaultId(topic.getId());
        boolean changed = false;
        for(Topic topicBasedOnDefault : topics) {
            if(topic.getName() != null && !topic.getName().equals(topicBasedOnDefault.getName())
                    && (oldTopic.getName() == null || oldTopic.getName().equals(topicBasedOnDefault.getName()))) {

                topicBasedOnDefault.setName(topic.getName());
                topicBasedOnDefault.setAlias(topic.getAlias());
                changed = true;
            }
            if(topic.getDescription() != null && !topic.getDescription().equals(topicBasedOnDefault.getDescription())
                    && (oldTopic.getDescription() == null || oldTopic.getDescription().equals(topicBasedOnDefault.getDescription()))) {

                topicBasedOnDefault.setDescription(topic.getDescription());
                changed = true;
            }
            if(topic.getIcon() != null && !topic.getIcon().equals(topicBasedOnDefault.getIcon())
                    && (oldTopic.getIcon() == null || oldTopic.getIcon().equals(topicBasedOnDefault.getIcon()))) {

                topicBasedOnDefault.setIcon(topic.getIcon());
                changed = true;
            }

            if(!changed) {
//                break;
                continue;
            }

//            topicBasedOnDefault.setName(topic.getName());
//            topicBasedOnDefault.setDescription(topic.getDescription());
            topicBasedOnDefault.setUpdateDate(topic.getUpdateDate());
            topicDAO.save(topicBasedOnDefault);
        }
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/delete", method = RequestMethod.DELETE)
    public boolean deleteTopic(@PathVariable("stakeholderId") String stakeholderId,
                               @PathVariable("topicId") String topicId,
                               @RequestParam(required = false) String children) {
        log.debug("delete topic");
        log.debug("Id: "+topicId + " - Stakeholder: "+stakeholderId);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder != null) {

            List<String> roles = rolesUtils.getRoles();
            if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Delete topic: You are not authorized to update stakeholder with id: "+stakeholderId);
            }

            Topic<String> topic = topicDAO.findById(topicId);
            if(topic != null) {

                if(topic.getDefaultId() != null && !rolesUtils.hasCreateAndDeleteAuthority(roles, stakeholder.getType())) {
                    // EXCEPTION - Access denied
                    throw new ForbiddenException("Delete topic: You are not authorized to delete a default Topic in stakeholder with id: "+stakeholderId);
                }

                List<String> topics = stakeholder.getTopics();
                int index = topics.indexOf(topicId);
                if(index != -1) {
                    // this topic belongs in default profile
                    if(stakeholder.getDefaultId() == null && children != null) {
                        onDeleteDefaultTopic(topicId, stakeholderId, children);
                    }

//                    for(String categoryId : topic.getCategories()) {
//                        Category<String> category = categoryDAO.findById(categoryId);
//                        if(category == null) {
//                            // EXCEPTION - Category not found
//                            throw new EntityNotFoundException("Delete topic: Category with id: "+categoryId+" not found (category exists in topic: "+topicId+")");
//                        }
//
//                        for(String subCategoryId : category.getSubCategories()) {
//                            SubCategory<String> subcategory = subCategoryDAO.findById(subCategoryId);
//                            if (subcategory == null) {
//                                // EXCEPTION - SubCategory not found
//                                throw new EntityNotFoundException("Delete topic: SubCategory with id: "+subCategoryId+" not found (subcategory exists in category: "+categoryId+")");
//                            }
//
//                            for(String chartSectionId : subcategory.getCharts()) {
//                                Section<String> chartSection = sectionDAO.findById(chartSectionId);
//                                if (chartSection == null) {
//                                    // EXCEPTION - Section not found
//                                    throw new EntityNotFoundException("Delete topic: Section with id: "+chartSectionId+" not found (section exists in subcategory: "+subCategoryId+")");
//                                }
//
//                                for (String chartId : chartSection.getIndicators()) {
//                                    indicatorDAO.delete(chartId);
//                                }
//                                subcategory.setCharts(null);
//                                sectionDAO.delete(chartSectionId);
//                            }
//
//                            for(String numberSectionId : subcategory.getNumbers()) {
//                                Section<String> numberSection = sectionDAO.findById(numberSectionId);
//                                if (numberSection == null) {
//                                    // EXCEPTION - Section not found
//                                    throw new EntityNotFoundException("Delete topic: Section with id: "+numberSectionId+" not found (section exists in subcategory: "+subCategoryId+")");
//                                }
//
//                                for (String numberId : numberSection.getIndicators()) {
//                                    indicatorDAO.delete(numberId);
//                                }
//                                subcategory.setNumbers(null);
//                                sectionDAO.delete(numberSectionId);
//                            }
//
//                            subCategoryDAO.delete(subCategoryId);
//                        }
//                        category.setSubCategories(null);
//                        categoryDAO.delete(categoryId);
//                    }
                    categoryController.deleteTree(topic);

                    topic.setCategories(null);

                    topics.remove(index);
                    stakeholderDAO.save(stakeholder);

                    topicDAO.delete(topicId);
                    log.debug("Topic deleted!");
                } else {
                    // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
                    throw new PathNotValidException("Delete topic: Topic with id: "+topicId+" not found in Stakeholder: "+stakeholderId);
                }

            } else {
                // EXCEPTION - Topic not found
                throw new EntityNotFoundException("Delete topic: Topic with id: "+topicId+" not found");
            }
        } else {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Delete topic: Stakeholder with id: "+stakeholderId+" not found");
        }
        return true;
    }


    public boolean onDeleteDefaultTopic(String defaultTopicId, String defaultStakeholderId, String children) {
        if(children.equals("delete")) {
            List<Stakeholder> stakeholders = stakeholderDAO.findByDefaultId(defaultStakeholderId);
            List<Topic> topics = topicDAO.findByDefaultId(defaultTopicId);

            for(Stakeholder stakeholder : stakeholders) {
                Iterator<Topic> topicsIterator = topics.iterator();
                while(topicsIterator.hasNext()) {
                    Topic topic = topicsIterator.next();

                    String topicId = topic.getId();

                    if(stakeholder.getTopics() != null && stakeholder.getTopics().contains(topicId)) {
                        topicsIterator.remove();

                        stakeholder.getTopics().remove(topicId);
                        stakeholderDAO.save(stakeholder);

                        categoryController.deleteTree(topic);

                        topicDAO.delete(topicId);
                        log.debug("Topic with id: "+topicId+" deleted!");

                        break;
                    }
                }
            }
        } else if(children.equals("disconnect")) {
            List<Topic> topics = topicDAO.findByDefaultId(defaultTopicId);
            for(Topic topic : topics) {
                categoryController.disConnectTree(topic);

                topic.setDefaultId(null);
                topicDAO.save(topic);

                log.debug("DefaultId for Topic with id: "+topic.getId()+" cleared!");
            }
        }
        return true;
    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/reorder", method = RequestMethod.POST)
    public List<Topic> reorderTopics(@PathVariable("stakeholderId") String stakeholderId,
                                     @RequestBody List<String> topics) {
        log.debug("reorder topics");
        log.debug("Stakeholder: "+stakeholderId);

        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if(stakeholder != null) {

            List<String> roles = rolesUtils.getRoles();
            if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Reorder topics: You are not authorized to update stakeholder with id: "+stakeholderId);
            }

            List<String> oldTopics = stakeholder.getTopics();
            for (String topicId : oldTopics) {
                if (!topics.contains(topicId)) {
                    topics.add(topicId);
                }
            }
            stakeholder.setTopics(topics);

            List<Topic> topicsFull = new ArrayList<>();
            for (String topicId : topics) {
                Topic topic = topicDAO.findById(topicId);
                if(topic == null) {
                    // EXCEPTION - Topic not found
                    throw new EntityNotFoundException("Reorder Topics: Topic with id: " + topicId + " not found");
                }
                topicsFull.add(topic);
            }

            stakeholderDAO.save(stakeholder);
            log.debug("Topics reordered!");

            return topicsFull;
        } else {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Reorder topics: Stakeholder with id: "+stakeholderId+" not found");
        }
    }

//    @RequestMapping(value = "/{stakeholderId}/{topicId}/toggle-status", method = RequestMethod.POST)
//    public Boolean toggleTopicStatus(@PathVariable("stakeholderId") String stakeholderId,
//                                     @PathVariable("topicId") String topicId) {
//        log.debug("toggle topic status (isActive)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId);
//
//        Topic topic = topicDAO.findById(topicId);
//        if (topic == null) {
//            // EXCEPTION - Topic not found
//            throw new EntityNotFoundException("Toggle topic status: Topic with id: "+topicId+" not found");
//        }
//        topic.setIsActive(!topic.getIsActive());
//
//        this.toggleTopic(stakeholderId, topic);
//
//        return topic.getIsActive();
//    }
//
//    @RequestMapping(value = "/{stakeholderId}/{topicId}/toggle-access", method = RequestMethod.POST)
//    public Boolean toggleTopicAccess(@PathVariable("stakeholderId") String stakeholderId,
//                                         @PathVariable("topicId") String topicId) {
//        log.debug("toggle topic access (isPublic)");
//        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId);
//
//        Topic topic = topicDAO.findById(topicId);
//        if (topic == null) {
//            // EXCEPTION - Topic not found
//            throw new EntityNotFoundException("Toggle topic access: Topic with id: "+topicId+" not found");
//        }
//        topic.setIsPublic(!topic.getIsPublic());
//
//        this.toggleTopic(stakeholderId, topic);
//
//        return topic.getIsPublic();
//    }

    @PreAuthorize("isAuthenticated()")
    @RequestMapping(value = "/{stakeholderId}/{topicId}/change-visibility", method = RequestMethod.POST)
    public Visibility changeTopicVisibility(@PathVariable("stakeholderId") String stakeholderId,
                                            @PathVariable("topicId") String topicId,
                                            @RequestParam("visibility") Visibility visibility) {
        log.debug("change topic visibility: "+visibility);
        log.debug("Stakeholder: "+stakeholderId + " - Topic: "+topicId);

        Topic topic = topicDAO.findById(topicId);
        if (topic == null) {
            // EXCEPTION - Topic not found
            throw new EntityNotFoundException("Change topic visibility: Topic with id: "+topicId+" not found");
        }
        topic.setVisibility(visibility);

        this.toggleTopic(stakeholderId, topic);

        return topic.getVisibility();
    }

    public void toggleTopic(String stakeholderId, Topic topic) {
        Stakeholder<String> stakeholder = stakeholderDAO.findById(stakeholderId);

        if (stakeholder != null) {

            List<String> roles = rolesUtils.getRoles();
            if(!rolesUtils.hasUpdateAuthority(roles, stakeholder.getType(), stakeholder.getAlias())) {
                // EXCEPTION - Access denied
                throw new ForbiddenException("Toggle topic: You are not authorized to update stakeholder with id: "+stakeholderId);
            }

            if (stakeholder.getTopics().contains(topic.getId())) {
                topicDAO.save(topic);
                log.debug("Topic toggled!");
            } else {
                // EXCEPTION - Topic not found in Stakeholder: stakeholder.getAlias();
                throw new PathNotValidException("Toggle topic: Topic with id: "+topic.getId()+" not found in Stakeholder: "+stakeholderId);
            }
        } else {
            // EXCEPTION - Stakeholder not found
            throw new EntityNotFoundException("Toggle topic: Stakeholder with id: "+stakeholderId+" not found");
        }
    }

    public void deleteTree(Stakeholder stakeholder) {
        List<String> topics = stakeholder.getTopics();
        for(String topicId : topics) {
            Topic topic = topicDAO.findById(topicId);
            if (topic == null) {
                // EXCEPTION - Topic not found
                throw new EntityNotFoundException("Topic delete tree: Topic with id: "+topicId+" not found (topic exists in stakeholder: "+stakeholder.getId()+")");
            }

            categoryController.deleteTree(topic);

            topicDAO.delete(topicId);
        }
    }
}
