package eu.dnetlib.repo.manager.client.datasources.utils;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.regexp.shared.RegExp;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.Widget;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.repo.manager.client.services.RepositoryService;
import eu.dnetlib.repo.manager.client.services.RepositoryServiceAsync;
import eu.dnetlib.gwt.client.MyFormGroup;
import eu.dnetlib.repo.manager.client.widgets.TextBox;
import eu.dnetlib.repo.manager.client.widgets.ValueChangeEvent;
import eu.dnetlib.repo.manager.client.widgets.ValueChangeHandler;
import eu.dnetlib.repo.manager.shared.Constants;
import org.gwtbootstrap3.client.ui.Alert;
import org.gwtbootstrap3.client.ui.Form;
import org.gwtbootstrap3.client.ui.ListBox;
import org.gwtbootstrap3.client.ui.RadioButton;
import org.gwtbootstrap3.client.ui.constants.AlertType;
import org.gwtbootstrap3.client.ui.constants.ButtonType;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Created by stefania on 12/17/15.
 */
public class SelectRepositoryWidget implements IsWidget {

    private FlowPanel selectRepositoryPanel = new FlowPanel();

    private Alert errorAlert = new Alert();

    private HTML infoLabel = new HTML();

    private Form form = new Form();
    private ListBox countriesListBox = new ListBox();
    private TextBox autocomplete = new TextBox();

    private FlowPanel repositoriesPanel = new FlowPanel();

    private List<Repository> repositoriesForCountryList = new ArrayList<Repository>();
    private List<Repository> matchingRepositoriesForCountryList = new ArrayList<Repository>();

    private Repository selectedRepository;

    private RepositoryServiceAsync repositoryService = GWT.create(RepositoryService.class);

    private String mode;

    public SelectRepositoryWidget(Map<String, String> countries, final String mode) {

        this.mode = mode;

        selectRepositoryPanel.addStyleName("animated fadeInRight");
        selectRepositoryPanel.addStyleName("stepContent");

        errorAlert.setType(AlertType.DANGER);
        errorAlert.setDismissable(false);
        errorAlert.setVisible(false);
        selectRepositoryPanel.add(errorAlert);

        infoLabel.addStyleName("alert alert-info");
        if(mode.equals(Constants.REPOSITORY_MODE_OPENDOAR)) {

            repositoryService.getLatestUpdateDateOfList(mode, new AsyncCallback<String>() {

                @Override
                public void onFailure(Throwable caught) {
                    infoLabel.setHTML("Please make sure your repository is registered in <a href=\"http://www.opendoar.org\" " +
                            "target=\"_blank\"><i class=\"fa fa-external-link\" style=\"margin-right:3px\"></i>OpenDOAR</a>.");
                }

                @Override
                public void onSuccess(String result) {
                    infoLabel.setHTML("Please make sure your repository is registered in <a href=\"http://www.opendoar.org\" " +
                            "target=\"_blank\"><i class=\"fa fa-external-link\" style=\"margin-right:3px\"></i>OpenDOAR</a>.<br>" +
                            "<b>Last Updated: </b>" + result);
                }
            });

        } else if(mode.equals(Constants.REPOSITORY_MODE_RE3DATA)) {

            repositoryService.getLatestUpdateDateOfList(mode, new AsyncCallback<String>() {

                @Override
                public void onFailure(Throwable caught) {
                    infoLabel.setHTML("Please make sure your repository is registered in <a href=\"http://www.re3data.org\" " +
                            "target=\"_blank\"><i class=\"fa fa-external-link\" style=\"margin-right:3px\"></i>Re3data</a>.");
                }

                @Override
                public void onSuccess(String result) {
                    infoLabel.setHTML("Please make sure your repository is registered in <a href=\"http://www.re3data.org\" " +
                            "target=\"_blank\"><i class=\"fa fa-external-link\" style=\"margin-right:3px\"></i>Re3data</a>.<br>" +
                            "<b>Last Updated: </b>" + result);
                }
            });
        }

        selectRepositoryPanel.add(infoLabel);

        countriesListBox.addItem("-- none selected --", "noneSelected");
        for(String country : countries.keySet()) {
            countriesListBox.addItem(countries.get(country), country);
        }
        countriesListBox.addChangeHandler(new ChangeHandler() {
            @Override
            public void onChange(ChangeEvent event) {

                repositoriesPanel.clear();

                repositoriesForCountryList.clear();
                matchingRepositoriesForCountryList.clear();

                if(!countriesListBox.getSelectedValue().equals("noneSelected")) {

                    final HTML loadingWheel = new HTML("<div class=\"loader-big\" style=\"text-align: center; padding-top: 20px; " +
                            "color: rgb(47, 64, 80); font-weight: bold;\">Retrieving repositories for selected country</div>" +
                            "<div class=\"whiteFilm\"></div>");
                    repositoriesPanel.addStyleName("loading");
                    repositoriesPanel.add(loadingWheel);

                    repositoryService.getRepositoriesByCountry(countriesListBox.getSelectedValue(), mode,
                            new AsyncCallback<List<Repository>>() {

                                @Override
                                public void onFailure(Throwable caught) {

                                    Alert errorLabel = new Alert();
                                    errorLabel.setText("System error retrieving repositories for this country");
                                    errorLabel.setType(AlertType.DANGER);
                                    errorLabel.setDismissable(false);

                                    repositoriesPanel.removeStyleName("loading");
                                    repositoriesPanel.remove(loadingWheel);
                                    repositoriesPanel.add(errorLabel);
                                }

                                @Override
                                public void onSuccess(List<Repository> repositories) {

                                    repositoriesPanel.removeStyleName("loading");
                                    repositoriesPanel.remove(loadingWheel);

                                    repositoriesForCountryList.addAll(repositories);

                                    if(autocomplete.getValue()!=null && !autocomplete.getValue().trim().equals("")) {

                                        for (Repository repository : repositoriesForCountryList) {
                                            if (repository.getOfficialName().toLowerCase().contains(autocomplete.getValue().toLowerCase())) {
                                                matchingRepositoriesForCountryList.add(repository);
                                            }
                                        }

                                    } else {

                                        matchingRepositoriesForCountryList.addAll(repositoriesForCountryList);
                                    }

                                    repositoriesPanel.clear();
                                    drawRepositoriesPanel(matchingRepositoriesForCountryList);

                                    autocomplete.setEnabled(true);
                                }
                            });
                } else {
                    autocomplete.setEnabled(false);
                }
            }
        });

        form.add(new MyFormGroup(false, "Country", countriesListBox));

        autocomplete.setValueChangeHandler(new ValueChangeHandler() {
            @Override
            public void handle(ValueChangeEvent valueChangeEvent) {

                matchingRepositoriesForCountryList.clear();

                if(valueChangeEvent.getNewValue()!=null && !valueChangeEvent.getNewValue().trim().equals("")) {

                    for (Repository repository : repositoriesForCountryList) {
                        if (repository.getOfficialName().toLowerCase().contains(valueChangeEvent.getNewValue().toLowerCase())) {
                            matchingRepositoriesForCountryList.add(repository);
                        }
                    }

                    repositoriesPanel.clear();
                    drawRepositoriesPanel(matchingRepositoriesForCountryList);

                } else {

                    matchingRepositoriesForCountryList.addAll(repositoriesForCountryList);

                    repositoriesPanel.clear();
                    drawRepositoriesPanel(matchingRepositoriesForCountryList);
                }
            }
        });
        autocomplete.setEnabled(false);
        form.add(new MyFormGroup(false, "Search by repository name", autocomplete));

        selectRepositoryPanel.add(form);

        selectRepositoryPanel.add(repositoriesPanel);

    }

    @Override
    public Widget asWidget() {
        return selectRepositoryPanel;
    }

    public Repository getSelectedRepository() {

        errorAlert.setVisible(false);

        if(selectedRepository==null) {
            errorAlert.setVisible(true);
            errorAlert.setText("You need to select a repository first.");
        }

        return selectedRepository;
    }

    private void drawRepositoriesPanel(List<Repository> matchingRepositoriesForCountryList) {

        String modeName = "";
        if(mode.equals(Constants.REPOSITORY_MODE_OPENDOAR))
            modeName = "OpenDOAR";
        else
            modeName = "Re3data";

        HTML labelForCountry = new HTML("<h2>"  + modeName + " Repositories in " + countriesListBox.getSelectedValue() + "</h2>");
        repositoriesPanel.add(labelForCountry);

        addRepositoriesRadioListToPanel(matchingRepositoriesForCountryList, repositoriesPanel);
    }

    private void addRepositoriesRadioListToPanel(List<Repository> repositories, FlowPanel panel) {

        if(repositories.isEmpty()) {

            Alert warningAlert = new Alert();
            warningAlert.setType(AlertType.WARNING);
            warningAlert.setDismissable(false);
            warningAlert.setText("No repositories found");

            panel.add(warningAlert);

        } else {

            Form repositoriesRadioButtonsForm = new Form();
            repositoriesRadioButtonsForm.addStyleName("repositoriesRadioButtonForm");

            for(final Repository repository : repositories) {

                String repositoryName = "";
                if(autocomplete.getValue()!=null && !autocomplete.getValue().trim().equals("")) {

                    //TODO make it not lose case sensitivity
                    repositoryName = RegExp.compile(autocomplete.getValue(), "gi")
                            .replace(repository.getOfficialName(), "<mark>" + autocomplete.getValue() + "</mark>");
                } else {
                    repositoryName = repository.getOfficialName();
                }

                final RadioButton radioButton;
                if(repository.isRegistered()) {
                    radioButton = new RadioButton("repository", repositoryName +
                            "<span class=\"label label-warning registeredLabel\">Registered</span>" +
                            "<a target=\"_blank\" href=\"" + repository.getWebsiteUrl() + "\"><i class=\"fa " +
                            "fa-external-link externalLink\"></i></a>", true);
                    radioButton.setEnabled(false);
                } else {
                    radioButton = new RadioButton("repository", repositoryName +
                            "<a target=\"_blank\" href=\"" + repository.getWebsiteUrl() + "\"><i class=\"fa " +
                            "fa-external-link externalLink\"></i></a>", true);
                }
                radioButton.setType(ButtonType.LINK);
                radioButton.addChangeHandler(new ChangeHandler() {
                    @Override
                    public void onChange(ChangeEvent event) {
                        if(radioButton.getValue()) {
                            selectedRepository = repository;
                        }
                    }
                });
                repositoriesRadioButtonsForm.add(new MyFormGroup(false, null, radioButton));
            }

            panel.add(repositoriesRadioButtonsForm);
        }
    }
}
