package eu.dnetlib.repo.manager.server.services;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.gwt.server.service.SpringGwtRemoteServiceServlet;
import eu.dnetlib.repo.manager.client.services.BrokerService;
import eu.dnetlib.repo.manager.service.controllers.BrokerApi;
import eu.dnetlib.repo.manager.service.controllers.RepositoryApi;
import eu.dnetlib.repo.manager.shared.BrokerException;
import eu.dnetlib.repo.manager.shared.RepositoryServiceException;
import eu.dnetlib.repo.manager.shared.Term;
import eu.dnetlib.repo.manager.shared.Tuple;
import eu.dnetlib.repo.manager.shared.broker.*;
import eu.dnetlib.repos.RepoApi;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import java.io.IOException;
import java.util.*;

/**
 * Created by stefanos on 10/26/16.
 */
@SuppressWarnings("serial")
@Service("brokerService")
public class BrokerServiceImpl extends SpringGwtRemoteServiceServlet implements BrokerService {

    private static final Logger LOGGER = Logger
            .getLogger(BrokerServiceImpl.class);

    @Autowired
    private RepoApi repoAPI;


    @Autowired
    private RepositoryApi repositoryApi;
    @Autowired
    private BrokerApi brokerApi;


    @Override
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        LOGGER.info("broker service init");
    }

    @Value("${services.broker.url}:${services.broker.port}/${services.broker.api}${services.broker.openaire}")
    private String openairePath;
    @Value("${services.broker.url}:${services.broker.port}/${services.broker.api}")
    private String apiPath;

    /**
     * @param datasourceName the name of the data source
     * @return a list of BrowseEvent entries for that datasource
     * @throws BrokerException containing the error code from the server
     * @author stefanos
     */
    @Override
    public List<BrowseEntry> getTopicsForDatasource(String datasourceName) throws BrokerException {
        return brokerApi.getTopicsForDatasource(datasourceName);
    }

    /**
     * @param datasourceName the name of the datasource
     * @param topic          the name of the topic to filter
     * @param page           the page number
     * @return an Events page with a constant 50 max number of entires
     * @throws BrokerException containing the error code from the server
     */
    @Override
    public EventsPage showEvents(String datasourceName, String topic, long page) throws BrokerException{
        JSONObject params = new JSONObject();
        try {
            params.put("datasourceName",datasourceName);
            params.put("topic",topic);
            params.put("page",String.valueOf(page));
            return brokerApi.showEvents(params.toString());
        } catch (JSONException e) {
            LOGGER.debug("Error on show events",e);
        }
        return null;
    }

    /**
     * @param advQueryObject a pojo class containing the filter parameters
     * @param page           the number of the page
     * @param pageSize       the page size
     * @return
     * @throws BrokerException
     */
    @Override
    public EventsPage advancedShowEvents(AdvQueryObject advQueryObject, long page, long pageSize) throws BrokerException {

        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("page",String.valueOf(page));
            jsonObject.put("pagesize",String.valueOf(pageSize));
            ObjectMapper mapper = new ObjectMapper();
            String json_advQueryObject = mapper.writeValueAsString(advQueryObject);
            jsonObject.put("advQueryObject",json_advQueryObject);
            return brokerApi.advancedShowEvents(jsonObject.toString());
        } catch (Exception e) {
            LOGGER.debug("Error on advanced show events",e);
        }
        return null;
    }

    @Override
    public DatasourcesBroker getDatasourcesOfUser(String userEmail, boolean includeShared, boolean includeByOthers)
            throws BrokerException {

        JSONObject params = new JSONObject();
        try {
            params.put("userEmail",userEmail);
            params.put("includeShared",includeShared);
            params.put("includeByOthers",includeByOthers);
            return brokerApi.getDatasourcesOfUser(params.toString());
        } catch (JSONException e) {
            LOGGER.debug("Error on get datasources of user",e);
        }
        return null;
      /*  DatasourcesBroker ret = new DatasourcesBroker();
        try {
            LOGGER.debug("In getDatasourcesOfUser");

            //set for user
            ret.setDatasourcesOfUser(getDatasourcesOfUserType(this.repoAPI.getRepositoriesOfUser(userEmail, false)));

            //set for shared
            if (includeShared) {
                //TODO whatever nikonas was saying
                List<String> sharedDatasourceIds = new ArrayList<String>();
                ret.setSharedDatasources(getDatasourcesOfUserType(this.repoAPI.getReposByIds(sharedDatasourceIds)));
            }

            //set others
            if (includeByOthers) {
                ret.setDatasourcesOfOthers(getDatasourcesOfUserType(this.repoAPI.getRepositoriesOfUser(userEmail, true)));
            }

        } catch (RepositoryServiceException e) {
            e.printStackTrace();
            throw new BrokerException(e);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ret;*/
    }

    @Override
    public Map<String, List<SimpleSubscriptionDesc>> getSimpleSubscriptionsOfUser(String userEmail) throws BrokerException {
        return brokerApi.getSimpleSubscriptionsOfUser(userEmail);
    }

    @Override
    public Subscription subscribe(OpenaireSubscription obj) throws BrokerException {
        return brokerApi.subscribe(obj);
    }

    @Override
    public void unsubscribe(String subscriptionId) throws BrokerException {
        brokerApi.unsubscribe(subscriptionId);
    }

    @Override
    public Subscription getSubscription(String subscriptionId) throws BrokerException {
        return brokerApi.getSubscription(subscriptionId);
    }

    @Override
    public void unsubscribe(List<String> subscriptionIds) throws BrokerException {
        for (String subscriptionId : subscriptionIds) {
            unsubscribe(subscriptionId);
        }
    }

    public Map<String,Term> getDnetTopics() throws BrokerException {
        return brokerApi.getDnetTopics();
    }

    /**
     * Helper class to aggregate the datasources topic sizes by datasource name.
     *
     * @param repositories to be aggregated
     * @return a List of BrowseEntry with the official name and the number of topics for each repo and logo url
     * @throws BrokerException
     */
    private List<Tuple<BrowseEntry, String>> getDatasourcesOfUserType(List<Repository> repositories) throws BrokerException {

        //get user entries
        LOGGER.debug("getDatasourcesOfUserType : " + repositories.size());
        List<Tuple<BrowseEntry, String>> entries = new ArrayList<>();
        for (Repository repo : repositories) {
            BrowseEntry temp = new BrowseEntry();
            temp.setValue(repo.getOfficialName());
            temp.setSize(new Long(0));
            for (BrowseEntry e : this.getTopicsForDatasource(repo.getOfficialName())) {
                temp.setSize(temp.getSize() + e.getSize());
            }
            Tuple<BrowseEntry, String> tup = new Tuple<>(temp, repo.getLogoUrl());
            entries.add(tup);
        }

        // sort the collection by the second field of the tuple which is size
        Collections.sort(entries, new Comparator<Tuple<BrowseEntry, String>>() {
            @Override
            public int compare(Tuple<BrowseEntry, String> e1, Tuple<BrowseEntry, String> e2) {
                return (int) (e2.getFirst().getSize().longValue() - e1.getFirst().getSize().longValue());
            }
        });

        return entries;
    }

    @Override
    public EventsPage getNotificationsBySubscriptionId(String subscriptionId,String page,String size) throws BrokerException {
        return brokerApi.getNotificationsBySubscriptionId(subscriptionId,page,size);
    }

    @Override
    public Map<String, List<Subscription>> getSubscriptionsOfUser(String userEmail) throws BrokerException {
        return brokerApi.getSubscriptionsOfUser(userEmail);
    }

}
