package eu.dnetlib.repo.manager.server.services;

import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.domain.data.PiwikInfo;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.domain.data.RepositoryInterface;
import eu.dnetlib.domain.enabling.Vocabulary;
import eu.dnetlib.domain.functionality.UserProfile;
import eu.dnetlib.gwt.server.service.SpringGwtRemoteServiceServlet;
import eu.dnetlib.repo.manager.client.services.RepositoryService;
import eu.dnetlib.repo.manager.server.utils.EmailUtils;
import eu.dnetlib.repo.manager.server.utils.LocalVocabularies;
import eu.dnetlib.repo.manager.service.controllers.RepositoryApi;
import eu.dnetlib.repo.manager.shared.*;
import eu.dnetlib.repos.RepoApi;
import gr.uoa.di.driver.enabling.vocabulary.VocabularyLoader;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.log4j.Logger;
import org.json.JSONException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.net.URL;

/**
 * Created by nikonas on 12/8/15.
 */
@SuppressWarnings("serial")
@Service("repositoryService")
public class RepositoryServiceImpl extends SpringGwtRemoteServiceServlet implements RepositoryService {

    private static final Logger LOGGER = Logger
            .getLogger(RepositoryServiceImpl.class);

    @Autowired
    private RepoApi repoAPI;

    @Autowired
    private EmailUtils emailUtils;

    private final String[] vocabularyNames = {"dnet:countries", "dnet:datasource_typologies", "dnet:compatibilityLevel"};

    @Autowired
    private VocabularyLoader vocabularyLoader;

    private Map<String, Vocabulary> vocabularyMap = new ConcurrentHashMap<String, Vocabulary>();

    @Value("${services.repo-manager.repository.testing.mode}")
    private boolean testingMode;

    @Autowired
    private PiwikDAO piwikDAO;

    @Value("${services.repomanager.analyticsURL}")
    private String analyticsURL;

    @Value("${services.repomanager.usageStatisticsDiagramsBaseURL}")
    private String usageStatisticsDiagramsBaseURL;

    @Value("${services.repomanager.usageStatisticsNumbersBaseURL}")
    private String usageStatisticsNumbersBaseURL;

    private static final String PIWIK_SCRIPT = StringEscapeUtils.escapeHtml("<!-- Piwik -->\n" +
            "<script type=\"text/javascript\">\n" +
            "\tvar _paq = _paq || [];\n" +
            "\t_paq.push(['enableLinkTracking']);\n" +
            "\t(function() {\n" +
            "\t\tvar u=\"//analytics.openaire.eu/\";\n" +
            "\t\t_paq.push(['setTrackerUrl', u+'piwik.php']);\n" +
            "\t\t_paq.push(['setSiteId', $$$]);\n" +
            "\t\t<% if(handle != null){%>\n" +
            "\t\t\t_paq.push(['setCustomVariable', 1, 'oaipmhID',\"oai:<%= baseUrl %>:<%=handle %>\", 'page']);\n" +
            "\t\t\t_paq.push(['trackPageView']);\n" +
            "\t\t<}>\n" +
            "\t\tvar d=document, g=d.createElement('script'), s=d.getElementsByTagName('script')[0];\n" +
            "\t\tg.type='text/javascript'; g.async=true; g.defer=true; g.src=u+'piwik.js'; s.parentNode.insertBefore(g,s);\n" +
            "\t})();\n" +
            "</script>\n" +
            "<noscript>\n" +
            "\t<p>\n" +
            "\t\t<img src=\"//analytics.openaire.eu/piwik.php?idsite=47\" style=\"border:0;\" alt=\"\" />\n" +
            "\t</p>\n" +
            "</noscript>\n" +
            "<!— End Piwik Code —>");


    @Autowired
    private RepositoryApi repositoryApi;



    @PostConstruct
    public void init() {
        this.loadVocabularies();
    }

    @Override
    public Tuple<List<Repository>, List<Repository>> getRepositoriesByCountry(String country, String mode, boolean includeUnknownCountries) throws RepositoryServiceException {
        try {
            if (testingMode)
                return this.getRepositoriesByCountryTesting(country, mode, includeUnknownCountries);
            LOGGER.debug("Getting repositories of country: " + country + " with type: " + mode + " and includeUnknownCountries: " + includeUnknownCountries);

            Tuple<List<Repository>, List<Repository>> retTuple = new Tuple<List<Repository>, List<Repository>>();
//            List<Repository> reposList = this.repoAPI.getRepositoriesPerCountry(mode).get(country);

            Boolean managed = null;
            List<Repository> reposList = repositoryApi.getRepositoriesByCountry(country,mode,managed);
            if (reposList == null) {
                retTuple.setFirst(new ArrayList<Repository>());
//                if (!includeUnknownCountries) {
//                    throw new RepositoryServiceException("registration.noReposForThisCountry", RepositoryServiceException.ErrorCode.NO_REPOS_FOR_THIS_COUNTRY);
//                }
            } else {
                retTuple.setFirst(reposList);
            }

            if (includeUnknownCountries) {
                List<Repository> withoutCountryList = this.repoAPI.getRepositoriesPerCountry(mode).get("Without Country");
                List<Repository> unknownCountryList = this.repoAPI.getRepositoriesPerCountry(mode).get("UNKNOWN");
                List<Repository> totalList = new ArrayList<Repository>();
                if (withoutCountryList != null)
                    totalList.addAll(withoutCountryList);
                if (unknownCountryList != null)
                    totalList.addAll(unknownCountryList);
                retTuple.setSecond(totalList);
            }

            return retTuple;

        } catch (Exception e) {
            LOGGER.error("Error while getting repositories of country: " + country + " with type: " + mode + " and includeUnknownCountries: " + includeUnknownCountries, e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public List<Repository> getRepositoriesByCountry(String country, String mode) throws RepositoryServiceException {
        return this.getRepositoriesByCountry(country, mode, false).getFirst();
    }

    @Override
    public DatasourcesCollection getRepositoriesOfUser(String userEmail, boolean includeShared, boolean includeByOthers) throws RepositoryServiceException {

        DatasourcesCollection retDatasources = new DatasourcesCollection();
        try {
            LOGGER.debug("Getting repositories of user: " + userEmail + " . IncludeShared: "
                    + includeShared + " . IncludeByOthers: " + includeByOthers);
            int page = 0;
            String size = "50";
            List<Repository> resultSet = repositoryApi.getRepositoriesOfUser(userEmail,String.valueOf(page),size);
            while(resultSet.size() > 0 ){
                retDatasources.getDatasourcesOfUser().addAll(resultSet);
                page++;
                resultSet = repositoryApi.getRepositoriesOfUser(userEmail,String.valueOf(page),size);
            }

/*
	if (includeShared) {
                //TODO create dao to save-get shared datasourcesIDs
                List<String> sharedDatasourceIds = new ArrayList<String>();
                retDatasources.setSharedDatasources(this.repoAPI.getReposByIds(sharedDatasourceIds));
                //geting Piwik Info
                for(Repository repository: retDatasources.getSharedDatasources())
                    repository.setPiwikInfo(this.getPiwikSiteForRepository(repository.getId()));
            }

            if (includeByOthers) {
                retDatasources.setDatasourcesOfOthers(this.repoAPI.getRepositoriesOfUser(userEmail, true));
                //geting Piwik Info
                for(Repository repository: retDatasources.getDatasourcesOfOthers())
                    repository.setPiwikInfo(this.getPiwikSiteForRepository(repository.getId()));
            }
*/
        } catch (JSONException e) {
            LOGGER.error("Error while getting repositories of user: " + userEmail + " . IncludeShared: " + includeShared + " . IncludeByOthers: " + includeByOthers, e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
        return retDatasources;
    }

    @Override
    public List<String> getRepositoryUrlsOfUser(String userEmail, boolean includeShared, boolean includeByOthers) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting repositories(urls) of user: " + userEmail + " . IncludeShared: " + includeShared + " . IncludeByOthers: " + includeByOthers);
            List<String> retRepos = new ArrayList<String>();

            int page = 0;
            String size = "50";
            List<String> resultSet = repositoryApi.getUrlsOfUserRepos(userEmail,String.valueOf(page),size);
            while(resultSet.size() > 0 ){
                retRepos.addAll(resultSet);
                page++;
                resultSet = repositoryApi.getUrlsOfUserRepos(userEmail,String.valueOf(page),size);
            }
            return retRepos;

        } catch (Exception e) {
            LOGGER.error("Error while getting repositories(urls) of user: " + userEmail + " . IncludeShared: " + includeShared + " . IncludeByOthers: " + includeByOthers, e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public Repository getRepository(String repoId) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting repository with id: " + repoId);
            Repository repo = repositoryApi.getRepositoryById(repoId);

            if (repo != null) {
                for (RepositoryInterface iFace : repo.getInterfaces()) {
                    if (!iFace.getContentDescription().equals("file::hybrid") && iFace.getAccessProtocol().equalsIgnoreCase("oai")) {
                        iFace.setComplianceName(getComplianceName(iFace.getCompliance()));
                        if (iFace.getCompliance().equals("notCompatible"))
                            iFace.setComplianceName("not compatible");
                    }
                }

                //geting Piwik Info
                repo.setPiwikInfo(this.getPiwikSiteForRepository(repoId));

            } else
                throw new RepositoryServiceException("registration.repositoryNotExists", RepositoryServiceException.ErrorCode.REPOSITORY_NOT_EXISTS);
            return repo;

        } catch (Exception e) {
            LOGGER.error("Error while getting repository with id: " + repoId, e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public Map<String, String> getCountries(Boolean existingOnly, String mode) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting countries");
            List<String> countries = new ArrayList<String>();

            Map<String, String> countriesMap = new TreeMap<String, String>();

            if (existingOnly) {
                LOGGER.debug("using the repositories map");
                countries.addAll(this.repoAPI.getRepositoriesByCountry(mode).keySet());
            } else {
                LOGGER.debug("using \"dnet:countries\" vocabulary");
                countries.addAll(this.getVocabulary("dnet:countries").getEnglishNames());
            }
//            countries.addAll(repositoryApi.getDnetCountries());
            for (String country : countries) {
                countriesMap.put(country, WordUtils.capitalizeFully(country));
            }

            return countriesMap;

        } catch (Exception e) {
            LOGGER.error("Error while getting getting countries", e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public Map<String, String> getCountries() throws RepositoryServiceException {
        return this.getCountries(false, null);
    }

    @Override
    public List<Timezone> getTimezones() throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting timezones from file");
//            return repositoryApi.getTimezones();
            return LocalVocabularies.timezones;
        } catch (Exception e) {
            LOGGER.error("Error while getting timezones from file", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }

    }

    @Override
    public List<String> getTypologies() throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting typologies from file");
           // return repositoryApi.getTypologies();
            return LocalVocabularies.typologies;
        } catch (Exception e) {
            LOGGER.error("Error while getting typologies from file", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public Map<String, String> getDatasourceClasses(String mode) throws RepositoryServiceException {
        return repositoryApi.getDatasourceClasses(mode);
    }

    @Override
    public Map<String, String> getCompatibilityClasses(String mode) throws RepositoryServiceException {
        return repositoryApi.getCompatibilityClasses(mode);
    }

    @Override
    public void storeRepository(Repository repo, String mode) throws RepositoryServiceException {

        try {
           /* LOGGER.debug("Storing repository with name: " + repo.getOfficialName());
            //List<RepositoryInterface> interfacesToRegister = new ArrayList<RepositoryInterface>();
            JSONObject params = new JSONObject();
            params.put("datatype", mode);
            ObjectMapper mapper = new ObjectMapper();
            String json_repo = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(repo);
            params.put("repository", json_repo);*/


            //repositoryApi.addRepository(params.toString());
            repositoryApi.addRepository(mode,repo);
            LOGGER.debug("Repository with name: " + repo.getOfficialName() + " stored!");

        }catch (Exception e) {
            emailUtils.reportException(e);
            LOGGER.error("Error while storing repository with name: " + repo.getOfficialName(), e);
            throw new RepositoryServiceException("Error while storing repository", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void updateRepositoryInformation(Repository repo) throws RepositoryServiceException {
        try {
            LOGGER.debug("Updating information of repo: " + repo.getOfficialName());

	    //TODO SOS, check old API

            //this.repoAPI.updateRepositoryInformation(repo);
            repositoryApi.updateEnglishName(repo.getId(),repo.getEnglishName());
            repositoryApi.updateLatitude(repo.getId(), String.valueOf(repo.getLatitude()));
            repositoryApi.updateLongitude(repo.getId(), String.valueOf(repo.getLongitude()));
            repositoryApi.updateOfficialName(repo.getId(),repo.getOfficialName());
            repositoryApi.updateLogoUrl(repo.getId(),repo.getLogoUrl());
            repositoryApi.updateTimezone(repo.getId(),String.valueOf(repo.getTimezone()));
            repositoryApi.updatePlatform(repo.getId(),repo.getTypology());

        } catch (Exception e) {
            LOGGER.error("Error while updating information of repo: " + repo.getOfficialName(), e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public RepositoryInterface updateInterface(RepositoryInterface iFace, String repoId, String datatype) throws RepositoryServiceException {
        try {
            LOGGER.debug("updating interface with id: " + iFace.getId());
            RepositoryInterface retIface = null;
            retIface = this.repoAPI.updateRepositoryInterfaceWithoutChecks(repoId, iFace, datatype);
            retIface.setComplianceName(this.getComplianceName(retIface.getCompliance()));

            return retIface;
        } catch (Exception e) {
            LOGGER.error("error updating interface with id: " + iFace.getId(), e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public RepositoryInterface insertInterface(RepositoryInterface iFace, String repoId, String datatype) throws RepositoryServiceException {
        try {
            LOGGER.debug("inserting interface with id: " + iFace.getId());
            RepositoryInterface retIface = null;
            //retIface = this.repoAPI.insertRepositoryInterfaceWithoutChecks(repoId, iFace, datatype);

          /*  JSONObject params = new JSONObject();
            params.put("datatype",datatype);
            params.put("repoId",repoId);
            params.put("iFace",iFace);*/
            //retIface = repositoryApi.addRepositoryInterface(params.toString());

            retIface = repositoryApi.addRepositoryInterface(datatype,repoId,iFace);
            retIface.setComplianceName(this.getComplianceName(retIface.getCompliance()));
            return retIface;

        } catch (Exception e) {
            LOGGER.error("error updating interface with id: " + iFace.getId(), e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public void deleteInterface(String repoId, RepositoryInterface iFace, String datatype) throws RepositoryServiceException {
        List<RepositoryInterface> iFaces = new ArrayList<RepositoryInterface>();
        iFaces.add(iFace);
        this.deleteInterfaces(repoId, iFaces, datatype);
    }

    @Override
    public void deleteInterfaces(String repoId, List<RepositoryInterface> iFaces, String datatype) throws RepositoryServiceException {
        try {
            LOGGER.debug("deleting interfaces of repo: " + repoId);
            //this.repoAPI.deleteRepositoryInterfacesWithoutChecks(repoId, iFaces, datatype);

            for(RepositoryInterface iFace : iFaces) {
                LOGGER.info("deleting repository interface with url/set/id: " + iFace.getBaseUrl() + "/"
                        + iFace.getAccessSet() + "/" + iFace.getId());
                repositoryApi.deleteRepositoryInterface(iFace.getId());
            }

        } catch (Exception e) {
            LOGGER.error("deleting interfaces of repo: " + repoId, e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }

    }

    @Override
    public DatasourceVocabularies getDatasourceVocabularies(String mode) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting vocabularies for datasource with type: " + mode);
            DatasourceVocabularies vocs = new DatasourceVocabularies();
            vocs.setCountries(this.getCountries());
            vocs.setDatasourceClasses(this.getDatasourceClasses(mode));
            vocs.setTimezones(this.getTimezones());
            vocs.setTypologies(this.getTypologies());
            vocs.setCompatibilityLevels(this.getCompatibilityClasses(mode));

            return vocs;

        } catch (Exception e) {
            LOGGER.error("Error while getting vocabularies for datasource with type: \" + mode", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    private Tuple<List<Repository>, List<Repository>> getRepositoriesByCountryTesting(String country, String mode, boolean includeUnknownCountries) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting testing repositories of country: " + country + " with type: " + mode + " and includeUnknownCountries: " + includeUnknownCountries);

            Tuple<List<Repository>, List<Repository>> retTuple = new Tuple<List<Repository>, List<Repository>>();
            List<Repository> reposList = new ArrayList<Repository>();
            reposList.add(this.repoAPI.getRepository("opendoar____::1356"));
            reposList.add(this.repoAPI.getRepository("opendoar____::2678"));
            reposList.add(this.repoAPI.getRepository("opendoar____::2980"));
            reposList.add(this.repoAPI.getRepository("opendoar____::1347"));
            reposList.add(this.repoAPI.getRepository("opendoar____::2984"));

            retTuple.setFirst(reposList);

            if (includeUnknownCountries) {
                List<Repository> totalList = new ArrayList<Repository>();
                totalList.add(this.repoAPI.getRepository("opendoar____::3000"));
                totalList.add(this.repoAPI.getRepository("opendoar____::1027"));
                totalList.add(this.repoAPI.getRepository("opendoar____::1096"));

                retTuple.setSecond(totalList);
            }

            return retTuple;

        } catch (Exception e) {
            LOGGER.error("Error while getting testing repositories of country: " + country + " with type: " + mode + " and includeUnknownCountries: " + includeUnknownCountries, e);
            if (e instanceof RepositoryServiceException) {
                throw (RepositoryServiceException) e;
            } else {
                emailUtils.reportException(e);
                throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
            }
        }
    }

    @Override
    public String getLatestUpdateDateOfList(String mode) throws RepositoryServiceException {
        try {
            LOGGER.debug("Getting latest update date of list: " + mode);
            return this.repoAPI.getListLatestUpdate(mode).split("T")[0];

        } catch (Exception e) {
            LOGGER.error("Error while getting latest update date of list: " + mode, e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public PiwikInfo getPiwikSiteForRepository(String repoId) throws RepositoryServiceException {
        try {
            LOGGER.debug("Repo id -> " + repoId);
            return this.piwikDAO.getPiwikSiteForRepo(repoId);
        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    @Override
    public void enableMetricsForRepository(Repository repository, UserProfile requestor) throws RepositoryServiceException {
        
        try {
            String URL = analyticsURL + "siteName=" + URLEncoder.encode(repository.getOfficialName(), "UTF-8") + "&url=" + URLEncoder.encode(repository.getWebsiteUrl(), "UTF-8");
            Map<String, Object> map = new ObjectMapper().readValue(new URL(URL), Map.class);

            String siteId = null;
            if(map.get("value")!=null) {
                siteId = map.get("value").toString();
            }

            String authenticationToken = "32846584f571be9b57488bf4088f30ea";

            PiwikInfo piwikInfo = new PiwikInfo();
            piwikInfo.setRepositoryId(repository.getId());
            piwikInfo.setRepositoryName(repository.getOfficialName());
            piwikInfo.setCountry(repository.getCountryName());
            piwikInfo.setSiteId(siteId);
            piwikInfo.setAuthenticationToken(authenticationToken);
            piwikInfo.setRequestorEmail(requestor.getEmail());
            piwikInfo.setRequestorName(requestor.getFirstname() + " " + requestor.getLastname());
            piwikInfo.setValidated(false);

            this.piwikDAO.savePiwikInfo(piwikInfo);

            emailUtils.sendAdministratorRequestToEnableMetrics(piwikInfo);
            emailUtils.sendUserRequestToEnableMetrics(piwikInfo);

        } catch (UnsupportedEncodingException uee) {
            LOGGER.error("Error while creating piwikScript URL", uee);
            emailUtils.reportException(uee);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        } catch (IOException ioe) {
            LOGGER.error("Error while creating piwik site", ioe);
            emailUtils.reportException(ioe);
            throw new RepositoryServiceException("login.generalError", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        } catch (Exception e) {
            LOGGER.error("Error while sending email to administrator or user about the request to enable metrics", e);
            emailUtils.reportException(e);
        }
    }

    @Override
    public String getPiwikScriptForRepository(String repoId) throws RepositoryServiceException {
        try {
            PiwikInfo piwikInfo = this.piwikDAO.getPiwikSiteForRepo(repoId);

            String piwikScript = PIWIK_SCRIPT.replace("$$$", piwikInfo.getSiteId());
            return piwikScript;

        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    @Override
    public List<PiwikInfo> getPiwikSitesForRepositories() throws RepositoryServiceException {
        try {

            List<PiwikInfo> piwikInfos = new ArrayList<>();
            piwikInfos = this.piwikDAO.getPiwikSitesForRepos();

            return piwikInfos;

        } catch (EmptyResultDataAccessException e) {
            LOGGER.error("Error while getting list of piwik sites: ", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("General error", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void markPiwikSiteAsValidated(String repositoryId) throws RepositoryServiceException {
        try {
            this.piwikDAO.markPiwikSiteAsValidated(repositoryId);

            PiwikInfo piwikInfo = this.piwikDAO.getPiwikSiteForRepo(repositoryId);
            emailUtils.sendAdministratorMetricsEnabled(piwikInfo);
            emailUtils.sendUserMetricsEnabled(piwikInfo);

        } catch (EmptyResultDataAccessException e) {
            LOGGER.error("Error while approving piwik site: ", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("General error", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        } catch (Exception e) {
            LOGGER.error("Error while sending email to administrator or user about the enabling of metrics", e);
            emailUtils.reportException(e);
        }
    }

    @Override
    public MetricsInfo getMetricsInfoForRepository(String repoId) throws RepositoryServiceException {
        try {

            MetricsInfo metricsInfo = new MetricsInfo();
            metricsInfo.setDiagramsBaseURL(this.usageStatisticsDiagramsBaseURL);
            metricsInfo.setMetricsNumbers(getMetricsNumbers(getOpenAIREId(repoId)));
            return metricsInfo;

        } catch (Exception e) {
            LOGGER.error("Error while getting metrics info for repository: ", e);
            emailUtils.reportException(e);
            throw new RepositoryServiceException("General error", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    private MetricsNumbers getMetricsNumbers(String openAIREID) throws BrokerException {

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(this.usageStatisticsNumbersBaseURL + openAIREID + "/clicks");

        //create new template engine
        RestTemplate template = new RestTemplate();
        template.getMessageConverters().add(new MappingJackson2HttpMessageConverter());
        ResponseEntity<MetricsNumbers> resp;
        try {
            //communicate with endpoint
            resp = template.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<MetricsNumbers>() {
                    });
        } catch (RestClientException e) {
            throw e;
        }

        return resp.getBody();
    }

    private String getCountryCode(String countryName) {
        Vocabulary countries = this.getVocabulary("dnet:countries");

        return countries.getEncoding(countryName);
    }

    private String getDatasourceClassCode(String datasourceClassName) {
        Vocabulary datasourceClasses = this.getVocabulary("dnet:datasource_typologies");

        return datasourceClasses.getEncoding(datasourceClassName);
    }

    private String getComplianceName(String complianceCode) {
        Vocabulary compatibilityLevels = this.getVocabulary("dnet:compatibilityLevel");

        return compatibilityLevels.getEnglishName(complianceCode);
    }

    private Vocabulary getVocabulary(String vocName) {

        if (!vocabularyMap.containsKey(vocName)) {
            vocabularyMap.put(vocName, vocabularyLoader.getVocabulary(vocName, Locale.ENGLISH, Locale.ROOT));
        }
        return vocabularyMap.get(vocName);
    }

    @Scheduled(fixedRate = 3600000)
    private void loadVocabularies() {
        LOGGER.debug("loading vocabularies");
        for (String vocName : vocabularyNames) {
            vocabularyMap.put(vocName, vocabularyLoader.getVocabulary(vocName, Locale.ENGLISH, Locale.ROOT));
        }
    }

    private String getOpenAIREId(String repoId) {

        if (repoId != null && repoId.contains("::")) {
            return repoId.split("::")[0] + "::" + DigestUtils.md5Hex(repoId.split("::")[1]);
        }

        return null;
    }

    @Override
    public Aggregations getRepositoryAggregations(String repoId) throws Exception {
        return repositoryApi.getRepositoryAggregations(repoId);
    }
}