package eu.dnetlib.repo.manager.server.services;

import eu.dnetlib.domain.functionality.UserProfile;
import eu.dnetlib.gwt.server.service.SpringGwtRemoteServiceServlet;
import eu.dnetlib.repo.manager.client.services.UserService;
import eu.dnetlib.repo.manager.server.utils.EmailUtils;
import eu.dnetlib.repo.manager.shared.Tuple;
import eu.dnetlib.repo.manager.shared.UserAccessException;
import eu.dnetlib.users.UserApi;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Created by nikonas on 12/7/15.
 */
@Service("userService")
public class UserServiceImpl extends SpringGwtRemoteServiceServlet implements UserService {

    private static final Logger LOGGER = Logger
            .getLogger(UserServiceImpl.class);

    @Autowired
    private UserApi userAPI;

    @Autowired
    private EmailUtils emailUtils;


    public void init(ServletConfig config) throws ServletException {

        LOGGER.info("initializing user service impl ");
        super.init(config);

    }

    @Override
    public Tuple<UserProfile, String> login(String email_username, String password) throws UserAccessException {
        LOGGER.info("Checking credentials for user " + email_username);
        try {

            String email = email_username;

            Pattern rfc2822 = Pattern.compile("^[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?$");
            if (!rfc2822.matcher(email_username.trim().toLowerCase()).matches()) {
                LOGGER.debug("user logged in using username");
                email = this.userAPI.getEmailFromUsername(email_username);
            }
            if (email == null) {
                throw new UserAccessException("login.userNotExists", UserAccessException.ErrorCode.INVALID_USERNAME);
            }
            if (!this.userAPI.userExists(email)) {
                throw new UserAccessException("login.userNotExists", UserAccessException.ErrorCode.INVALID_USERNAME);
            }
            if (!this.userAPI.isUserActivated(email)) {
                throw new UserAccessException("login.notActivated", UserAccessException.ErrorCode.NOT_ACTIVATED);
            }
            if (!this.userAPI.correctCreds(email, password)) {
                throw new UserAccessException("login.InvalidPassword", UserAccessException.ErrorCode.INVALID_PASSWORD);
            }

            UserProfile userProfile = this.userAPI.getUser(email);
            String role = "";

            String[] adminEmails = new String[] {"stefania.martziou@gmail.com" , "antleb@di.uoa.gr", "ant.lebesis@gmail.com", "natalia@di.uoa.gr", "pedroprincipe@sdum.uminho.pt", "dpierrakos@gmail.com", "jochen.schirrwagen@uni-bielefeld.de", "aenne.loehden@uni-bielefeld.de"};
            if(Arrays.asList(adminEmails).contains(userProfile.getEmail()))
                role = "admin";

            return new Tuple<>(userProfile, role);

        } catch (Exception e) {
            LOGGER.error("An error occurred while checking credentials for user " + email_username, e);
            emailUtils.reportException(e);

            if (e instanceof UserAccessException) {
                throw (UserAccessException) e;
            }
            else {
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
            }
        }

    }

    @Override
    public Tuple<UserProfile, String> getUserByEmail(String email) throws UserAccessException {
        LOGGER.info("Getting user with email " + email);
        try {

            UserProfile userProfile = this.userAPI.getUser(email);
            String role = "";

            String[] adminEmails = new String[] {"stefania.martziou@gmail.com" , "antleb@di.uoa.gr", "ant.lebesis@gmail.com", "natalia@di.uoa.gr", "pedroprincipe@sdum.uminho.pt", "dpierrakos@gmail.com", "jochen.schirrwagen@uni-bielefeld.de", "aenne.loehden@uni-bielefeld.de"};
            if(Arrays.asList(adminEmails).contains(userProfile.getEmail()))
                role = "admin";

            return new Tuple<>(userProfile, role);

        } catch (Exception e) {
            LOGGER.error("An error occurred while getting user with email " + email, e);
            emailUtils.reportException(e);

            throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void register(UserProfile userProfile) throws UserAccessException {

        try {
            LOGGER.info("Registering user " + userProfile.getEmail());

            Pattern rfc2822 = Pattern.compile("^[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?$");
            if (!rfc2822.matcher(userProfile.getEmail().trim().toLowerCase()).matches()) {
                throw new UserAccessException("login.notValidEmail", UserAccessException.ErrorCode.INVALID_EMAIL_FORMAT);
            }

            if (this.userAPI.usernameExists(userProfile.getUsername())) {
                throw new UserAccessException("login.usernameAlreadyExists", UserAccessException.ErrorCode.USERNAME_ALREADY_EXISTS);
            }
            if (this.userAPI.userExists(userProfile.getEmail())) {
                throw new UserAccessException("login.mailAlreadyExists", UserAccessException.ErrorCode.MAIL_ALREADY_EXISTS);
            }

//            String activationId = "TEST";
            String activationId = this.userAPI.addUser(userProfile.getUsername(), userProfile.getEmail(), userProfile.getPassword(), userProfile.getFirstname(), userProfile.getLastname(), userProfile.getInstitution());

            emailUtils.sendActivationEmail(userProfile, activationId);

        } catch (Exception e) {
            LOGGER.error("Error while registering user " + userProfile.getEmail(), e);
            emailUtils.reportException(e);

            if (e instanceof UserAccessException)
                throw (UserAccessException) e;
            else
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }

    }

    @Override
    public void activateUser(String activationId) throws UserAccessException {
        try {
            LOGGER.info("Activating user with activation with activation id " + activationId);

            if (!this.userAPI.activateUser(activationId))
                throw new UserAccessException("registration.okAccountAlreadyActivation", UserAccessException.ErrorCode.ALREADY_ACTIVATED);
        } catch (Exception e) {
            LOGGER.error("Error while activating user account with activation id " + activationId, e);
            emailUtils.reportException(e);

            if (e instanceof UserAccessException)
                throw (UserAccessException) e;
            else
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void updateUser(UserProfile userProfile) throws UserAccessException {
        try {
            LOGGER.info("Editing user " + userProfile.getUsername());
            Pattern rfc2822 = Pattern.compile("^[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?$");
            if (!rfc2822.matcher(userProfile.getEmail().trim().toLowerCase()).matches()) {
                throw new UserAccessException("login.notValidEmail", UserAccessException.ErrorCode.INVALID_EMAIL_FORMAT);
            }

            String currentEmail = this.userAPI.getEmailFromUsername(userProfile.getUsername());
            if (!userProfile.getEmail().equalsIgnoreCase(currentEmail)) {
                if (this.userAPI.userExists(userProfile.getEmail())) {
                    throw new UserAccessException("login.mailAlreadyExists", UserAccessException.ErrorCode.MAIL_ALREADY_EXISTS);
                }
            }

            this.userAPI.editUser(userProfile);

        } catch (Exception e) {
            LOGGER.error("Error while editing user " + userProfile.getUsername(), e);
            if (e instanceof UserAccessException)
                throw (UserAccessException) e;
            else
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void prepareResetPassword(String email) throws UserAccessException {

        try {
            LOGGER.debug("Sending password recovery to user " + email);
            if (!this.userAPI.userExists(email)) {
                throw new UserAccessException("login.userNotExists", UserAccessException.ErrorCode.INVALID_USERNAME);
            }
            List<String> to = new ArrayList<String>();
            to.add(email);
            String securityCode = this.userAPI.prepareResetPassword(email);

            emailUtils.sendResetPasswordEmail(email, securityCode);

        } catch (Exception e) {
            LOGGER.error("Error while sending password recovery to user " + email, e);
            emailUtils.reportException(e);

            if (e instanceof UserAccessException)
                throw (UserAccessException) e;
            else
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void resetPassword(String securityCode, String password) throws UserAccessException {
        try {
            LOGGER.debug("Reseting password with security code " + securityCode);

            if (securityCode.length() == 0) {
                throw new UserAccessException("resetPassword.wrongSecurityCode", UserAccessException.ErrorCode.WRONG_SECURITY_CODE);
            }

            this.userAPI.resetPassword(securityCode, password);

        } catch (Exception e) {
            LOGGER.error("Error while reseting password with security code " + securityCode);
            emailUtils.reportException(e);

            if (e instanceof UserAccessException)
                throw (UserAccessException) e;
            else
                throw new UserAccessException("login.generalError", UserAccessException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void resendActivation(String email) throws UserAccessException {

    }

}
