package eu.dnetlib.repo.manager.server.services;

import eu.dnetlib.api.functionality.ValidatorService;
import eu.dnetlib.api.functionality.ValidatorServiceException;
import eu.dnetlib.domain.functionality.validator.JobForValidation;
import eu.dnetlib.domain.functionality.validator.RuleSet;
import eu.dnetlib.domain.functionality.validator.StoredJob;
import eu.dnetlib.gwt.server.service.SpringGwtRemoteServiceServlet;
import eu.dnetlib.repo.manager.client.services.ValidationService;
import eu.dnetlib.repo.manager.server.utils.EmailUtils;
import eu.dnetlib.repo.manager.server.utils.OaiTools;
import eu.dnetlib.repo.manager.service.controllers.MonitorApi;
import eu.dnetlib.repo.manager.service.controllers.ValidatorApi;
import eu.dnetlib.repo.manager.shared.*;
import gr.uoa.di.driver.util.ServiceLocator;
import org.apache.log4j.Logger;
import org.json.JSONException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by nikonas on 22/12/15.
 */
@Service("validationService")
public class ValidationServiceImpl extends SpringGwtRemoteServiceServlet implements ValidationService {

    private static final Logger LOGGER = Logger
            .getLogger(ValidationServiceImpl.class);

    @Autowired
    private EmailUtils emailUtils;

    @Value("${services.repo-manager.deploy.environment}")
    private String deployEnvironment;

    private ServiceLocator<ValidatorService> validatorServiceLocator;

    //private Map<String, List<RuleSet>> rulesetMap = new ConcurrentHashMap<String, List<RuleSet>>();

    public void init(ServletConfig config) throws ServletException {

        LOGGER.info("initializing validation service impl ");
        super.init(config);
    }

    @Autowired
    private ValidatorApi validatorApi;
    @Autowired
    private MonitorApi monitorApi;

    @PostConstruct
    public void initRulesets() {
        //this.loadRulesets();
    }

    @Override
    public List<String> getSets(String baseUrl) throws ValidationServiceException {
        return validatorApi.getSetsOfRepository(baseUrl);
    }

    @Override
    public Boolean identifyRepository(String baseUrl) throws ValidationServiceException {
        return validatorApi.identifyRepo(baseUrl);
    }

    @Override
    public InterfaceInformation getInterfaceInformation(String baseUrl) throws ValidationServiceException {
        try {
            LOGGER.debug("Getting interface information with url: " + baseUrl);
            InterfaceInformation interfaceInformation = new InterfaceInformation();
            interfaceInformation.setIdentified(this.identifyRepository(baseUrl));
            if (interfaceInformation.isIdentified())
                interfaceInformation.setSets(this.getSets(baseUrl));

            return interfaceInformation;
        } catch (Exception e) {
            LOGGER.error("Error getting interface information with url: " + baseUrl, e);
//            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public List<RuleSet> getRuleSets(String validationMode) throws ValidationServiceException {
        try {
            return validatorApi.getRuleSets(validationMode);
        } catch (Exception e) {
            LOGGER.error("Error getting rulesets", e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public RuleSet getRuleSet(String acronym) throws ValidationServiceException {
        return validatorApi.getRuleSet(acronym);
    }

    @Override
    public void submitValidationJob(JobForValidation job) throws ValidationServiceException {
        try {
            validatorApi.submitJobForValidation(job);
        } catch (Exception e) {
            LOGGER.error("Error while submitting validation job", e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public void reSubmitValidationJob(int jobId)
            throws ValidationServiceException {
        try {
            validatorApi.reSubmitJobForValidation(String.valueOf(jobId));
        } catch (JSONException e) {
            LOGGER.error("Error while re-submitting validation job", e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public StoredJob getJobSummary(int jobId, String groupBy)
            throws ValidationServiceException {

        try {
            return monitorApi.getJobSummary(String.valueOf(jobId),groupBy);
        } catch (Exception e) {
            LOGGER.error("Error getting job summary for id: " + jobId, e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public List<StoredJob> getJobsOfUser(String user, String jobType, Integer offset, Integer limit, String dateFrom, String dateTo, String validationStatus) throws ValidationServiceException {
        try {
            LOGGER.debug("getting jobs of user " + user);
            List<StoredJob> jobs;
            //TODO update after declaring roles
//            if (this.userOverridesRepoRegistration(user))
//                jobs = getValidationService().getStoredJobs(null, jobType, offset, limit, null, null);
//            else
            jobs = getValidationService().getStoredJobsNew(user, jobType, offset, limit, dateFrom, dateTo, validationStatus);
            return jobs;
//            return monitorApi.getJobsOfUser(user, jobType, offset, limit, dateFrom, dateTo, validationStatus);
        } catch (Exception e) {
            LOGGER.error("Error getting jobs of user " + user, e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public int getJobsTotalNumberOfUser(String user, String jobType, String validationStatus) throws ValidationServiceException {
        try {
            LOGGER.debug("getting jobs of user " + user);
            int sum;
            //TODO update after declaring roles
//            if (this.userOverridesRepoRegistration(user))
//                sum = getValidationService().getStoredJobsTotalNumber(null, jobType);
//            else
            sum = getValidationService().getStoredJobsTotalNumberNew(user, jobType, validationStatus);
            return sum;
        } catch (Exception e) {
            LOGGER.error("Error getting jobs of user " + user, e);
            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    @Override
    public JobsOfUser getJobsOfUser(String user, String jobType, Integer offset, Integer limit, String dateFrom, String dateTo, String validationStatus, boolean includeJobsTotal) throws ValidationServiceException {
        try {
            LOGGER.debug("getting jobs of user " + user + " totalJobs included: " + includeJobsTotal);
            JobsOfUser retJobs = new JobsOfUser();

            return monitorApi.getJobsOfUser(user, jobType, String.valueOf(offset), String.valueOf(limit),
                    dateFrom, dateTo, validationStatus, String.valueOf(includeJobsTotal));

        } catch (Exception e) {
            LOGGER.error("getting jobs of user " + user + " totalJobs included: " + includeJobsTotal, e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }


    private ValidatorService getValidationService() {
        return this.validatorServiceLocator.getService();
    }

    public ServiceLocator<ValidatorService> getValidatorServiceLocator() {
        return validatorServiceLocator;
    }

    public void setValidatorServiceLocator(ServiceLocator<ValidatorService> validatorServiceLocator) {
        this.validatorServiceLocator = validatorServiceLocator;
    }
}
