package eu.dnetlib.repo.manager.service.utils;

import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.domain.data.RepositoryInterface;
import eu.dnetlib.repo.manager.shared.*;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

public class Converter {

    private static final Logger LOGGER = Logger.getLogger(Converter.class);
    
    public static Repository jsonToRepositoryObject(JSONObject repositoryObject) throws JSONException {

        Repository repository = new Repository();


        LOGGER.debug("datasource response -> " + repositoryObject);
        JSONObject datasource = repositoryObject.getJSONObject("datasource");

        if( datasource.equals(null))
            return null;
        
        repository.setActivationId(datasource.get("activationId").toString());
        repository.setAggregator(datasource.get("aggregator").toString());
        repository.setCertificates(datasource.get("certificates").toString());
        repository.setCitationGuidelineUrl(datasource.get("citationguidelineurl").toString());
        repository.setCollectedFrom( datasource.get("collectedfrom").toString());

        repository.setContactEmail(datasource.get("contactemail").toString());
        if(repository.getContactEmail().equals("null"))
            repository.setContactEmail("");

        repository.setDatabaseAccessRestriction(datasource.get("databaseaccessrestriction").toString());
        repository.setDatabaseAccessType(datasource.get("databaseaccesstype").toString());
        repository.setDataUploadRestriction(datasource.get("datauploadrestriction").toString());
        repository.setDataUploadType(datasource.get("datauploadtype").toString());
        repository.setDateOfCollection(convertStringToDate( datasource.get("dateofcollection").toString()));
        repository.setDateOfValidation(convertStringToDate( datasource.get("dateofvalidation").toString()));

        repository.setDescription(datasource.get("description").toString());
        if(repository.getDescription().equals("null"))
            repository.setDescription("");

        repository.setEissn(datasource.get("eissn").toString());

        repository.setEnglishName( datasource.get("englishname").toString());
        if(repository.getEnglishName().equals("null"))
            repository.setEnglishName("");


        repository.setId(datasource.get("id").toString());
        repository.setIssn(datasource.get("issn").toString());
        repository.setOdLanguages(datasource.get("languages").toString());
        repository.setLatitude( toDouble(datasource.get("latitude").toString()));
        repository.setLissn(datasource.get("lissn").toString());

        repository.setLogoUrl(datasource.get("logourl").toString());
        if(repository.getLogoUrl().equals("null"))
            repository.setLogoUrl("");

        repository.setLongitude(toDouble(datasource.get("longitude").toString()));
        //datasource.get("managed");
        repository.setMissionStatementUrl(datasource.get("missionstatementurl").toString());
        repository.setNamespacePrefix(datasource.get("namespaceprefix").toString());
        repository.setOdContentTypes(datasource.get("od_contenttypes").toString());
        repository.setOfficialName(datasource.get("officialname").toString());
        if(repository.getOfficialName().equals("null"))
            repository.setOfficialName("");

        repository.setPidSystems(datasource.get("pidsystems").toString());
        //datasource.get("platform");
        repository.setProvenanceActionClass( datasource.get("provenanceaction").toString());
        repository.setQualityManagementKind(datasource.get("qualitymanagementkind").toString());
        repository.setRegisteredBy(datasource.get("registeredby").toString());

        if(Objects.equals(repository.getRegisteredBy(),"null"))
            repository.setRegistered(true);

        repository.setReleaseEndDate(convertStringToDate(datasource.get("releaseenddate").toString()));
        repository.setReleaseStartDate(convertStringToDate(datasource.get("releasestartdate").toString()));
        repository.setServiceProvider(Boolean.valueOf(datasource.get("serviceprovider").toString()));
        //datasource.get("subjects");
        Double timezone = toDouble(datasource.get("timezone").toString());
        repository.setTimezone(timezone!=null?timezone:0.0);
        repository.setTypology(datasource.get("platform").toString());
        repository.setVersioning(Boolean.valueOf(datasource.get("versioning").toString()));
        repository.setWebsiteUrl(datasource.get("websiteurl").toString());
        repository.setDatasourceClass(datasource.get("typology").toString());

        //TODO change organization to list
        repository.setOrganization( ((JSONArray)datasource.get("organizations")).getJSONObject(0).get("legalname").toString());
        String countryCode = ((JSONArray)datasource.get("organizations")).getJSONObject(0).get("country").toString();
        repository.setCountryCode(countryCode);


        String collectedFrom = datasource.get("collectedfrom").toString();
        //TODO check data consistency
        String type = "UNKNOWN";
        if (collectedFrom.equalsIgnoreCase("openaire____::opendoar")) {
            type = "opendoar";
        } else if (collectedFrom.equalsIgnoreCase("openaire____::re3data")) {
            type = "re3data";
        } else if (collectedFrom.equalsIgnoreCase("infrastruct_::openaire")) {
            type = "journal";
        }

        repository.setDatasourceType(type);


        return repository;
    }



    public static Date convertStringToDate(String date){

        if(Objects.equals(date, "null"))
            return null;

        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        try {
            return formatter.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static String convertDateToString(Date date){

        if(Objects.equals(date, null))
            return null;

        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        return formatter.format(date);
    }

    public static Double toDouble(String number){
        if(Objects.equals(number, "null"))
            return 0.0;
        else
            return Double.valueOf(number);
    }

    public static List<Repository> jsonToRepositoryList(JSONObject json) throws JSONException {

        List<Repository> resultSet = new ArrayList<>();
        JSONArray rs = json.getJSONArray("datasourceInfo");
        for(int i=0;i<rs.length();i++)
            resultSet.add(jsonToRepositoryObject( rs.getJSONObject(i)) );
        return resultSet;
    }

    public static List<RepositoryInterface> jsonToRepositoryInterfaceList(JSONArray rs) throws JSONException {

        List<RepositoryInterface> resultSet = new ArrayList<>();
        for(int i=0;i<rs.length();i++)
            resultSet.add(jsonToRepositoryInterfaceObject( rs.getJSONObject(i)) );
        return resultSet;
    }

    public static RepositoryInterface jsonToRepositoryInterfaceObject(JSONObject repositoryInterfaceObject) throws JSONException {

        RepositoryInterface repositoryInterface = new RepositoryInterface();

        repositoryInterface.setBaseUrl(repositoryInterfaceObject.get("baseurl").toString());
        repositoryInterface.setContentDescription(repositoryInterfaceObject.get("contentdescription").toString());
        repositoryInterface.setId(repositoryInterfaceObject.get("id").toString());
        repositoryInterface.setMetadataIdentifierPath(repositoryInterfaceObject.get("metadataIdentifierPath").toString());
        repositoryInterface.setAccessProtocol(repositoryInterfaceObject.get("protocol").toString());
        repositoryInterface.setTypology(repositoryInterfaceObject.get("typology").toString());
        repositoryInterface.setDesiredCompatibilityLevel(repositoryInterfaceObject.get("compatibility").toString());
        repositoryInterface.setActive(Boolean.parseBoolean(repositoryInterfaceObject.get("active").toString()));
        repositoryInterface.setRemovable(Boolean.parseBoolean(repositoryInterfaceObject.get("removable").toString()));
        repositoryInterface.setCompliance(repositoryInterfaceObject.get("compatibility").toString());

        Map<String, String> accessParams = new HashMap<>();
        Map<String, String> extraFields = new HashMap<>();

        ObjectMapper mapper = new ObjectMapper();
        JSONArray apiparams = repositoryInterfaceObject.getJSONArray("apiparam");

        for(int i=0;i<apiparams.length();i++)
            accessParams.put(apiparams.getJSONObject(i).getString("param"),apiparams.getJSONObject(i).getString("value"));

        return repositoryInterface;
    }

    public static String repositoryObjectToJson(Repository repository) throws JSONException {

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("activationId",repository.getActivationId());
        jsonObject.put("aggregator",repository.getAggregator());
        jsonObject.put("certificates",repository.getCertificates());
        jsonObject.put("citationguidelineurl",repository.getCitationGuidelineUrl());
        jsonObject.put("collectedfrom",repository.getCollectedFrom());
        jsonObject.put("contactemail",repository.getContactEmail());
        jsonObject.put("databaseaccessrestriction",repository.getDatabaseAccessRestriction());
        jsonObject.put("databaseaccesstype",repository.getDatabaseAccessType());
        jsonObject.put("datauploadrestriction",repository.getDataUploadRestriction());
        jsonObject.put("datauploadtype",repository.getDataUploadType());
        jsonObject.put("dateofcollection",convertDateToString(repository.getDateOfCollection()));
        jsonObject.put("dateofvalidation",convertDateToString(repository.getDateOfValidation()));
        jsonObject.put("description",repository.getDescription());
        jsonObject.put("eissn",repository.getEissn());
        jsonObject.put("englishname",repository.getEnglishName());
        jsonObject.put("id",repository.getId());
        jsonObject.put("issn",repository.getIssn());
        jsonObject.put("languages",repository.getOdLanguages());
        jsonObject.put("latitude",repository.getLatitude().toString());
        jsonObject.put("lissn",repository.getLissn());
        jsonObject.put("logourl",repository.getLogoUrl());
        jsonObject.put("longitude",repository.getLongitude().toString());
        jsonObject.put("missionstatementurl",repository.getMissionStatementUrl());
        jsonObject.put("namespaceprefix",repository.getNamespacePrefix());
        jsonObject.put("od_contenttypes",repository.getOdContentTypes());
        jsonObject.put("officialname",repository.getOfficialName());
        jsonObject.put("pidsystems",repository.getPidSystems());
        jsonObject.put("provenanceaction",repository.getProvenanceActionClass());
        jsonObject.put("qualitymanagementkind",repository.getQualityManagementKind());
        jsonObject.put("registeredby",repository.getRegisteredBy());
        jsonObject.put("releaseenddate",convertDateToString(repository.getReleaseEndDate()));
        jsonObject.put("releasestartdate",convertDateToString(repository.getReleaseStartDate()));
        jsonObject.put("serviceprovider",repository.getServiceProvider());
        jsonObject.put("timezone",repository.getTimezone());
        jsonObject.put("typology",repository.getTypology());
        jsonObject.put("versioning",repository.getVersioning());
        jsonObject.put("websiteurl",repository.getWebsiteUrl());

        //datasource.get("managed");
        //datasource.get("platform");
        //datasource.get("subjects");
        return jsonObject.toString();
    }

    public static String repositoryInterfaceObjectToJson(Repository repository,RepositoryInterface repositoryInterface) throws JSONException {

        JSONObject jsonObject = new JSONObject();

        jsonObject.put("baseurl",repositoryInterface.getBaseUrl());
        jsonObject.put("contentdescription",repositoryInterface.getContentDescription());
        jsonObject.put("id",repositoryInterface.getId());
        jsonObject.put("metadataIdentifierPath",repositoryInterface.getMetadataIdentifierPath());
        jsonObject.put("protocol",repositoryInterface.getAccessProtocol());
        jsonObject.put("typology",repositoryInterface.getTypology());
        jsonObject.put("compatibility",repositoryInterface.getDesiredCompatibilityLevel());
        jsonObject.put("datasource",repository.getId());
        jsonObject.put("metadataIdentifierPath",repositoryInterface.getMetadataIdentifierPath());
        jsonObject.put("protocol",repositoryInterface.getAccessProtocol());
        jsonObject.put("removable",repositoryInterface.isRemovable());
        jsonObject.put("active",repositoryInterface.isActive());


        JSONArray apiparams = new JSONArray();
        for(String param: repositoryInterface.getAccessParams().keySet()){
            JSONObject jo = new JSONObject();
            jo.put("param",param);
            jo.put("value",repositoryInterface.getAccessParams().get(param));
            apiparams.put(jo);
        }
        jsonObject.put("apiparam",apiparams);


        jsonObject.put("lastCollectionDate",repositoryInterface.getLastCollectionDate());
        //jsonObject.put("lastCollectionMdid",repositoryInterface);
        //jsonObject.put("lastCollectionTotal");
        //jsonObject.put("lastDownloadDate");
//        jsonObject.put("lastDownloadMdid");
//        jsonObject.put("lastDownloadTotal");
//        jsonObject.put("lastValidationJob");
        //jsonObject.put("lastAggregationDate");
        //jsonObject.put("lastAggregationMdid");
        //jsonObject.put("lastAggregationTotal");

        return jsonObject.toString();
    }

    public static ArrayList<String> readFile(String filename) {
        String line;
        ArrayList<String> list = new ArrayList<String>();
        try {
            //InputStream in = Converter.class.getResourceAsStream("resources/eu/dnetlib/repo/manager/service/utils/"+filename);
            InputStream in = Converter.class.getClass().getResourceAsStream("/eu/**/" + filename);
            BufferedReader br = new BufferedReader(new InputStreamReader(in));
            while((line = br.readLine()) != null) {
                list.add(line.trim());
            }
            br.close();
        } catch (IOException e) {
            LOGGER.debug("Error opening file!");
            e.printStackTrace();
        }
        return list;
    }

    public static List<AggregationDetails> getAggregationHistoryFromJson(JSONObject repositoryObject) throws JSONException {

       /* if( repositoryObject.get("aggregationHistory").toString().equals("[]") ||
                repositoryObject.get("aggregationHistory")!= null)
            return null;*/
       if(repositoryObject.get("aggregationHistory").toString().equals("[]"))
           return null;


        JSONArray rs = new JSONArray(repositoryObject.get("aggregationHistory").toString());

        LOGGER.debug(rs.length());

        List<AggregationDetails> aggregationDetailsList = new ArrayList<>();
        for(int i=0;i<rs.length();i++)
            aggregationDetailsList.add(jsonToAggregationDetails(rs.getJSONObject(i)));
        return aggregationDetailsList;
    }

    private static AggregationDetails jsonToAggregationDetails(JSONObject aggregationObject) throws JSONException {

        AggregationDetails aggregationDetails = new AggregationDetails();

        aggregationDetails.setAggregationStage(aggregationObject.get("aggregationStage").toString());
        if(aggregationObject.has("collectionMode"))
            aggregationDetails.setCollectionMode(aggregationObject.get("collectionMode").toString());
        aggregationDetails.setDate(convertStringToDate(aggregationObject.get("date").toString()));
        aggregationDetails.setNumberOfRecords(Integer.parseInt(aggregationObject.get("numberOfRecords").toString()));
        return aggregationDetails;
    }

    public static AggregationDetails getLastCollectionFromJson(JSONObject repositoryObject) throws JSONException {

        if( repositoryObject.get("lastCollection").equals(null))
            return null;

        return jsonToAggregationDetails(repositoryObject.getJSONObject("lastCollection"));
    }

    public static AggregationDetails getLastTransformationFromJson(JSONObject repositoryObject) throws JSONException {

        if( repositoryObject.get("lastTransformation").equals(null))
            return null;

        return jsonToAggregationDetails(repositoryObject.getJSONObject("lastTransformation"));
    }

    public static List<Timezone> toTimezones(List<String> timezones) {

        List<Timezone> tmz = new ArrayList<>();
        for(String t : timezones){
            String[] s = t.split("\t");
            tmz.add(new Timezone(s[1],Double.parseDouble(s[0])));
        }
        return tmz;
    }
}