package eu.dnetlib.repo.manager.service.controllers;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.repo.manager.shared.BrokerException;
import eu.dnetlib.repo.manager.shared.Term;
import eu.dnetlib.repo.manager.shared.Tuple;
import eu.dnetlib.repo.manager.shared.broker.*;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;

@Component
public class BrokerApiImpl implements BrokerApi {

    @Autowired
    private RepositoryApiImpl repoAPI;
    @Value("${services.broker.url}:${services.broker.port}/${services.broker.api}${services.broker.openaire}")
    private String openairePath;
    @Value("${services.broker.url}:${services.broker.port}/${services.broker.api}")
    private String apiPath;
    @Value("${topic_types.url}")
    private String topicsURL;

    private static final org.apache.log4j.Logger LOGGER = org.apache.log4j.Logger
            .getLogger(BrokerApiImpl.class);

    private RestTemplate restTemplate = null;

    private HttpHeaders httpHeaders;

    private HashMap<String,Term> topics = new HashMap<String, Term>();

    @PostConstruct
    private void initDnetTopicsMap() {

        restTemplate = new RestTemplate();
        restTemplate.getMessageConverters().add(new MappingJackson2HttpMessageConverter());

        httpHeaders = new HttpHeaders();
        httpHeaders.set("Content-Type", "application/json");

        LOGGER.debug("Init dnet topics!");
        InputStream is = null;
        try {
            is = new URL(topicsURL).openStream();
            ObjectMapper mapper = new ObjectMapper();
            JsonNode root = mapper.readTree(is);
            for (JsonNode term : root.path("terms") )
                topics.put(term.path("code").textValue(), parseTerm(term));
        } catch (IOException e) {
            LOGGER.debug(e);
            e.printStackTrace();
        }
    }

    private Term parseTerm(JsonNode term) {
        return new Term(term.path("englishName").textValue(),term.path("nativeName").textValue(),
                term.path("encoding").textValue(),term.path("code").textValue());
    }


    @Override
    public DatasourcesBroker getDatasourcesOfUser(String params) throws JSONException {
        JSONObject json_params = new JSONObject(params);
        DatasourcesBroker ret = new DatasourcesBroker();

        String userEmail = json_params.getString("userEmail");
        boolean includeShared = Boolean.parseBoolean( json_params.getString("includeShared") );
        boolean includeByOthers = Boolean.parseBoolean( json_params.getString("includeByOthers") );

        try {
            ret.setDatasourcesOfUser(getDatasourcesOfUserType(getRepositoriesOfUser(userEmail)));
            if (includeShared) {
                //TODO whatever nikonas was saying
                List<String> sharedDatasourceIds = new ArrayList<String>();
                ret.setSharedDatasources(getDatasourcesOfUserType(getRepositoriesByIds(sharedDatasourceIds)));
            }

            if (includeByOthers) {
                ret.setDatasourcesOfOthers(getDatasourcesOfUserType(getRepositoriesOfUser(userEmail)));
            }
        } catch (BrokerException e) {
            e.printStackTrace();
        }

        return ret;
    }

    @Override
    public List<BrowseEntry> getTopicsForDatasource(@PathVariable("datasourceName")  String datasourceName) throws BrokerException {
        final String service = "/topicsForDatasource";

        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(openairePath + service)
                .queryParam("ds", datasourceName);

        ResponseEntity<List<BrowseEntry>> resp;
        try {
            resp = restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<List<BrowseEntry>>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }

        return resp.getBody();
    }

    @Override
    public EventsPage advancedShowEvents(String params) throws BrokerException, JSONException ,IOException {
        JSONObject json_params = new JSONObject(params);

        String page = json_params.getString("page");
        String pagesize = json_params.getString("pagesize");
        String json_advQueryObject = json_params.getString("advQueryObject");

        ObjectMapper mapper = new ObjectMapper();
        AdvQueryObject advQueryObject = mapper.readValue(json_advQueryObject, AdvQueryObject.class);

        final String service = "/events/{page}/{pageSize}";

        Map<String, Long> uriParams = new HashMap<>();
        uriParams.put("page", Long.parseLong(page));
        uriParams.put("pageSize", Long.parseLong(pagesize));

        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(openairePath + service);

        MultiValueMap<String, String> headers = new LinkedMultiValueMap<>();
        headers.add("Content-Type", "application/json");

        advQueryObject.setPage(Long.parseLong(page));

        HttpEntity<AdvQueryObject> entity = new HttpEntity<>(advQueryObject, headers);

        ResponseEntity<EventsPage> resp;
        try {
            resp = restTemplate.exchange(
                    builder.buildAndExpand(uriParams).encode().toUri(),
                    HttpMethod.POST,
                    entity,
                    new ParameterizedTypeReference<EventsPage>() {
                    }
            );
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }
        return resp.getBody();


    }


    private List<Tuple<BrowseEntry, String>> getDatasourcesOfUserType(List<Repository> repositories) throws BrokerException {

        List<Tuple<BrowseEntry, String>> entries = new ArrayList<>();
        for (Repository repo : repositories) {
            BrowseEntry temp = new BrowseEntry();
            temp.setValue(repo.getOfficialName());
            temp.setSize(new Long(0));
            for (BrowseEntry e : getTopicsForDatasource(repo.getOfficialName())) {
                temp.setSize(temp.getSize() + e.getSize());
            }
            Tuple<BrowseEntry, String> tup = new Tuple<>(temp, repo.getLogoUrl());
            entries.add(tup);
        }

        // sort the collection by the second field of the tuple which is size
        Collections.sort(entries, new Comparator<Tuple<BrowseEntry, String>>() {
            @Override
            public int compare(Tuple<BrowseEntry, String> e1, Tuple<BrowseEntry, String> e2) {
                return (int) (e2.getFirst().getSize().longValue() - e1.getFirst().getSize().longValue());
            }
        });

        return entries;
    }



    private List<Repository> getRepositoriesOfUser(String userEmail) throws JSONException {

        int page = 0;
        int size = 50;
        List<Repository> rs ;
        List<Repository> resultSet = new ArrayList<>();

        while (true){
            rs = repoAPI.getRepositoriesOfUser(userEmail, String.valueOf(page), String.valueOf(size));
            resultSet.addAll(rs);
            page+=1;
            if(rs.size() == 0) break;
        }
        return resultSet;
    }

    private List<Repository> getRepositoriesByIds(List<String> sharedDatasourceIds) {
        return null;
    }

    @Override
    public EventsPage showEvents(String params) throws BrokerException, JSONException {

        JSONObject json_params = new JSONObject(params);

        String datasourceName = json_params.getString("datasourceName");
        String topic = json_params.getString("topic");
        String page = json_params.getString("page");

        final String service = "/showEvents";

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(openairePath + service)
                .queryParam("ds", datasourceName)
                .queryParam("topic", topic)
                .queryParam("page", page);

        ResponseEntity<EventsPage> resp;
        try {
            //communicate with endpoint
            resp = restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<EventsPage>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }
        return resp.getBody();
    }

    @Override
    public Map<String, List<SimpleSubscriptionDesc>> getSimpleSubscriptionsOfUser(@PathVariable("userEmail") String userEmail) throws BrokerException {

        final String service = "/subscriptions";

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(openairePath + service)
                .queryParam("email", userEmail);

        //create new template engine
        ResponseEntity<Map<String, List<SimpleSubscriptionDesc>>> resp;
        try {
            //communicate with endpoint
            resp = restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<Map<String, List<SimpleSubscriptionDesc>>>() {
                    });
        } catch (RestClientException e) {
            LOGGER.debug("Error " , e);
            throw new BrokerException(e);
        }
        return resp.getBody();
    }

    @Override
    public Subscription subscribe(OpenaireSubscription obj) throws BrokerException {
        final String service = "/subscribe";

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(openairePath + service);

        HttpEntity<OpenaireSubscription> entity = new HttpEntity<>(obj, httpHeaders);

        //create new template engine
        RestTemplate template = new RestTemplate();
        template.getMessageConverters().add(new MappingJackson2HttpMessageConverter());
        ResponseEntity<Subscription> resp;
        try {
            //communicate with endpoint
            resp = restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.POST,
                    entity,
                    new ParameterizedTypeReference<Subscription>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }

        return resp.getBody();
    }

    @Override
    public void unsubscribe(@PathVariable("subscriptionId") String subscriptionId) throws BrokerException {
        final String service = "/subscriptions/" + subscriptionId;

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(apiPath + service);

        try {
            //communicate with endpoint
            restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.DELETE,
                    null,
                    new ParameterizedTypeReference<Void>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }
    }

    @Override
    public Subscription getSubscription(@PathVariable("subscriptionId") String subscriptionId) throws BrokerException {
        final String service = "/subscriptions/" + subscriptionId;

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(apiPath + service);

        ResponseEntity<Subscription> resp;
        try {
            //communicate with endpoint
            resp = restTemplate.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<Subscription>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }
        return resp.getBody();
    }

    @Override
    public Map<String, Term> getDnetTopics() throws BrokerException {
        return topics;
    }

    @Override
    public EventsPage getNotificationsBySubscriptionId(@PathVariable("subscriptionId") String subscriptionId,
                                                       @PathVariable("page") String page,
                                                       @PathVariable("size") String size
                                                       ) throws BrokerException {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(openairePath + "/notifications/")
                .path("/{id}/{page}/{size}/")
                .build().expand(subscriptionId,page, size).encode();

        ResponseEntity<EventsPage> resp;
        try {
            resp = restTemplate.exchange(
                    uriComponents.toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<EventsPage>() {
                    });
        } catch (RestClientException e) {
            throw new BrokerException(e);
        }
        return resp.getBody();
    }

    @Override
    public Map<String, List<Subscription>> getSubscriptionsOfUser(@PathVariable("userEmail") String userEmail)
            throws BrokerException {

        Map<String, List<SimpleSubscriptionDesc>> simpleSubs = getSimpleSubscriptionsOfUser(userEmail);
        Map<String,List<Subscription>> subs = new HashMap<>();
        List<Subscription> subscriptions = null;

        for(String s:simpleSubs.keySet()){
            List<SimpleSubscriptionDesc> simpleSubscriptionDescs = simpleSubs.get(s);
            for(SimpleSubscriptionDesc simpleSubscriptionDesc : simpleSubscriptionDescs) {
                subscriptions = new ArrayList<>();
                subscriptions.add(getSubscription(simpleSubscriptionDesc.getId()));
            }
            subs.put(s,subscriptions);
        }
        return subs;
    }


}
