package eu.dnetlib.repo.manager.service.controllers;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.domain.data.RepositoryInterface;
import eu.dnetlib.domain.enabling.Vocabulary;
import eu.dnetlib.repo.manager.service.utils.Converter;
import eu.dnetlib.repo.manager.shared.*;
import gr.uoa.di.driver.enabling.vocabulary.VocabularyLoader;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.PostConstruct;
import javax.ws.rs.QueryParam;
import java.sql.Timestamp;
import java.text.Normalizer;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

@Component
public class RepositoryApiImpl implements RepositoryApi {

    @Value("${api.baseAddress}")
    private String baseAddress;

    private RestTemplate restTemplate = null;

    private HttpHeaders httpHeaders;

    private final String[] vocabularyNames = {"dnet:countries", "dnet:datasource_typologies", "dnet:compatibilityLevel"};

    private static final Logger LOGGER = Logger.getLogger(RepositoryApiImpl.class);

    @Value("${services.repomanager.usageStatisticsDiagramsBaseURL}")
    private String usageStatisticsDiagramsBaseURL;

    @Value("${services.repomanager.usageStatisticsNumbersBaseURL}")
    private String usageStatisticsNumbersBaseURL;

    @Autowired
    private VocabularyLoader vocabularyLoader;

    @Autowired
    private PiWikApi piWikApi;

    private Map<String, Vocabulary> vocabularyMap = new ConcurrentHashMap<String, Vocabulary>();

    private Map<String, String> countriesMap = new HashMap<>();
    private Map<String, String> inverseCountriesMap = new HashMap<>();


    @PostConstruct
    private void init() {
        LOGGER.debug("Initialization method of repository api!");

        restTemplate = new RestTemplate();
        restTemplate.getMessageConverters().add(new MappingJackson2HttpMessageConverter());

        httpHeaders = new HttpHeaders();
        httpHeaders.set("Content-Type", "application/json");

        for (String vocName : vocabularyNames) {
            vocabularyMap.put(vocName, vocabularyLoader.getVocabulary(vocName, Locale.ENGLISH, Locale.ROOT));
        }

        Country[] countries = getCountries();
        for (Country c : countries) {
            countriesMap.put(c.getName(), c.getCode());
            inverseCountriesMap.put(c.getCode(), c.getName());
        }


    }

    @Override
    public List<String> testAggregations() throws JSONException {

        int page = 0;
        int size = 1000;

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/list/")
                .path("/{page}/{size}/")
                .build().expand(page, size).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        List<String> ids = new ArrayList<>();
        while (!rs.equals("[]")) {

            ids.addAll(getIdsWithNonEmptyAggregations(rs));

            LOGGER.debug("Checked " + (page + 1) * size + " records!");

            page += 1;
            uriComponents = UriComponentsBuilder
                    .fromHttpUrl(baseAddress + "/ds/list/")
                    .path("/{page}/{size}/")
                    .build().expand(page, size).encode();
            rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        }

        return ids;
    }

    private List<String> getIdsWithNonEmptyAggregations(String rs) throws JSONException {

        JSONArray ids = new JSONArray(rs);
        List<String> agg_ids = new ArrayList<>();
        for (int i = 0; i < ids.length(); i++) {
            String id = ids.getString(i);
            Aggregations aggregations = getRepositoryAggregations(id);
            if (aggregations.getAggregationHistory() != null)
                agg_ids.add(id);
        }
        return agg_ids;
    }

    @Override
    public Country[] getCountries() {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/countries")
                .build().encode();
        return restTemplate.getForObject(uriComponents.toUri(), Country[].class);
    }


    @Override
    public List<Repository> getRepositoriesByCountry(@PathVariable("country") String country,
                                                     @PathVariable("mode") String mode,
                                                     @RequestParam(value = "managed",required=false) Boolean managed) throws JSONException {

        LOGGER.debug("Getting repositories by country!");
        int page = 0;
        int size = 100;

        String countryCode = countriesMap.get(country);
        String filterKey = "UNKNOWN";
        if (mode.equalsIgnoreCase("opendoar")) {
            filterKey = "openaire____::opendoar";
        } else if (mode.equalsIgnoreCase("re3data")) {
            filterKey = "openaire____::re3data";
        } else if (mode.equalsIgnoreCase("jour_aggr")) {
            filterKey = "infrastruct_::openaire";
        }

        LOGGER.debug("Country code equals : " + countryCode);
        LOGGER.debug("Filter mode equals : " + filterKey);
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/search/country/")
                .path("/{page}/{size}/")
                .queryParam("country", countryCode)
                .queryParam("managed", managed)
                .build().expand(page, size).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        JSONArray jsonArray = (JSONArray) new JSONObject(rs).get("datasourceInfo");

        List<Repository> resultSet = new ArrayList<>();
        while (jsonArray.length() > 0 ) {
            List<Repository> rep = Converter.jsonToRepositoryList(new JSONObject(rs));

            Collection<Repository> repos = this.getRepositoriesByMode(filterKey, rep);
            resultSet.addAll(repos);

            page += 1;
            uriComponents = UriComponentsBuilder
                    .fromHttpUrl(baseAddress + "/ds/search/country/")
                    .path("/{page}/{size}/")
                    .queryParam("country", countryCode)
                    .queryParam("managed", managed)
                    .build().expand(page, size).encode();
            rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
            jsonArray = (JSONArray) new JSONObject(rs).get("datasourceInfo");
        }
        for (Repository r : resultSet)
            this.getRepositoryInfo(r);

        return resultSet;
    }

    private void getRepositoryInfo(Repository r) throws JSONException {
        r.setInterfaces(this.getRepositoryInterface(r.getId()));
        r.setPiwikInfo(piWikApi.getPiwikSiteForRepo(r.getId()));
        r.setCountryName(getCountryName(r.getCountryCode()));
    }


    private Collection<Repository> getRepositoriesByMode(String mode, List<Repository> rs) {

        List<Repository> reps = new ArrayList<>();
        for (Repository r : rs) {
            if (r.getCollectedFrom().equals(mode))
                reps.add(r);
        }
        return reps;
    }

    @Override
    public List<Repository> getRepositoriesOfUser(@PathVariable("userEmail") String userEmail,
                                                  @PathVariable("page") String page,
                                                  @PathVariable("size") String size) throws JSONException {

        LOGGER.debug("Retreiving repositories of user : " + userEmail );
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/search/registeredby/")
                .path("/{page}/{size}/")
                .queryParam("registeredBy", userEmail)
                .build().expand(page, size).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        List<Repository> repos = Converter.jsonToRepositoryList(new JSONObject(rs));
        for (Repository r : repos)
            this.getRepositoryInfo(r);

        return repos;
    }

    @Override
    public Repository getRepositoryById(@PathVariable("id") String id) throws JSONException {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/get/")
                .path("/{id}/")
                .build().expand(id).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        Repository repo = Converter.jsonToRepositoryObject(new JSONObject(rs));
        if (repo != null)
            getRepositoryInfo(repo);
        return repo;
    }


    @Override
    public Aggregations getRepositoryAggregations(@PathVariable("id") String id) throws JSONException {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/get/")
                .path("/{id}/")
                .build().expand(id).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        JSONObject repository = new JSONObject(rs);

        Aggregations aggregations = new Aggregations();

        try {
            aggregations.setAggregationHistory(Converter.getAggregationHistoryFromJson(repository));
            aggregations.setLastCollection(Converter.getLastCollectionFromJson(repository));
            aggregations.setLastTransformation(Converter.getLastTransformationFromJson(repository));
            return aggregations;
        } catch (JSONException e) {
            LOGGER.debug("JSON aggregation exception ", e);
            throw e;
        }

    }

    @Override
    public List<Repository> getRepositoriesByName(@PathVariable("name") String name,
                                                  @PathVariable("page") String page,
                                                  @PathVariable("size") String size) throws JSONException {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/search/name/")
                .path("/{page}/{size}")
                .queryParam("name", name)
                .build().expand(page, size).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        List<Repository> repos = Converter.jsonToRepositoryList(new JSONObject(rs));
        for (Repository r : repos)
            getRepositoryInfo(r);
        return repos;
    }

    @Override
    public List<RepositoryInterface> getRepositoryInterface(@PathVariable("id") String id) throws JSONException {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/api/")
                .path("/{id}/")
                .build().expand(id).encode();

        String rs = restTemplate.getForObject(uriComponents.toUri(), String.class);
        return Converter.jsonToRepositoryInterfaceList(new JSONArray(rs));
    }

    @Override
    public void addRepository(@RequestParam("datatype") String datatype,
                              @RequestBody Repository repository) throws Exception {

        /*JSONObject json_params = new JSONObject(params);
        String datatype = json_params.getString("datatype");
        String json_repository = json_params.getString("repository");
        Repository repository = null;*/
        ObjectMapper mapper = new ObjectMapper();
        mapper.disable(DeserializationFeature.FAIL_ON_UNK‌​NOWN_PROPERTIES);
        try {
            String json_repo = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(repository);
            LOGGER.debug("repository -> " + json_repo);
          //  repository = mapper.readValue(json_repository, Repository.class);
        } catch (Exception e) {
            LOGGER.debug("Error parsing repository ", e);
            throw  e;
        }
        repository = this.setRepositoryFeatures(datatype,repository);

        LOGGER.debug("storing " + datatype + " repository with id: " + repository.getId());
        if (!datatype.equalsIgnoreCase("opendoar") && !datatype.equalsIgnoreCase("re3data")) {
            if (datatype.equalsIgnoreCase("journal") || datatype.equalsIgnoreCase("aggregator")) {
                LOGGER.debug("looking if " + datatype + " " + repository.getOfficialName() + " is already in datasources");
                if (getRepositoryById(repository.getId()) != null) {
                    String retMessage = datatype + " '" + repository.getOfficialName() + "' is already in datasources.";
                    repository.getInterfaces().clear();
                    LOGGER.debug(retMessage);
                } else {
                    LOGGER.debug(datatype + " " + repository.getOfficialName() + " is not in datasources. Inserting..");
                    this.storeRepository(repository);
                }
            }
        } else {
            this.updateRepository(repository);
        }

        LOGGER.debug("Inserting Interfaces");
        Iterator var11 = repository.getInterfaces().iterator();

        while (var11.hasNext()) {
            RepositoryInterface iFace = (RepositoryInterface) var11.next();
            if (!iFace.getBaseUrl().isEmpty() && !iFace.getDesiredCompatibilityLevel().isEmpty()) {
                if (iFace.getId() != null && !iFace.getId().isEmpty()) {
                    LOGGER.debug("updating iface..");
                    this.updateInterface(datatype,iFace);
                    LOGGER.debug("updated successfully");
                } else {
                    LOGGER.debug("adding new iface..");
                    this.registerRepositoryInterface(repository.getId(),iFace,datatype);
                }
            }
        }
    }

    private void updateRepository(Repository repository) {
        this.updateRegisteredByValue(repository.getId(),repository.getRegisteredBy());
        this.updateEnglishName(repository.getId(),repository.getEnglishName());
        this.updateLogoUrl(repository.getId(),repository.getLogoUrl());
        this.updateTimezone(repository.getId(), String.valueOf(repository.getTimezone()));
        //TODO update datasource type
    }

    private void updateRegisteredByValue(String id, String registeredBy) {

        LOGGER.debug("Updating registered by value with  : " + registeredBy );
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/registeredby/")
                .queryParam("dsId",id)
                .queryParam("registeredBy", registeredBy)
                .build()
                .encode();

        restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    private Repository setRepositoryFeatures(String datatype, Repository repository) {

        //TODO update map
        repository.setCountryCode(countriesMap.get(repository.getCountryName()));

        repository.setActivationId(UUID.randomUUID().toString());
        // repo.setRegisteredBy((String) session.get(LocalVocabularies.loggedInField));

        if (datatype.equals("opendoar") || datatype.equals("re3data")) {
            repository.setProvenanceActionClass("sysimport:crosswalk:entityregistry");
        } else if (datatype.equals("journal")) {
            repository.setProvenanceActionClass("user:insert");
            repository.setCollectedFrom("infrastruct_::openaire");
            if (repository.getIssn() != null && repository.getIssn().length() == 0)
                repository.setIssn(com.unboundid.util.Base64.encode(repository.getOfficialName()).substring(0, 8));
            repository.setId("openaire____::issn" + repository.getIssn());
            repository.setNamespacePrefix("issn" + repository.getIssn());
        } else if (datatype.equals("aggregator")) {
            repository.setProvenanceActionClass("user:insert");
            repository.setCollectedFrom("infrastruct_::openaire");
            repository.setId("openaire____::" + com.unboundid.util.Base64.encode(repository.getOfficialName()));
            repository.setNamespacePrefix(Normalizer.normalize(repository.getOfficialName().toLowerCase().replace(" ", "_"), Normalizer.Form.NFD).replaceAll("[^a-zA-Z0-9]", ""));
            if (repository.getNamespacePrefix().length() > 12) {
                repository.setNamespacePrefix(repository.getNamespacePrefix().substring(0, 12));
            } else {
                while (repository.getNamespacePrefix().length() < 12)
                    repository.setNamespacePrefix(repository.getNamespacePrefix().concat("_"));
            }
        }
        return repository;
    }

    private void updateInterface(String datatype,RepositoryInterface iFace) {
        //TODO call update base url
        //((DatasourceManagerService) this.dmService.getService()).updateBaseUrl(repo.getId(), iFace.getId(), iFace.getBaseUrl());
        if (!iFace.getAccessSet().isEmpty()) {
            LOGGER.debug("set not empty: " + iFace.getAccessSet());
            //TODO call update method for access params
            // ((DatasourceManagerService) this.dmService.getService()).updateAccessParam(repo.getId(), iFace.getId(), "set", iFace.getAccessSet(), false);
        } else {
            //TODO call deleteAccessParamOrExtraField
            //((DatasourceManagerService) this.dmService.getService()).deleteAccessParamOrExtraField(repo.getId(), iFace.getId(), "set");
        }
        //TODO update content description
        //((DatasourceManagerService) this.dmService.getService()).updateContentDescription(repo.getId(), iFace.getId(), "metadata");
        if (datatype.equals("re3data")) {
            //TODO call update access params
            //  ((DatasourceManagerService) this.dmService.getService()).updateAccessParam(repo.getId(), iFace.getId(), "format", "oai_datacite", false);
            iFace.setAccessFormat("oai_datacite");
        } else {
            //TODO call update access params
            //((DatasourceManagerService) this.dmService.getService()).updateAccessParam(repo.getId(), iFace.getId(), "format", "oai_dc", false);
            iFace.setAccessFormat("oai_dc");
        }

    }

    private RepositoryInterface createRepositoryInterface(Repository repo, RepositoryInterface iFace, String datatype) {

        iFace.setContentDescription("metadata");
        iFace.setCompliance("UNKNOWN");
        if (datatype.equals("re3data")) {
            iFace.setAccessFormat("oai_datacite");
        } else {
            iFace.setAccessFormat("oai_dc");
        }

        if (repo.getDatasourceClass() != null && !repo.getDatasourceClass().isEmpty()) {
            iFace.setTypology(repo.getDatasourceClass());
        } else if (datatype.equalsIgnoreCase("journal")) {
            iFace.setTypology("pubsrepository::journal");
        } else if (datatype.equalsIgnoreCase("aggregator")) {
            iFace.setTypology("aggregator::pubsrepository::unknown");
        } else if (datatype.equalsIgnoreCase("opendoar")) {
            iFace.setTypology("pubsrepository::unknown");
        } else if (datatype.equalsIgnoreCase("re3data")) {
            iFace.setTypology("datarepository::unknown");
        }

        iFace.setRemovable(true);
        iFace.setAccessProtocol("oai");
        iFace.setMetadataIdentifierPath("//*[local-name()='header']/*[local-name()='identifier']");
       // iFace.setId("api_________::" + repo.getId() + "::" + UUID.randomUUID().toString().substring(0, 8));
        if (iFace.getAccessSet().isEmpty()) {
            LOGGER.debug("set is empty: " + iFace.getAccessSet());
            iFace.removeAccessSet();
        }
        return iFace;
    }

    private void storeRepository(Repository repository) throws JSONException {

        Date utilDate = new Date();
        Timestamp date = new Timestamp(utilDate.getTime());
        repository.setDateOfCollection(date);
        repository.setAggregator("OPENAIRE");

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/add/")
                .build()
                .encode();
        String json_repository = Converter.repositoryObjectToJson(repository);
        HttpEntity<String> httpEntity = new HttpEntity <String> (json_repository,httpHeaders);
        restTemplate.postForObject(uriComponents.toUri(),httpEntity,String.class);
    }

    @Override
    public void deleteRepositoryInterface(@PathVariable("id") String id){
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/api/")
                .path("/{id}/")
                .build().expand(id).encode();
        restTemplate.delete(uriComponents.toUri());
    }

    @Override
    public RepositoryInterface addRepositoryInterface(@RequestParam("datatype") String datatype,
                                                      @RequestParam("repoId") String repoId,
                                                      @RequestBody RepositoryInterface repositoryInterface) throws JSONException {

     /*   JSONObject json_params = new JSONObject(params);
       // LOGGER.debug(params);
        //TODO iFace parameter from gui. Object to json
        String datatype = json_params.getString("datatype");
        String repoId = json_params.getString("repoId");
        String json_iFace = json_params.getString("iFace");
        RepositoryInterface iFace = null;

        ObjectMapper mapper = new ObjectMapper();
        try {
            iFace = mapper.readValue(json_iFace, RepositoryInterface.class);
        } catch (Exception e1) {
            LOGGER.debug("Error parsing repository interface ", e1);
        }*/
        return registerRepositoryInterface(repoId,repositoryInterface,datatype);
    }

    private RepositoryInterface registerRepositoryInterface(String repoId, RepositoryInterface iFace, String datatype) {
        Repository e = null;
        try {
            e = this.getRepositoryById(repoId);
            iFace = createRepositoryInterface(e,iFace,datatype);
            ObjectMapper mapper = new ObjectMapper();

            String json_interface = Converter.repositoryInterfaceObjectToJson(e,iFace);
            LOGGER.debug("iFace equals -> "  + json_interface);
            UriComponents uriComponents = UriComponentsBuilder
                    .fromHttpUrl(baseAddress + "/ds/api/add/")
                    .build()
                    .encode();


            HttpEntity<String> httpEntity = new HttpEntity <String> (json_interface,httpHeaders);
            restTemplate.postForObject(uriComponents.toUri(),httpEntity,String.class);
            return iFace;

        } catch (JSONException e1) {
            LOGGER.debug("Error parsing json ",e1);
        }
        return null;
    }

    @Override
    public List<String> getDnetCountries() {
        LOGGER.debug("Getting dnet-countries!");
        return Converter.readFile("countries.txt");
    }

    @Override
    public List<String> getTypologies() {
        return Converter.readFile("typologies.txt");
    }

    @Override
    public List<Timezone> getTimezones() {
        List<String> timezones =  Converter.readFile("timezones.txt");
        return Converter.toTimezones(timezones);
    }

    @Override
    public String updateManagedStatus(@RequestParam(value = "id")   String id,
                                      @RequestParam(value = "managed")  String managed) {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/manage/")
                .queryParam("id",id)
                .queryParam("managed",managed)
                .build().encode();

        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateEnglishName(@RequestParam(value = "id")   String id,
                                    @RequestParam(value = "englishname")  String englishName) {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/englishname/")
                .queryParam("dsId",id)
                .queryParam("englishname",englishName)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);


    }

    @Override
    public String updateLatitude(@RequestParam(value = "id")   String id,
                                 @RequestParam(value = "latitude")  String latitude) {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/latitude/")
                .queryParam("dsId",id)
                .queryParam("latitude",latitude)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateLongitude(@RequestParam(value = "id")   String id,
                                  @RequestParam(value = "longitude")  String longitude) {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/longitude/")
                .queryParam("dsId",id)
                .queryParam("longitude",longitude)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateOfficialName(@RequestParam(value = "id")   String id,
                                     @RequestParam(value = "officialname")  String officialname) {

        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/officialname/")
                .queryParam("dsId",id)
                .queryParam("officialname",officialname)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateTimezone(@RequestParam(value = "id")   String id,
                                 @RequestParam(value = "timezone")  String timezone) {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/timezone")
                .queryParam("dsId",id)
                .queryParam("timezone",timezone)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateTypology(@RequestParam(value = "id")   String id,
                                 @RequestParam(value = "typology")  String typology) {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/typology")
                .queryParam("dsId",id)
                .queryParam("typology",typology)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updateLogoUrl(@RequestParam(value = "id")   String id,
                                @RequestParam(value = "logoUrl") String logoUrl) {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/logourl")
                .queryParam("dsId",id)
                .queryParam("logourl",logoUrl)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public String updatePlatform(String id, String platform) {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/ds/platform")
                .queryParam("dsId",id)
                .queryParam("platform",platform)
                .build().encode();
        return restTemplate.postForObject(uriComponents.toUri(), null,String.class);
    }

    @Override
    public List<String> getUrlsOfUserRepos(@PathVariable("user_email") String user_email,
                                           @PathVariable("page") String page,
                                           @PathVariable("size") String size) throws JSONException {
        UriComponents uriComponents = UriComponentsBuilder
                .fromHttpUrl(baseAddress + "/api/baseurl/")
                .path("/{page}/{size}")
                .queryParam("userEmail",user_email)
                .build().expand(page,size).encode();
        return Arrays.asList(restTemplate.getForObject(uriComponents.toUri(), String[].class));
    }

    @Override
    public List<String> getDatasourceVocabularies(@PathVariable("mode") String mode) {

        List<String> resultSet = new ArrayList<>();
        for (Map.Entry<String, String> entry : this.getVocabulary("dnet:datasource_typologies").getAsMap().entrySet()) {
            if (mode.equalsIgnoreCase("aggregator")) {
                if (entry.getKey().contains("aggregator"))
                    resultSet.add(entry.getValue());
            } else if (mode.equalsIgnoreCase("journal")) {
                if (entry.getKey().contains("journal"))
                    resultSet.add(entry.getValue());
            } else if (mode.equalsIgnoreCase("opendoar")) {
                if (entry.getKey().contains("pubsrepository"))
                    resultSet.add(entry.getValue());
            } else if (mode.equalsIgnoreCase("re3data")) {
                if (entry.getKey().contains("datarepository"))
                    resultSet.add(entry.getValue());
            }
        }


        return resultSet;
    }

    private Vocabulary getVocabulary(String vocName) {

        if (!vocabularyMap.containsKey(vocName)) {
            vocabularyMap.put(vocName, vocabularyLoader.getVocabulary(vocName, Locale.ENGLISH, Locale.ROOT));
        }
        return vocabularyMap.get(vocName);
    }


    @Override
    public Map<String, String> getCompatibilityClasses(@PathVariable("mode") String mode)  {

        LOGGER.debug("Getting compatibility classes for mode: " + mode);
        Map<String, String> retMap = new HashMap<String, String>();

        Map<String, String> compatibilityClasses = this.getVocabulary("dnet:compatibilityLevel").getAsMap();
        boolean foundData = false;
        for (Map.Entry<String, String> entry : compatibilityClasses.entrySet()) {
            if (mode.equalsIgnoreCase(Constants.REPOSITORY_MODE_ALL))
                return compatibilityClasses;
            else if (mode.equalsIgnoreCase(Constants.REPOSITORY_MODE_RE3DATA)) {
                if (entry.getKey().matches("^openaire[1-9].0_data$")) {
                    retMap.put(entry.getKey(), entry.getValue());
                    foundData = true;
                }
            } else {
                if (entry.getKey().matches("^openaire[1-9].0$") || entry.getKey().equals("driver"))
                    retMap.put(entry.getKey(), entry.getValue());
            }
        }

        //TODO TO BE REMOVED WHEN VOCABULARIES ARE UPDATED
        if (mode.equalsIgnoreCase(Constants.REPOSITORY_MODE_RE3DATA) && !foundData)
            retMap.put("openaire2.0_data", "OpenAIRE Data (funded, referenced datasets)");

        return retMap;
    }

    @Override
    public Map<String, String> getDatasourceClasses(@PathVariable("mode") String mode)  {

        LOGGER.debug("Getting datasource classes for mode: " + mode);

        Map<String, String> retMap = new HashMap<String, String>();

        for (Map.Entry<String, String> entry : this.getVocabulary("dnet:datasource_typologies").getAsMap().entrySet()) {
            if (mode.equalsIgnoreCase("aggregator")) {
                if (entry.getKey().contains("aggregator"))
                    retMap.put(entry.getKey(), entry.getValue());
            } else if (mode.equalsIgnoreCase("journal")) {
                if (entry.getKey().contains("journal"))
                    retMap.put(entry.getKey(), entry.getValue());
            } else if (mode.equalsIgnoreCase("opendoar")) {
                if (entry.getKey().contains("pubsrepository"))
                    retMap.put(entry.getKey(), entry.getValue());
            } else if (mode.equalsIgnoreCase("re3data")) {
                if (entry.getKey().contains("datarepository"))
                    retMap.put(entry.getKey(), entry.getValue());
            }
        }
        return retMap;

    }

    @Override
    public String getCountryName(String countryCode) {
        return inverseCountriesMap.get(countryCode);
    }

    @Override
    public MetricsInfo getMetricsInfoForRepository(@PathVariable("repoId")  String repoId) throws RepositoryServiceException {
        try {

            MetricsInfo metricsInfo = new MetricsInfo();
            metricsInfo.setDiagramsBaseURL(this.usageStatisticsDiagramsBaseURL);
            metricsInfo.setMetricsNumbers(getMetricsNumbers(getOpenAIREId(repoId)));
            return metricsInfo;

        } catch (Exception e) {
            LOGGER.error("Error while getting metrics info for repository: ", e);
            //emailUtils.reportException(e);
            throw new RepositoryServiceException("General error", RepositoryServiceException.ErrorCode.GENERAL_ERROR);
        }
    }

    private MetricsNumbers getMetricsNumbers(String openAIREID) throws BrokerException {

        //build the uri params
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(this.usageStatisticsNumbersBaseURL + openAIREID + "/clicks");

        //create new template engine
        RestTemplate template = new RestTemplate();
        template.getMessageConverters().add(new MappingJackson2HttpMessageConverter());
        ResponseEntity<MetricsNumbers> resp;
        try {
            //communicate with endpoint
            resp = template.exchange(
                    builder.build().encode().toUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<MetricsNumbers>() {
                    });
        } catch (RestClientException e) {
            throw e;
        }

        return resp.getBody();
    }

    private String getOpenAIREId(String repoId) {

        if (repoId != null && repoId.contains("::")) {
            return repoId.split("::")[0] + "::" + DigestUtils.md5Hex(repoId.split("::")[1]);
        }

        return null;
    }


}
