package eu.dnetlib.repo.manager.service.controllers;

import eu.dnetlib.api.functionality.ValidatorServiceException;
import eu.dnetlib.domain.functionality.validator.StoredJob;
import eu.dnetlib.repo.manager.service.utils.OaiTools;
import eu.dnetlib.repo.manager.shared.InterfaceInformation;
import eu.dnetlib.repo.manager.shared.ValidationServiceException;
import gr.uoa.di.driver.util.ServiceLocator;
import eu.dnetlib.domain.functionality.validator.JobForValidation;
import eu.dnetlib.domain.functionality.validator.RuleSet;
import eu.dnetlib.repo.manager.shared.Constants;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import eu.dnetlib.api.functionality.ValidatorService;
import org.apache.log4j.Logger;
import org.json.JSONException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.PathVariable;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;


@Component
public class ValidatorApiImpl implements ValidatorApi{

    @Autowired
    private MonitorApiImpl monitorApi;

    @Resource(name = "validatorServiceLocator")
    private ServiceLocator<ValidatorService> validatorServiceLocator;

    private ValidatorService getValidationService() {
        return this.validatorServiceLocator.getService();
    }

    public ServiceLocator<ValidatorService> getValidatorServiceLocator() {
        return validatorServiceLocator;
    }

    public void setValidatorServiceLocator(ServiceLocator<ValidatorService> validatorServiceLocator) {
        this.validatorServiceLocator = validatorServiceLocator;
    }

    private Map<String, List<RuleSet>> rulesetMap = new ConcurrentHashMap<String, List<RuleSet>>();

    private static final Logger LOGGER = Logger
            .getLogger(ValidatorApiImpl.class);

    @PostConstruct
    private void loadRules(){
        LOGGER.debug("PostConstruct method! Load rules!");
        try {
            for (RuleSet ruleSet : getValidationService().getRuleSets()) {
                if (ruleSet.getVisibility() != null && ruleSet.getVisibility().contains("development")) {
                    String key = "";
                    if (ruleSet.getGuidelinesAcronym().matches("^openaire[1-9].0_data$"))
                        key = Constants.VALIDATION_MODE_DATA;
                    else if (ruleSet.getGuidelinesAcronym().matches("^openaire[1-9].0$") || ruleSet.getGuidelinesAcronym().equals("driver"))
                        key = Constants.VALIDATION_MODE_LITERATURE;
                    else if (ruleSet.getGuidelinesAcronym().matches("^openaire[1-9].0_cris$"))
                        key = Constants.VALIDATION_MODE_CRIS;

                    if (rulesetMap.containsKey(key))
                        rulesetMap.get(key).add(ruleSet);
                    else {
                        List<RuleSet> ruleSets = new ArrayList<RuleSet>();
                        ruleSets.add(ruleSet);
                        rulesetMap.put(key, ruleSets);
                    }
                }
            }
        } catch (ValidatorServiceException e) {
            e.printStackTrace();
        }

    }

    @Override
    public void submitJobForValidation(JobForValidation jobForValidation) {
        LOGGER.debug("Submit job for validation with id : " + jobForValidation.getDatasourceId());
        try {
            this.getValidationService().submitValidationJob(jobForValidation);
        } catch (ValidatorServiceException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void reSubmitJobForValidation(String jobId) throws JSONException {
        LOGGER.debug("Resubmit validation job with id : " + jobId);
        StoredJob job = monitorApi.getJobSummary(jobId,"all");
        Set<Integer> contentRules = new HashSet<Integer>();
        Set<Integer> usageRules = new HashSet<Integer>();

        RuleSet ruleSet = null;
        for (List<RuleSet> ruleSets : this.rulesetMap.values()) {
            for (RuleSet rSet : ruleSets)
                if (rSet.getGuidelinesAcronym().equals(job.getDesiredCompatibilityLevel())) {
                    ruleSet = rSet;
                    break;
                }
        }

        for (int ruleId : job.getRules()) {
            if (ruleSet.getContentRulesIds().contains(ruleId))
                contentRules.add(ruleId);
            else if (ruleSet.getUsageRulesIds().contains(ruleId))
                usageRules.add(ruleId);
        }
        if (!contentRules.isEmpty())
            job.setSelectedContentRules(contentRules);
        if (!usageRules.isEmpty())
            job.setSelectedUsageRules(usageRules);
        this.submitJobForValidation(job);
    }

    @Override
    public List<RuleSet> getRuleSets(@PathVariable("mode") String mode) {
        LOGGER.info("Getting rulesets for mode: " + mode);
        return rulesetMap.get(mode);
    }

    @Override
    public List<String> getSetsOfRepository(@PathVariable("url") String url) {
        LOGGER.debug("Getting sets of repository with url : " + url);
        try {
            return OaiTools.getSetsOfRepo(url);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean identifyRepo(@PathVariable("url") String url) {
        LOGGER.debug("Identify repository with url : " + url);
        try {
            return OaiTools.identifyRepository(url);
        } catch (Exception e) {
            LOGGER.error("Error while identifying repository with url: " + url, e);
            return false;
        }
    }

    @Override
    public RuleSet getRuleSet(@PathVariable("acronym") String acronym) {
        LOGGER.debug("Getting ruleset with acronym : " + acronym);
        RuleSet ruleSet = null;
        try {
            for (List<RuleSet> ruleSets : this.rulesetMap.values()) {
                for (RuleSet rSet : ruleSets)
                    if (rSet.getGuidelinesAcronym().equals(acronym)) {
                        ruleSet = rSet;
                        break;
                    }
            }
            return ruleSet;
        } catch (Exception e) {
            LOGGER.error("Error getting ruleset", e);
            return null;
        }
    }

    @Override
    public List<StoredJob> getStoredJobsNew(String user, String jobType, Integer offset, Integer limit, String dateFrom,
                                            String dateTo, String validationStatus) throws ValidatorServiceException {
        return getValidationService().getStoredJobsNew(user, jobType, offset, limit, dateFrom, dateTo, validationStatus);
    }

    @Override
    public int getStoredJobsTotalNumberNew(String user, String jobType, String validationStatus) throws  ValidatorServiceException {
        return getValidationService().getStoredJobsTotalNumberNew(user, jobType, validationStatus);
    }

    @Override
    public InterfaceInformation getInterfaceInformation(@PathVariable("baseUrl") String baseUrl) throws ValidationServiceException {
        try {
            LOGGER.debug("Getting interface information with url: " + baseUrl);
            InterfaceInformation interfaceInformation = new InterfaceInformation();
            interfaceInformation.setIdentified(this.identifyRepo(baseUrl));
            if (interfaceInformation.isIdentified())
                interfaceInformation.setSets(this.getSetsOfRepository(baseUrl));

            return interfaceInformation;
        } catch (Exception e) {
            LOGGER.error("Error getting interface information with url: " + baseUrl, e);
//            emailUtils.reportException(e);
            throw new ValidationServiceException("login.generalError", ValidationServiceException.ErrorCode.GENERAL_ERROR);
        }
    }


}
