package eu.dnetlib.data.search.transform.config;


import eu.dnetlib.data.search.transform.*;
import eu.dnetlib.data.search.transform.formatter.Formatter;
import eu.dnetlib.data.search.transform.formatter.XSLTFormatter;
import eu.dnetlib.data.search.transform.utils.VelocityUtil;
import eu.dnetlib.data.search.utils.vocabulary.VocabularyManager;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

import java.io.InputStream;
import java.io.StringWriter;
import java.util.*;

public class SearchRegistry implements TransformerRegistry, FormatterRegistry {
	private Logger logger = Logger.getLogger(this.getClass());

	private Configuration config = null;
	private VocabularyManager vocabularyManager = null;

	private Map<Locale,HashMap<String,Transformer>> transformers = new HashMap<Locale, HashMap<String,Transformer>>();
	private Map<String, Formatter> formatters = new HashMap<String, Formatter>();

	@Override
	public Transformer getTransformer(String name, Locale locale)  {
		logger.debug("Requested transformer " + name + " locale " + locale);
		Transformer tx = null;

		if(name == null) {
			return null;
		}

		if (locale == null) {
			locale = config.getDefaultLocale();
		}

		try {
			if (this.transformers.get(locale)!= null) {
				tx = this.transformers.get(locale).get(name);

			} else { //not map for locale
				logger.debug("Transformation map " + config.getTransformationsMap().keySet());
				tx = createChainTransformer(name, locale,  config.getTransformationsMap().get(name));
				Map<String, Transformer> map = new HashMap<String, Transformer>();
				map.put(name, tx);
				transformers.put(locale, (HashMap<String, Transformer>) map);
			}

			if (tx == null) { // there is the local map but not the transformer
				tx = createChainTransformer(name, locale,  config.getTransformationsMap().get(name));
				transformers.get(locale).put(name, tx);
			}

		} catch (Exception e) {
			logger.error("Cannot create transformer " + name, e);
			throw new IllegalArgumentException("Cannot create transformer " + name, e);
		}

		logger.debug("Returned in get transformer" + tx);
		return tx;
	}

	@Override
	public Formatter getFormatter(String name) {
		Formatter fm = this.formatters.get(name);

		try {
			if (fm == null) {
				FormatterConfiguration formatterConfiguration = config.getFormattersMap().get(name);
				String formatterXslt = formatterConfiguration.getXsl_file();
				String template = formatterConfiguration.getTemplate();

				logger.debug("Configuration for formatter: " + formatterXslt + ", " + template);

				if (formatterXslt == null) {
					logger.error("Cannot create formatter " + name);
					throw new IllegalArgumentException("Cannot create formatter " + name);
				}

				if (template == null) {
					fm = createFormatter(formatterXslt);
				} else {
					fm = createFormatter(formatterXslt, template);
				}

				formatters.put(name, fm);
			}

		} catch (Exception e) {
			logger.error("Cannot create formatter " + name);
			throw new IllegalArgumentException("Cannot create formatter " + name, e);
		}

		return fm;
	}

	private Transformer createChainTransformer(String name, Locale locale, List<Transformation> transformations) throws SearchFactoryException {
		List<XPathTrasformation> temp = new ArrayList<XPathTrasformation>();
		List<Transformer> forChaining = new ArrayList<Transformer>();
		for (Transformation tr: transformations){
			if (tr instanceof XSLTTransformation) {
				try {
					if (!temp.isEmpty()){
						forChaining.add(createTransformer(locale, temp));
					}
					temp.clear();
					forChaining.add(createTransformer((XSLTTransformation)tr));
				} catch (Exception e) {
					logger.warn("Error creating a transformer with name: "+name+" and locale: "+locale.getDisplayName(), e);
					throw new SearchFactoryException("Error creating a transformer with name: "+name+" and locale: "+locale.getDisplayName(), e);

				}
			}
			else {
				temp.add((XPathTrasformation)tr);
			}
		}
		if (!temp.isEmpty()){
			try {
				forChaining.add(createTransformer(locale, temp));
			} catch (Exception e) {
				logger.warn("Error creating a transformer with name: " + name + " and locale: "+locale.getDisplayName(), e);
				throw new SearchFactoryException("Error creating a transformer with name: " + name + " and locale: "+locale.getDisplayName(), e);
			}
		}
		return new ChainTransformer(name, forChaining);
	}

	private Transformer createTransformer(XSLTTransformation transformation) throws Exception {
		String xsl_file = transformation.getXslt();
		InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(xsl_file);
		StringWriter writer = new StringWriter();
		IOUtils.copy(inputStream, writer);
		String xsl = writer.toString();

		XsltTransformer OptimusPrime = new XsltTransformer(xsl);

		writer.close();
		IOUtils.closeQuietly(inputStream);

		return OptimusPrime;
	}

	private Transformer createTransformer(Locale locale, List<XPathTrasformation> transformations) throws Exception {
		VelocityEngine ve = VelocityUtil.getEngine();

		Template t = ve.getTemplate("/eu/dnetlib/data/search/transform/config/velocity.vm", "UTF-8");
		VelocityContext context = new VelocityContext();

		context.put("locale", locale);
		context.put("defLocale", config.getDefaultLocale());
		context.put("transformations", transformations);
		StringWriter writer = new StringWriter();
		t.merge(context, writer);
		XsltTransformer Megatron = new XsltTransformer(writer.toString());

		//logger.debug("Transformer xsl " + Megatron.getXslt());
		//logger.debug("Vocab paths are: " + loadVocabulariesPathMap(transformations, locale).toString());

		return Megatron;
	}

	private Formatter createFormatter(String xsl_file) throws Exception {
		logger.debug("Loading xslt " + xsl_file);
		InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(xsl_file);

		StringWriter writer = new StringWriter();
		IOUtils.copy(inputStream, writer);
		String xsl = writer.toString();

		XSLTFormatter Cybertron = new XSLTFormatter(xsl);
		writer.close();
		IOUtils.closeQuietly(inputStream);

		return Cybertron;
	}

	private Formatter createFormatter(String xsl_file, String template) throws Exception {
		logger.debug("Loading xslt " + xsl_file);
		logger.debug("Loading template " + template);

		InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(xsl_file);

		StringWriter writer = new StringWriter();
		IOUtils.copy(inputStream, writer);
		String xsl = writer.toString();

		XSLTFormatter Cybertron = new XSLTFormatter(xsl, template);
		writer.close();
		IOUtils.closeQuietly(inputStream);

		return Cybertron;
	}

	public Configuration getConfig() {
		return config;
	}

	public void setConfig(Configuration config) {
		this.config = config;
	}

	public void setVocabularyManager(VocabularyManager vocabularyManager) {
		this.vocabularyManager = vocabularyManager;
	}

	public VocabularyManager getVocabularyManager() {
		return vocabularyManager;
	}

}