package eu.dnetlib.data.search.web.api;

import eu.dnetlib.data.search.web.utils.RequestResponseHandler;
import eu.dnetlib.domain.data.SearchResult;
import org.apache.commons.lang3.StringEscapeUtils;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;
import java.util.Iterator;
import java.util.List;

/**
 * Created by kiatrop on 21/11/2016.
 */
public class APIResponseFormatter {

    //private static final Logger logger = Logger.getLogger(APIResponseFormatter.class);

    public static String compose404Message(String type, String message) {
        if(type.equals(MediaType.APPLICATION_JSON)) {
           return String.format(JSONResponseFormat.error404Message, StringEscapeUtils.escapeJson(message));
        }
        return String.format(XMLResponseFormat.error404Message, StringEscapeUtils.escapeXml10(message));
    }

    public static String compose400Message(String type, String message) {
         if(type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat.error400Message, StringEscapeUtils.escapeJson(message));
        }
        return String.format(XMLResponseFormat.error400Message, StringEscapeUtils.escapeXml10(message));
    }

    public static String compose500Message(String type, String message, String description) {
        if(type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat.error500Message, StringEscapeUtils.escapeJson(message), StringEscapeUtils.escapeJson(description));
        }
        return String.format(XMLResponseFormat.error500Message, StringEscapeUtils.escapeXml10(message), StringEscapeUtils.escapeXml10(description));
    }

    public static String createCountMeta(HttpServletRequest request, String query, int total, String type) {
        String selfPageUrl = pageUrlCreation(request);

        if  (type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat.countSearch, StringEscapeUtils.escapeJson(query), total, formatSelfPaging(type, selfPageUrl));
        }

        return String.format(XMLResponseFormat.countSearch, StringEscapeUtils.escapeXml10(query), total, formatSelfPaging(type, selfPageUrl));
    }

    public static String createEntityResponse(HttpServletRequest request, RequestResponseHandler.Entity entityType, String entity, String type) {
        if (type.equals(MediaType.APPLICATION_JSON)) {
            String cleanEntity = null;
            if (entity != null && !entity.isEmpty()) {
                cleanEntity = entity.substring(1, entity.length() - 1);
            } else {
                cleanEntity = "\"result\":{}";
            }
            return String.format(JSONResponseFormat.singleValueResponse, createMetaBasic(request, type), cleanEntity);
        }
        return String.format(XMLResponseFormat.singleValueResponse, createMetaBasic(request, type), entity);
    }

    public static String createEntitiesResponse(HttpServletRequest request, RequestResponseHandler.Entity entityType, String query, List<String> filters, SearchResult searchResult, boolean refine, String type, boolean special, boolean hasTitle) {

        if  (type.equals(MediaType.APPLICATION_JSON)) {
                return String.format(JSONResponseFormat.response, createMeta(request, type, query, filters, searchResult.getTotal(), searchResult.getPage(), searchResult.getSize()), formatSearchResults(type, searchResult.getSearchResults()), formatRefineResults(type, searchResult.getBrowseResults()));
        }

        if(type.equals("text/csv")) {
            if (hasTitle) {
                return String.format(CSVResponseFormat.response, CSVResponseFormat.appendTitle(entityType, special), formatSearchResults("text/csv", searchResult.getSearchResults()));
            }
            return String.format(CSVResponseFormat.response, "", formatSearchResults(type, searchResult.getSearchResults()));

        } else if (type.equals(MediaType.TEXT_HTML)) {
            return String.format(HTMLResponseFormat.response, formatSearchResults(type, searchResult.getSearchResults()));
        }

        //default xml
        return String.format(XMLResponseFormat.response, createMeta(request, type, query, filters, searchResult.getTotal(), searchResult.getPage(), searchResult.getSize()), formatSearchResults(type, searchResult.getSearchResults()), formatRefineResults(type, searchResult.getBrowseResults()));
    }

    private static String formatRefineResults(String type, List<String> refineResults) {
        if (type.equals(MediaType.APPLICATION_JSON)) {
            if (refineResults == null || refineResults.isEmpty()) {
                return ",\"refineResults\":{}";
            }

            return ",\"refineResults\": {" + appendJSONStrings(refineResults) + "}";
        }

        if (refineResults == null || refineResults.isEmpty()) {
            return "<refineResults></refineResults>";
        }

        return "<refineResults>" + appendXMLStrings(refineResults) + "</refineResults>";
    }

    private static String formatSearchResults(String type, List<String> searchResults) {
        if (searchResults == null || searchResults.isEmpty()) {
            return "";
        }

        if (type.equals(MediaType.APPLICATION_JSON)) {
            return appendJSONStrings(searchResults);
        }

        return appendXMLStrings(searchResults);
    }

    private static String appendXMLStrings(List<String> searchResults) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String xml:searchResults) {
            stringBuilder.append(xml);
            //logger.debug("APPEND " + xml);
        }
        return stringBuilder.toString();
    }


    /* Now using xsl TODO: check this functionality
    private static String appendCSVString(List<String> searchResults) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String xml:searchResults) {
            stringBuilder.append(xml.replaceAll("\"", "'"));
            //logger.debug("APPEND " + xml);
        }
        return stringBuilder.toString();
    } */

    private static String appendJSONStrings(List<String> searchResults) {
        StringBuilder stringBuilder = new StringBuilder();
        for (Iterator<String> iterator = searchResults.iterator(); iterator.hasNext();) {
            stringBuilder.append(iterator.next());
            if (iterator.hasNext()) {
               stringBuilder.append(",");
            }
        }
        return stringBuilder.toString();
    }

    public static String createMetaBasic(HttpServletRequest request, String type) {
        String selfPageUrl = pageUrlCreation(request);
        if  (type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat.metaBasic, formatSelfPaging(type, selfPageUrl));
        }
        return String.format(XMLResponseFormat.metaBasic, formatSelfPaging(type, selfPageUrl));
    }

    public static String createMeta(HttpServletRequest request, String type, String query, List<String> filters, int total, int currentOffset, int limit) {
        if  (type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat.metaSearch, StringEscapeUtils.escapeJson(query), createFilters(type, filters), total, currentOffset, limit, createPaging(request, type, total, currentOffset, limit));
        }
        return String.format(XMLResponseFormat.metaSearch, StringEscapeUtils.escapeXml10(query), createFilters(type, filters), total, currentOffset, limit, createPaging(request, type, total, currentOffset, limit));
    }

    public static String createFilters(String type, List<String> fieldQueries) {
        StringBuilder builder = new StringBuilder();
        if (fieldQueries != null) {
            if (type.equals(MediaType.APPLICATION_JSON)) {
                for (Iterator<String> iterator = fieldQueries.iterator(); iterator.hasNext(); ) {
                    builder.append("\"").append(StringEscapeUtils.escapeJson(iterator.next())).append("\"");
                    if (iterator.hasNext()) {
                        builder.append(",");
                    }
                }
            } else {
                for (Iterator<String> iterator = fieldQueries.iterator(); iterator.hasNext(); ) {
                    builder.append("<filter>").append(StringEscapeUtils.escapeXml10(iterator.next())).append("</filter>");
                }
            }
        }

        return builder.toString();
    }

    private static String createPaging(HttpServletRequest request, String type, int total, int currentOffset, int limit) {
        String url = request.getRequestURL().toString();
        String selfPageUrl = null;

        if (limit > 0) {
            String firstPageUrl = null;
            String lastPageUrl = null;
            String previousPageUrl = null;
            String nextPageUrl = null;

            if (type.equals(MediaType.APPLICATION_JSON)) {
                selfPageUrl = pageUrlCreation(url, currentOffset, limit);
                firstPageUrl = pageUrlCreation(url, 0, limit);
                lastPageUrl = pageUrlCreation(url, calculateLastPage(total, limit), limit);
                previousPageUrl = pageUrlCreation(url, calculatePreviousPage(total, currentOffset, limit), limit);
                nextPageUrl = pageUrlCreation(url, calculateNextPage(total, currentOffset, limit), limit);
            } else {
                selfPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, currentOffset, limit));
                firstPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, 0, limit));
                lastPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, calculateLastPage(total, limit), limit));
                previousPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, calculatePreviousPage(total, currentOffset, limit), limit));
                nextPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, calculateNextPage(total, currentOffset, limit), limit));
            }

            return formatPaging(type, firstPageUrl, lastPageUrl, previousPageUrl, nextPageUrl, selfPageUrl);

        }

        if (type.equals(MediaType.APPLICATION_JSON)) {
            selfPageUrl = pageUrlCreation(url, currentOffset, limit);
        } else {
            selfPageUrl = StringEscapeUtils.escapeXml10(pageUrlCreation(url, currentOffset, limit));
        }
        return formatSelfPaging(type, selfPageUrl);
    }

    private static String formatPaging(String type, String first, String last, String previous, String next, String self) {
        if (type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat._links, first, last, previous, next, self);
        }
        return String.format(XMLResponseFormat._links, first, last, previous, next, self);
    }

    private static String formatSelfPaging(String type, String self) {
        if (type.equals(MediaType.APPLICATION_JSON)) {
            return String.format(JSONResponseFormat._selflink, self);
        }

        return String.format(XMLResponseFormat._selflink, self);
    }

    private static int calculateLastPage(int total, int limit) {
        if (total <= limit) {
            return 0;

        } else {
            if (total%limit == 0) {
                return total / limit - 1;
            }
            return total/limit ;
        }
    }

    private static int calculatePreviousPage(int total, int currentOffset, int limit) {
        if (currentOffset-1 <= 0) {
            return 0;
        }

        int lastPage = calculateLastPage(total, limit);
        return (currentOffset-1 < lastPage) ? (currentOffset-1) : lastPage;
    }

    private static int calculateNextPage(int total, int currentOffset, int limit) {
        int lastPage = calculateLastPage(total, limit);
        return (currentOffset + 1 >= lastPage) ? lastPage:(currentOffset+1);
    }

    private static String pageUrlCreation(String urlPrefix, int offset, int limit) {
        return urlPrefix + "?page=" + offset + "&size=" + limit;
    }

    private static String pageUrlCreation(HttpServletRequest request) {
        StringBuffer requestURL = request.getRequestURL();
        String queryString = request.getQueryString();

        if (queryString == null) {
            return requestURL.toString();
        } else {
            return requestURL.append('?').append(queryString).toString();
        }
    }

}
