package eu.dnetlib.data.search.transform.config;

import eu.dnetlib.api.data.SearchService;
import eu.dnetlib.data.search.utils.vocabulary.ISVocabulary;
import eu.dnetlib.data.search.utils.vocabulary.IndexVocabulary;
import eu.dnetlib.data.search.utils.vocabulary.LocalVocabulary;
import eu.dnetlib.data.search.utils.vocabulary.Vocabulary;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import javax.xml.xpath.*;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

/**
 * 
 * @author kiatrop
 *
 */

/**
 * ConfigurationFactory creates a {@link Configuration} for the {@link SearchService}
 * based on an XML configuration file. 
 * 
 * configurationName is the name of the XML configuration file 
 * vocabulariesPath is the local folder that is the working folder for {@link SearchService}
 * configurationXml has the content of the XML configuration file 
 */
public class ConfigurationFactory {

	private Resource configurationName;
	private String vocabulariesPath;
	private String configurationXml;
	
	private static Logger logger = Logger.getLogger(ConfigurationFactory.class);
	
	public void init() throws IOException, URISyntaxException {		
		logger.info("Reading configuration file " + configurationName + " from classpath");
		
		InputStream stream = null;
		
		try {
			stream = configurationName.getInputStream();
			ByteArrayOutputStream baos = new ByteArrayOutputStream();

			IOUtils.copy(stream, baos);
			
			this.configurationXml = baos.toString();
		} finally {
			IOUtils.closeQuietly(stream);
		}
		
		logger.debug("Configuration read.\n" + configurationXml);
	}
	
	public Configuration createConfiguration(String id) throws ConfigurationFactoryException {
		Configuration configuration = new Configuration(id);  
		XPathFactory factory = XPathFactory.newInstance();
		XPath xPath = factory.newXPath();

		try {
			readLocales(xPath, configuration);
			readVocabularies(xPath, configuration);
			readTransformers(xPath, configuration);
			readFormatters(xPath, configuration);
			
		} catch (XPathExpressionException|IOException ex) {
			logger.error("Error parsing configuration file.", ex);
			throw new ConfigurationFactoryException("Error parsing configuration file.", ex);
			
		}
		
		return configuration;
	}
	
	private void readLocales(XPath xPath, Configuration configuration) throws XPathExpressionException {
		XPathExpression  xPathExpression = xPath.compile("/configuration/locales/locale");
		NodeList nodes = (NodeList) xPathExpression.evaluate(
				new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);
					
		for (int i = 0; i < nodes.getLength(); i++) {
			StringTokenizer tokenizer = new StringTokenizer(nodes.item(i).getAttributes().getNamedItem("name").getNodeValue(), "_");
			Locale locale = new Locale(tokenizer.nextToken(), tokenizer.nextToken());
			configuration.getLocales().add(locale);
			
			if (nodes.item(i).getAttributes().getNamedItem("default").getNodeValue().equals("true")) {
				configuration.setDefaultLocale(locale);
			}
		}
	}
	
	private void readVocabularies(XPath xPath, Configuration configuration) throws XPathExpressionException, IOException, ConfigurationFactoryException {		
		String vocabularyName = null;
		Vocabulary vocabulary = null;
		
		XPathExpression  xPathExpression = xPath.compile("/configuration/vocabularies/local_vocabulary");
		NodeList nodes = (NodeList) xPathExpression.evaluate(new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);		
		for (int i = 0; i < nodes.getLength(); i++) {
			vocabularyName = nodes.item(i).getAttributes().getNamedItem("name").getNodeValue();
			vocabulary = new LocalVocabulary(vocabularyName, nodes.item(i).getAttributes().getNamedItem("file").getNodeValue());
			configuration.getLocalVocabularyMap().put(vocabularyName, vocabulary);
		}
		
		xPathExpression = xPath.compile("/configuration/vocabularies/is_vocabulary");
		nodes = (NodeList) xPathExpression.evaluate(new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);
		
		for (int i = 0; i < nodes.getLength(); i++) {
			vocabularyName = nodes.item(i).getAttributes().getNamedItem("name").getNodeValue();
			String isVocabularyName = nodes.item(i).getAttributes().getNamedItem("xml_name").getNodeValue();
			vocabulary = new ISVocabulary(vocabularyName, isVocabularyName);
			configuration.getIsVocabularyMap().put(vocabularyName, vocabulary);
		}
		
		xPathExpression = xPath.compile("/configuration/vocabularies/index_vocabulary");
		nodes = (NodeList) xPathExpression.evaluate(new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);

		for (int i = 0; i < nodes.getLength(); i++) {
			vocabularyName = nodes.item(i).getAttributes().getNamedItem("name").getNodeValue();
			String query = nodes.item(i).getAttributes().getNamedItem("query").getNodeValue();
			String transformer = nodes.item(i).getAttributes().getNamedItem("transformer").getNodeValue();			
			vocabulary = new IndexVocabulary(vocabularyName, query, transformer);
			configuration.getIndexVocabularyMap().put(vocabularyName, vocabulary);
		}
	}

	private void readTransformers(XPath xPath, Configuration configuration) throws XPathExpressionException {
		XPathExpression xPathExpression = xPath.compile("/configuration/transformers/transformer");
		NodeList nodes = (NodeList) xPathExpression.evaluate(
				new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);

		//per transformer
		for (int i = 0; i < nodes.getLength(); i++) {
			NodeList transformationNodes = ((Element) nodes.item(i)).getElementsByTagName("transformation");
			
			List<Transformation> transformations = new ArrayList<Transformation>();
			for (int j=0; j < transformationNodes.getLength(); j++) {
				Element transformation = (Element) transformationNodes.item(j);

				//logger.debug("Tr " + transformation.getAttributes().getNamedItem("xslt"));
				if (transformation.getAttributes().getNamedItem("xslt") != null) {
					transformations.add(new XSLTTransformation(transformation.getAttributes().getNamedItem("xslt").getNodeValue()));
					//logger.debug("Tr2 " + transformation.getAttributes().getNamedItem("xslt").getNodeValue());
				} else {
					String match = transformation.getAttributes().getNamedItem("match").getNodeValue();
					String change = transformation.getAttributes().getNamedItem("change").getNodeValue();
					String vocabulary = transformation.getAttributes().getNamedItem("vocabulary").getNodeValue();
					
					transformations.add(new XPathTrasformation(match, change, vocabulary));
				}			
			}
			
			String transformerName = nodes.item(i).getAttributes().getNamedItem("name").getNodeValue();
			configuration.getTransformationsMap().put(transformerName, transformations);

		}
	}
	
	private void readFormatters(XPath xPath, Configuration configuration) throws XPathExpressionException {
		XPathExpression xPathExpression = xPath.compile("/configuration/formatters/formatter");
		NodeList nodes = (NodeList) xPathExpression.evaluate(
				new InputSource(new StringReader(configurationXml)), XPathConstants.NODESET);
		
		//per formatter
		for (int i = 0; i < nodes.getLength(); i++) {
            FormatterConfiguration formatterConfiguration = new FormatterConfiguration();
            formatterConfiguration.setXsl_file(nodes.item(i).getAttributes().getNamedItem("xslt").getNodeValue());

            if (nodes.item(i).getAttributes().getNamedItem("template") != null) {
                formatterConfiguration.setTemplate(nodes.item(i).getAttributes().getNamedItem("template").getNodeValue());
            }

            configuration.getFormattersMap().put(
                nodes.item(i).getAttributes().getNamedItem("name").getNodeValue(), formatterConfiguration);
			}
	}

	public void setConfigurationName(Resource configurationName) {
		this.configurationName = configurationName;
	}

	public void setVocabulariesPath(String vocabulariesPath) {
		this.vocabulariesPath = vocabulariesPath;
	}

	public Resource getConfigurationName() {
		return configurationName;
	}

	public String getVocabulariesPath() {
		return vocabulariesPath;
	}

	public String getConfigurationXml() {
		return configurationXml;
	}

	public static Logger getLogger() {
		return logger;
	}
}
