package eu.dnetlib.data.search.transform.formatter;

import eu.dnetlib.data.search.transform.FormatterException;
import eu.dnetlib.data.search.transform.utils.VelocityUtil;
import eu.dnetlib.domain.data.SearchResult;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class SimpleFormatter implements Formatter {

	private javax.xml.transform.Transformer xmlTransformer = null;
	private StringWriter xmlResultResource = new StringWriter();

	private Template t = null;
	
	private final Logger logger = Logger.getLogger(SimpleFormatter.class);
	
	public SimpleFormatter() throws FormatterException{
        StringWriter writer = null;
        InputStream inputStream = null;

        try	{
            VelocityEngine ve = VelocityUtil.getEngine();
            String clean_xsl = "/eu/dnetlib/data/search/transform/config/omit-xml-declaration.xsl";

			t = ve.getTemplate("/eu/dnetlib/data/search/transform/config/format.vm", "UTF-8");
            inputStream = this.getClass().getResource(clean_xsl).openStream();
			writer = new StringWriter();

			IOUtils.copy(inputStream, writer);		

            String xsl = writer.toString();
			synchronized(this) {
				xmlTransformer = TransformerFactory.newInstance().newTemplates(new StreamSource(new StringReader(xsl))).newTransformer();
                xmlTransformer.setOutputProperty("encoding", "UTF-8");
			}
		} catch (Exception e) {
			logger.error("Error while creating a velocity template.", e);
			throw new FormatterException("Error while creating a velocity template.", e);
		} finally {
            IOUtils.closeQuietly(writer);
            IOUtils.closeQuietly(inputStream);
        }
	}

    public SimpleFormatter(String template) throws FormatterException {
        StringWriter writer = null;
        InputStream inputStream = null;

        logger.debug("Creating formatter with template " + template);

        try	{
            VelocityEngine ve = VelocityUtil.getEngine();
            String clean_xsl = "/eu/dnetlib/data/search/transform/config/omit-xml-declaration.xsl";
            t = ve.getTemplate("/eu/dnetlib/data/search/transform/config/" + template, "UTF-8");

            inputStream = this.getClass().getResource(clean_xsl).openStream();
            writer = new StringWriter();

            IOUtils.copy(inputStream, writer);

            String xsl = writer.toString();
            synchronized(this) {
                xmlTransformer = TransformerFactory.newInstance().newTemplates(new StreamSource(new StringReader(xsl))).newTransformer();
            }
        } catch (Exception e) {
            logger.error("Error while creating a velocity template.", e);
            throw new FormatterException("Error while creating a velocity template.", e);

        } finally {
            IOUtils.closeQuietly(writer);
            IOUtils.closeQuietly(inputStream);
        }
    }

	@Override
	public String format(SearchResult result) throws FormatterException {
		VelocityContext context = null;
		try {
			context = new VelocityContext();
		} catch (Exception e) {
			logger.error("Error while creating a Velocity context.", e);
			FormatterException fe = new FormatterException("Error while creating a Velocity context.", e);
			throw fe;
		}
		try {
			context.put("page", result.getPage());
			context.put("query", StringEscapeUtils.escapeXml(result.getQuery()));
			context.put("size", result.getSize());
			context.put("total", result.getTotal());
			context.put("locale", result.getLocale());
			context.put("fields", result.getFields());

            SimpleDateFormat sdf = new SimpleDateFormat("EEE, d MMM yy HH:mm:ss z");
            context.put("currentDate", sdf.format(new Date()));

			if(result.getSearchResults() == null) {
				context.put("results", null);
			} else {
				//logger.info("After: " + clearDeclarations(result.getSearchResults()));
                context.put("results", clearDeclarations(result.getSearchResults()));
            }

			if (result.getBrowseResults() == null) {
				context.put("bresults", null);
			} else {
				context.put("bresults", clearDeclarations(result.getBrowseResults()));
			}
			
			StringWriter writer = new StringWriter();
			t.merge(context, writer);

			return writer.toString();
		
		} catch (Exception e) {
			String text = "Error while formatting.";
			if (result == null) {
				text += " And result is null.";
			}
			logger.error(text, e);
			FormatterException fe = new FormatterException("Error while formatting.", e);
			throw fe;
		}
	}
	
	public List<String> clearDeclarations(List<String> results) throws FormatterException{
		List<String> cleared = new ArrayList<String>();
		for(String result: results){
			xmlResultResource.getBuffer().setLength(0);		
			try {
				xmlTransformer.transform(new StreamSource(new StringReader(result)), new StreamResult(xmlResultResource));
				
			} catch (TransformerException te) {
				logger.warn("Error transforming xml: " + result, te);
				throw new FormatterException("Error transforming xml: " + result, te);
			}
			
			cleared.add(xmlResultResource.getBuffer().toString());
		}
		
		return cleared;
	}
	
}
