package eu.dnetlib.data.search.web.api;

import eu.dnetlib.api.data.SearchService;
import eu.dnetlib.api.data.SearchServiceException;
import eu.dnetlib.data.search.app.SearchServiceImpl;
import eu.dnetlib.data.search.transform.Transformer;
import eu.dnetlib.data.search.transform.config.SearchRegistry;
import eu.dnetlib.data.search.utils.cql.CQLQueryBuilder;
import eu.dnetlib.data.search.web.utils.RequestResponseHandler;
import eu.dnetlib.domain.data.SearchResult;
import io.micrometer.core.annotation.Timed;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

/**
 * Created by kiatrop on 2/6/2016.
 */

@Component
@Path("/v2")
public class SearchApiService {

    private static final Logger logger = Logger.getLogger(SearchApiService.class);

    @Autowired
    private SearchService searchService = null;

    @Autowired
    private SearchRegistry searchRegistry = null;

    @GET
    @Path("/api/resources")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/resources"}, longTask = false)
    public Response fetchResources(@QueryParam("query") String query,
                                   @DefaultValue("0") @QueryParam("page") final int offset,
                                   @DefaultValue("10") @QueryParam("size") final int limit,
                                   @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                   @QueryParam("fields") final List<String> refineFields,
                                   @QueryParam("sf") final List<String> specialFacets,
                                   @QueryParam("fq") final List<String> fieldQueries,
                                   @QueryParam("format") final String format,
                                   @QueryParam("type") final String type,
                                   @Context final HttpServletRequest request) {

        return getResponseByEntity(query, extractEntity(type), offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }

    @GET
    @Path("/api/resources2")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/resources2"}, longTask = false)
    public Response fetchResources(@QueryParam("query") final String query,
                                   @QueryParam("pid") final String pid,
                                   @QueryParam("pidtype") final String pidType,
                                   @DefaultValue("0") @QueryParam("page") final int offset,
                                   @DefaultValue("10") @QueryParam("size") final int limit,
                                   @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                   @QueryParam("fields") final List<String> refineFields,
                                   @QueryParam("sf") final List<String> specialFacets,
                                   @QueryParam("fq") final List<String> fieldQueries,
                                   @QueryParam("format") final String format,
                                   @QueryParam("type") final List<String> types,
                                   @Context final HttpServletRequest request) {

        return getResponseByEntityTypes(query, pid, pidType, types, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }


    @GET
    @Path("/api/results")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/results"}, longTask = false)
    public Response fetchResults(@QueryParam("q") final String keywords,
                                 @QueryParam("doi") final String doi,
                                 @QueryParam("sortBy") final String sortBy,
                                 @DefaultValue("0") @QueryParam("page") final int offset,
                                 @DefaultValue("10") @QueryParam("size") final int limit,
                                 @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                 @QueryParam("fields") final List<String> refineFields,
                                 @QueryParam("sf") final List<String> specialFacets,
                                 @QueryParam("fq") final List<String> fieldQueries,
                                 @QueryParam("format") final String format,
                                 @QueryParam("type") final List<String> types,
                                 @Context final HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(types, keywords, doi,sortBy, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.RESULT, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);

    }

    @GET
    @Path("/api/results/{resultsid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/results/{resultsid}"}, longTask = false)
    public Response fetchResultsById(@PathParam("resultsid") String resultsid,
                                          @QueryParam("format") final String format,
                                          @QueryParam("fq") final List<String> fieldQueries,
                                          @Context final HttpServletRequest request) {

        return getResponseByEntityId(request, RequestResponseHandler.Entity.RESULT, resultsid, format, fieldQueries);
    }

    @GET
    @Path("/api/results/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/results/count"}, longTask = false)
    public Response fetchResultsCount(@QueryParam("q") String query,
                                      @QueryParam("format") final String format,
                                      @QueryParam("fq") final List<String> fieldQueries,
                                      @QueryParam("types") final List<String> types,
                                      @Context final HttpServletRequest request)  {
        String fullQuery = buildSearchRequest(types, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/publications")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/publications"}, longTask = false)
    public Response fetchPublications(@QueryParam("q") final String keywords,
                                      @QueryParam("doi") final String doi,
                                      @QueryParam("sortBy") final String sortBy,
                                      @DefaultValue("0") @QueryParam("page") final int offset,
                                      @DefaultValue("10") @QueryParam("size") final int limit,
                                      @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                      @QueryParam("fields") final List<String> refineFields,
                                      @QueryParam("sf") final List<String> specialFacets,
                                      @QueryParam("fq") final List<String> fieldQueries,
                                      @QueryParam("format") final String format,
                                      @Context final HttpServletRequest request) {

        logger.debug("Publications request received");
        //TODO do not allow q and keywords both - SWITCH back if needed
        //String simpleQuery = buildSimpleSearchQuery(RequestResponseHandler.Entity.PUBLICATION, keywords) ;
        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.PUBLICATION, keywords, doi,sortBy, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.PUBLICATION, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }


    @GET
    @Path("/api/publications/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/publications/count"}, longTask = false)
    public Response fetchPublicationsCount(@QueryParam("q") String query,
                                           @QueryParam("format") final String format,
                                           @QueryParam("fq") final List<String> fieldQueries,
                                           @Context final HttpServletRequest request)  {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.PUBLICATION, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/publications/{publicationid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/publications/{publicationid}"}, longTask = false)
    public Response fetchPublicationsById(@PathParam("publicationid") String publicationid,
                                          @QueryParam("format") final String format,
                                          @QueryParam("fq") final List<String> fieldQueries,
                                          @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.PUBLICATION, publicationid, format, fieldQueries);
    }


    @GET
    @Path("/api/datasets")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasets"}, longTask = false)
    public Response fetchDatasets(@QueryParam("q") String keywords,
                                  @QueryParam("doi") final String doi,
                                  @QueryParam("sortBy") final String sortBy,
                                  @DefaultValue("0") @QueryParam("page") int offset,
                                  @DefaultValue("10") @QueryParam("size") int limit,
                                  @QueryParam("refine") @DefaultValue("false") boolean refine,
                                  @QueryParam("fields") final List<String> refineFields,
                                  @QueryParam("sf") final List<String> specialFacets,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @QueryParam("format") final String format,
                                  @Context HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.DATASET, keywords, doi, sortBy, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.DATASET, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }

    @GET
    @Path("/api/datasets/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasets/count"}, longTask = false)
    public Response fetchDatesetsCount(@QueryParam("q") String query,
                                       @QueryParam("format") final String format,
                                       @QueryParam("fq") final List<String> fieldQueries,
                                       @Context final HttpServletRequest request)  {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.DATASET, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/datasets/{datasetid}")
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasets/{datasetid}"}, longTask = false)
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    public Response fetchDatasets(@PathParam("datasetid") String datasetid,
                                  @QueryParam("format") final String format,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.DATASET, datasetid, format, fieldQueries);
    }

    @GET
    @Path("/api/software")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/software"}, longTask = false)
    public Response fetchSoftware(@QueryParam("ru") String keywords,
                                  @QueryParam("doi") final String doi,
                                  @QueryParam("sortBy") final String sortBy,
                                  @DefaultValue("0") @QueryParam("page") int offset,
                                  @DefaultValue("10") @QueryParam("size") int limit,
                                  @QueryParam("refine") @DefaultValue("false") boolean refine,
                                  @QueryParam("fields") final List<String> refineFields,
                                  @QueryParam("sf") final List<String> specialFacets,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @QueryParam("format") final String format,
                                  @Context HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.SOFTWARE, keywords, doi, sortBy, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.SOFTWARE, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }

    @GET
    @Path("/api/software/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/software/count"}, longTask = false)
    public Response fetchSoftwareCount(@QueryParam("q") String query,
                                       @QueryParam("format") final String format,
                                       @QueryParam("fq") final List<String> fieldQueries,
                                       @Context final HttpServletRequest request)  {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.SOFTWARE, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/software/{softwareid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/software/{softwareid}"}, longTask = false)
    public Response fetchSoftware(@PathParam("softwareid") String datasetid,
                                  @QueryParam("format") final String format,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.SOFTWARE, datasetid, format, fieldQueries);
    }

    @GET
    @Path("/api/other")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/other"}, longTask = false)
    public Response fetchOther(@QueryParam("q") String keywords,
                               @QueryParam("doi") final String doi,
                               @QueryParam("sortBy") final String sortBy,
                               @DefaultValue("0") @QueryParam("page") int offset,
                               @DefaultValue("10") @QueryParam("size") int limit,
                               @QueryParam("refine") @DefaultValue("false") boolean refine,
                               @QueryParam("fields") final List<String> refineFields,
                               @QueryParam("sf") final List<String> specialFacets,
                               @QueryParam("fq") final List<String> fieldQueries,
                               @QueryParam("format") final String format,
                               @Context HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.OTHER, keywords, doi, sortBy, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.OTHER, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }

    @GET
    @Path("/api/other/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/other"}, longTask = false)
    public Response fetchOtherCount(@QueryParam("q") String query,
                                    @QueryParam("format") final String format,
                                    @QueryParam("fq") final List<String> fieldQueries,
                                    @Context final HttpServletRequest request)  {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.OTHER, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/other/{otherid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/other/{otherid}"}, longTask = false)
    public Response fetchOther(@PathParam("otherid") String datasetid,
                               @QueryParam("format") final String format,
                               @QueryParam("fq") final List<String> fieldQueries,
                               @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.OTHER, datasetid, format, fieldQueries);
    }
    @GET
    @Path("/api/projects")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects"}, longTask = false)
    public Response fetchProjects(@QueryParam("q") String keywords,
                                  @DefaultValue("0") @QueryParam("page") int offset,
                                  @DefaultValue("10") @QueryParam("size") int limit,
                                  @QueryParam("refine") @DefaultValue("false") boolean refine,
                                  @QueryParam("fields") final List<String> refineFields,
                                  @QueryParam("sf") final List<String> specialFacets,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @QueryParam("format") final String format,
                                  @Context HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.PROJECT, keywords, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.PROJECT, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }


    @GET
    @Path("/api/projects/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/count"}, longTask = false)
    public Response fetchProjectsCount(@QueryParam("q") String query,
                                       @QueryParam("format") final String format,
                                       @QueryParam("fq") final List<String> fieldQueries,
                                       @Context final HttpServletRequest request) {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.PROJECT, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectid}"}, longTask = false)
    public Response fetchProjects(@PathParam("projectid") String projectid,
                                  @QueryParam("format") final String format,
                                  @QueryParam("fq") final List<String> fieldQueries,
                                  @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.PROJECT, projectid, format, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/publications")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectid}/publications"}, longTask = false)
    public Response fetchProjectPublications(@PathParam("projectId") String projectId,
                                             @DefaultValue("0") @QueryParam("page") int offset,
                                             @DefaultValue("10") @QueryParam("size") int limit,
                                             @QueryParam("format") final String format,
                                             @QueryParam("fq") final List<String> fieldQueries,
                                             @Context HttpServletRequest request)  {

        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relprojectid", projectId, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.PUBLICATION, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/publications/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectid}/publications/count"}, longTask = false)
    public Response fetchProjectPublicationsCount(@PathParam("projectId") String projectId,
                                                  @QueryParam("format") final String format,
                                                  @QueryParam("fq") final List<String> fieldQueries,
                                                  @Context final HttpServletRequest request) {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relprojectid", projectId, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/datasets")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/datasets"}, longTask = false)
    public Response fetchProjectDatasets(@PathParam("projectId") String projectId,
                                         @DefaultValue("0") @QueryParam("page") int offset,
                                         @DefaultValue("10") @QueryParam("size") int limit,
                                         @QueryParam("format") final String format,
                                         @QueryParam("fq") final List<String> fieldQueries,
                                         @Context HttpServletRequest request)  {

        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relprojectid", projectId, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.DATASET, offset, limit, format, request, false, null, null, fieldQueries);
    }


    @GET
    @Path("/api/projects/{projectId}/datasets/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/datasets/count"}, longTask = false)
    public Response fetchProjectDatasetsCount(@PathParam("projectId") String projectId,
                                              @DefaultValue("0") @QueryParam("page") int offset,
                                              @DefaultValue("10") @QueryParam("size") int limit,
                                              @QueryParam("format") final String format,
                                              @QueryParam("fq") final List<String> fieldQueries,
                                              @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relprojectid", projectId, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }



    @GET
    @Path("/api/projects/{projectId}/software/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/software/count"}, longTask = false)
    public Response fetchProjectSoftwareCount(@PathParam("projectId") String projectId,
                                              @DefaultValue("0") @QueryParam("page") int offset,
                                              @DefaultValue("10") @QueryParam("size") int limit,
                                              @QueryParam("format") final String format,
                                              @QueryParam("fq") final List<String> fieldQueries,
                                              @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.SOFTWARE, "relprojectid", projectId, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/software")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/software"}, longTask = false)
    public Response fetchProjectSoftware(@PathParam("projectId") String projectId,
                                         @DefaultValue("0") @QueryParam("page") int offset,
                                         @DefaultValue("10") @QueryParam("size") int limit,
                                         @QueryParam("format") final String format,
                                         @QueryParam("fq") final List<String> fieldQueries,
                                         @Context HttpServletRequest request)  {

        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.SOFTWARE, "relprojectid", projectId, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.SOFTWARE, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/other/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/other/count"}, longTask = false)
    public Response fetchProjectOtherCount(@PathParam("projectId") String projectId,
                                           @DefaultValue("0") @QueryParam("page") int offset,
                                           @DefaultValue("10") @QueryParam("size") int limit,
                                           @QueryParam("format") final String format,
                                           @QueryParam("fq") final List<String> fieldQueries,
                                           @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.OTHER, "relprojectid", projectId, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/projects/{projectId}/other")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/other"}, longTask = false)
    public Response fetchProjectOther(@PathParam("projectId") String projectId,
                                      @DefaultValue("0") @QueryParam("page") int offset,
                                      @DefaultValue("10") @QueryParam("size") int limit,
                                      @QueryParam("format") final String format,
                                      @QueryParam("fq") final List<String> fieldQueries,
                                      @Context HttpServletRequest request)  {

        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.OTHER, "relprojectid", projectId, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.OTHER, offset, limit, format, request, false, null, null, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/results")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/results"}, longTask = false)
    public Response fetchProjectResults(@PathParam("projectId") String projectId,
                                      @DefaultValue("0") @QueryParam("page") int offset,
                                      @DefaultValue("10") @QueryParam("size") int limit,
                                      @QueryParam("format") final String format,
                                      @QueryParam("fq") final List<String> fieldQueries,
                                      @Context HttpServletRequest request)  {

        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.RESULT, "relprojectid", projectId, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.RESULT, offset, limit, format, request, false, null, null, fieldQueries);
    }

    @GET
    @Path("/api/projects/{projectId}/results/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/projects/{projectId}/results/count"}, longTask = false)
    public Response fetchProjectResultsCount(@PathParam("projectId") String projectId,
                                           @DefaultValue("0") @QueryParam("page") int offset,
                                           @DefaultValue("10") @QueryParam("size") int limit,
                                           @QueryParam("format") final String format,
                                           @QueryParam("fq") final List<String> fieldQueries,
                                           @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.RESULT, "relprojectid", projectId, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/datasources")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasources"}, longTask = false)
    public Response fetchDatasources(@QueryParam("q") String keywords,
                                     @DefaultValue("0") @QueryParam("page") final int offset,
                                     @DefaultValue("10") @QueryParam("size") final int limit,
                                     @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                     @QueryParam("fields") final List<String> refineFields,
                                     @QueryParam("fq") final List<String> fieldQueries,
                                     @QueryParam("sf") final List<String> specialFacets,
                                     @QueryParam("format") final String format,
                                     @Context final HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.DATASOURCE, keywords, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.DATASOURCE, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }

    @GET
    @Path("/api/datasources/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasources/count"}, longTask = false)
    public Response fetchDatasourcesCount(@QueryParam("q") String query,
                                          @QueryParam("format") final String format,
                                          @QueryParam("fq") final List<String> fieldQueries,
                                          @Context final HttpServletRequest request) {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.DATASOURCE, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/datasources/{datasourceid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/datasources/{datasourceid}"}, longTask = false)
    public Response fetchDatasources(@PathParam("datasourceid") String datasourceid,
                                     @QueryParam("format") final String format,
                                     @QueryParam("fq") final List<String> fieldQueries,
                                     @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.DATASOURCE, datasourceid, format, fieldQueries);
    }


    @GET
    @Path("/api/organizations")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations"}, longTask = false)
    public Response fetchOrganisations(@QueryParam("q") String keywords,
                                       @DefaultValue("0") @QueryParam("page") final int offset,
                                       @DefaultValue("10") @QueryParam("size") final int limit,
                                       @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                       @QueryParam("fields") final List<String> refineFields,
                                       @QueryParam("sf") final List<String> specialFacets,
                                       @QueryParam("fq") final List<String> fieldQueries,
                                       @QueryParam("format") final String format,
                                       @Context final HttpServletRequest request) {

        String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.ORGANIZATION, keywords, fieldQueries);
        return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.ORGANIZATION, offset, limit, format, request, refine, refineFields, specialFacets, fieldQueries);
    }


    @GET
    @Path("/api/organizations/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/count"}, longTask = false)
    public Response fetchOrganizationsCount(@QueryParam("q") String query,
                                            @QueryParam("format") final String format,
                                            @QueryParam("fq") final List<String> fieldQueries,
                                            @Context final HttpServletRequest request) {
        String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.ORGANIZATION, query, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}"}, longTask = false)
    public Response fetchOrganizations(@PathParam("organizationid") String organizationid,
                                       @QueryParam("format") final String format,
                                       @QueryParam("fq") final List<String> fieldQueries,
                                       @Context final HttpServletRequest request) {
        return getResponseByEntityId(request, RequestResponseHandler.Entity.ORGANIZATION, organizationid, format, fieldQueries);
    }


    @GET
    @Path("/api/organizations/{organizationid}/datasources")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/datasources"}, longTask = false)
    public Response fetchOrganizationDatasources(@PathParam("organizationid") String organizationid,
                                                 @DefaultValue("0") @QueryParam("page") int offset,
                                                 @DefaultValue("10") @QueryParam("size") int limit,
                                                 @QueryParam("format") final String format,
                                                 @QueryParam("fq") final List<String> fieldQueries,
                                                 @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASOURCE, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.DATASOURCE, offset, limit, format, request, false, null, null, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/datasources/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/datasources/count"}, longTask = false)
    public Response fetchOrganizationDatasources(@PathParam("organizationid") String organizationid,
                                                 @QueryParam("format") final String format,
                                                 @QueryParam("fq") final List<String> fieldQueries,
                                                 @Context final HttpServletRequest request) {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASOURCE, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }


    @GET
    @Path("/api/organizations/{organizationid}/publications")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/publications"}, longTask = false)
    public Response fetchOrganizationPublications(@PathParam("organizationid") String organizationid,
                                                  @DefaultValue("0") @QueryParam("page") int offset,
                                                  @DefaultValue("10") @QueryParam("size") int limit,
                                                  @QueryParam("format") final String format,
                                                  @QueryParam("fq") final List<String> fieldQueries,
                                                  @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.PUBLICATION, offset, limit, format, request, false, null, null, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/publications/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/publications/count"}, longTask = false)
    public Response fetchOrganizationPublicationsCount(@PathParam("organizationid") String organizationid,
                                                       @DefaultValue("0") @QueryParam("page") int offset,
                                                       @DefaultValue("10") @QueryParam("size") int limit,
                                                       @QueryParam("format") final String format,
                                                       @QueryParam("fq") final List<String> fieldQueries,
                                                       @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/datasets")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/datasets"}, longTask = false)
    public Response fetchOrganizationDatasets(@PathParam("organizationid") String organizationid,
                                             @DefaultValue("0") @QueryParam("page") int offset,
                                             @DefaultValue("10") @QueryParam("size") int limit,
                                             @QueryParam("format") final String format,
                                             @QueryParam("fq") final List<String> fieldQueries,
                                             @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.DATASET, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/datasets/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/datasets/count"}, longTask = false)
    public Response fetchOrganizationDatasetsCount(@PathParam("organizationid") String organizationid,
                                                  @DefaultValue("0") @QueryParam("page") int offset,
                                                  @DefaultValue("10") @QueryParam("size") int limit,
                                                  @QueryParam("format") final String format,
                                                  @QueryParam("fq") final List<String> fieldQueries,
                                                  @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/software")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/software"}, longTask = false)
    public Response fetchOrganizationSoftware(@PathParam("organizationid") String organizationid,
                                             @DefaultValue("0") @QueryParam("page") int offset,
                                             @DefaultValue("10") @QueryParam("size") int limit,
                                             @QueryParam("format") final String format,
                                             @QueryParam("fq") final List<String> fieldQueries,
                                             @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.SOFTWARE, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.SOFTWARE, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/software/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/software/count"}, longTask = false)
    public Response fetchOrganizationSoftwareCount(@PathParam("organizationid") String organizationid,
                                                  @DefaultValue("0") @QueryParam("page") int offset,
                                                  @DefaultValue("10") @QueryParam("size") int limit,
                                                  @QueryParam("format") final String format,
                                                  @QueryParam("fq") final List<String> fieldQueries,
                                                  @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.SOFTWARE, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/other")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/other"}, longTask = false)
    public Response fetchOrganizationOther(@PathParam("organizationid") String organizationid,
                                              @DefaultValue("0") @QueryParam("page") int offset,
                                              @DefaultValue("10") @QueryParam("size") int limit,
                                              @QueryParam("format") final String format,
                                              @QueryParam("fq") final List<String> fieldQueries,
                                              @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.OTHER, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.OTHER, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/other/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/other/count"}, longTask = false)
    public Response fetchOrganizationOtherCount(@PathParam("organizationid") String organizationid,
                                                   @DefaultValue("0") @QueryParam("page") int offset,
                                                   @DefaultValue("10") @QueryParam("size") int limit,
                                                   @QueryParam("format") final String format,
                                                   @QueryParam("fq") final List<String> fieldQueries,
                                                   @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.OTHER, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/results")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/results"}, longTask = false)
    public Response fetchOrganizationResults(@PathParam("organizationid") String organizationid,
                                           @DefaultValue("0") @QueryParam("page") int offset,
                                           @DefaultValue("10") @QueryParam("size") int limit,
                                           @QueryParam("format") final String format,
                                           @QueryParam("fq") final List<String> fieldQueries,
                                           @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.RESULT, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.RESULT, offset, limit, format, request, false, null, null,fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/results/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/other/count"}, longTask = false)
    public Response fetchOrganizationResultsCount(@PathParam("organizationid") String organizationid,
                                                @DefaultValue("0") @QueryParam("page") int offset,
                                                @DefaultValue("10") @QueryParam("size") int limit,
                                                @QueryParam("format") final String format,
                                                @QueryParam("fq") final List<String> fieldQueries,
                                                @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.RESULT, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery, format, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/projects")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/projects"}, longTask = false)
    public Response fetchOrganizationProjects(@PathParam("organizationid") String organizationid,
                                              @DefaultValue("0") @QueryParam("page") int offset,
                                              @DefaultValue("10") @QueryParam("size") int limit,
                                              @QueryParam("format") final String format,
                                              @QueryParam("fq") final List<String> fieldQueries,
                                              @Context HttpServletRequest request)  {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PROJECT, "relorganizationid", organizationid, fieldQueries);
        return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.PROJECT, offset, limit, format, request, false, null, null, fieldQueries);
    }

    @GET
    @Path("/api/organizations/{organizationid}/projects/count")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/organizations/{organizationid}/projects/count"}, longTask = false)
    public Response fetchOrganizationProjectsCount(@PathParam("organizationid") String organizationid,
                                                   @QueryParam("format") final String format,
                                                   @QueryParam("fq") final List<String> fieldQueries,
                                                   @Context final HttpServletRequest request) {
        String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PROJECT, "relorganizationid", organizationid, fieldQueries);
        return getCount(request, fullQuery,format, fieldQueries);
    }

    /*

        @GET
        @Path("/api/people")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeople(@QueryParam("q") String keywords,
                                    @DefaultValue("0") @QueryParam("page") final int offset,
                                    @DefaultValue("10") @QueryParam("size") final int limit,
                                    @QueryParam("refine") @DefaultValue("false") final boolean refine,
                                    @QueryParam("fields") final List<String> refineFields,
                                    @QueryParam("fq") final List<String> fieldQueries,
                                    @QueryParam("format") final String format,
                                    @Context final HttpServletRequest request) {

            String simpleQuery = buildSearchRequest(RequestResponseHandler.Entity.PERSON, keywords, fieldQueries);
            return getResponseByEntity(simpleQuery, RequestResponseHandler.Entity.PERSON, offset, limit, format, request, refine, refineFields, fieldQueries);
        }


        @GET
        @Path("/api/people/count")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeopleCount(@QueryParam("q") String query,
                                         @QueryParam("format") final String format,
                                         @QueryParam("fq") final List<String> fieldQueries,
                                         @Context final HttpServletRequest request) {
            String fullQuery = buildSearchRequest(RequestResponseHandler.Entity.PERSON, query, fieldQueries);
            return getCount(request, fullQuery, format, fieldQueries);
        }

        @GET
        @Path("/api/people/{personid}")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeople(@PathParam("personid") String personid,
                                    @QueryParam("format") final String format,
                                    @QueryParam("fq") final List<String> fieldQueries,
                                    @Context final HttpServletRequest request) {
            return getResponseByEntityId(request, RequestResponseHandler.Entity.PERSON, personid, format, fieldQueries);
        }


        @GET
        @Path("/api/people/{personid}/publications")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeoplePublications(@PathParam("personid") String personid,
                                                @DefaultValue("0") @QueryParam("page") final int offset,
                                                @DefaultValue("10") @QueryParam("size") final int limit,
                                                @QueryParam("format") final String format,
                                                @QueryParam("fq") final List<String> fieldQueries,
                                                @Context final HttpServletRequest request) {
            String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relpersonid", personid, fieldQueries);
            return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.PUBLICATION, offset, limit, format, request, false, null, fieldQueries);
        }

        @GET
        @Path("/api/people/{personid}/publications/count")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeoplePublicationsCount(@PathParam("personid") String personid,
                                                     @QueryParam("format") final String format,
                                                     @QueryParam("fq") final List<String> fieldQueries,
                                                     @Context final HttpServletRequest request) {
            String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.PUBLICATION, "relpersonid", personid, fieldQueries);
            return getCount(request, fullQuery, format, fieldQueries);
        }


        @GET
        @Path("/api/people/{personid}/datasets")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeopleDatasets(@PathParam("personid") String personid,
                                            @DefaultValue("0") @QueryParam("page") final int offset,
                                            @DefaultValue("10") @QueryParam("size") final int limit,
                                            @QueryParam("format") final String format,
                                            @QueryParam("fq") final List<String> fieldQueries,
                                            @Context final HttpServletRequest request) {
            String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relpersonid", personid, fieldQueries);
            return getResponseByEntity(fullQuery, RequestResponseHandler.Entity.DATASET, offset, limit, format, request, false, null, fieldQueries);
        }

        @GET
        @Path("/api/people/{personid}/datasets/count")
        @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
        public Response fetchPeopleDatasetsCount(@PathParam("personid") String personid,
                                                 @QueryParam("format") final String format,
                                                 @QueryParam("fq") final List<String> fieldQueries,
                                                 @Context final HttpServletRequest request) {
            String fullQuery = builtEntity2EntityRelationQuery(RequestResponseHandler.Entity.DATASET, "relpersonid", personid, fieldQueries);
            return getCount(request, fullQuery, format, fieldQueries);
        }
    */
    @GET
    @Path("/api/reports")
    @Produces(MediaType.TEXT_PLAIN)
    @Timed(value = "http.server.request.duration", extraTags = {"referer", "portal", "uri", "/rest/v2/api/reports"}, longTask = false)
    public Response fetchReport(@QueryParam("query") String query,
                                @QueryParam("refine") @DefaultValue("false") boolean refine,
                                @QueryParam("fields") final List<String> refineFields,
                                @QueryParam("fq") final List<String> fieldQueries,
                                @QueryParam("type") final List<String> types,
                                @QueryParam("format") final String format,
                                @Context final HttpServletResponse response) {

        long startTime = System.nanoTime();
        logger.debug("Reports request received");

        final boolean special = isSpecialFormat(format);
        String responseType =  extractResponseFormat(format);

        if(checkTypes(types)) {
            return Response.status(Response.Status.BAD_REQUEST).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }

        RequestResponseHandler.Entity basicEntity = identifyBasicEntityType(types);
        if (basicEntity.equals(RequestResponseHandler.Entity.NONE)) {
            return Response.status(Response.Status.BAD_REQUEST).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }

        Transformer transformer = extractTransformer(format, basicEntity);

        StringBuilder queryBuilder = new StringBuilder();
        if (basicEntity.equals(RequestResponseHandler.Entity.RESULT)) {
            enhanceFieldQueryWithResultType(types, fieldQueries);

        } else {
            enhanceFieldQueryWithEntityType(basicEntity, fieldQueries);
        }

        final String fullquery;
        if (query==null || query.isEmpty()) {
            fullquery = "*";
        } else {
            fullquery = query;
        }

        logger.debug("fullquery " + query);
        logger.debug("fieldQueries " + fieldQueries);

        try {
            SearchResult sr = ((SearchServiceImpl)searchService).newSearch(fullquery, Locale.getDefault().toString(), refineFields, null, fieldQueries,0, 0, format, transformer,null,false);
            logger.debug("Total number of results " + sr.getTotal());


        } catch (SearchServiceException sse) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(APIResponseFormatter.
                    compose500Message(MediaType.APPLICATION_JSON, "Fail to fetch report.", sse.getMessage())).build();
        }

        StreamingOutput stream = new StreamingOutput() {
            @Override
            public void write(OutputStream os) throws IOException, WebApplicationException
            {
                if (hasTitle(format) && !format.equals("html")) {
                    os.write(CSVResponseFormat.appendTitle(basicEntity, special).getBytes());
                }
                try {
                    ((SearchServiceImpl)searchService).cursorSearch(fullquery, refineFields, null, fieldQueries, format, transformer, os);

                } catch (SearchServiceException sse) {
                    os.write(("Fail to return report. " + sse.getMessage()).getBytes());
                    logger.warn("Fail to return report. ", sse);
                }

                //os.close();
            }

        };

        long estimatedTime = System.nanoTime() - startTime;
        logger.debug("Reports search time " + estimatedTime/1000000 +  " milliseconds");
        return Response.ok().entity(stream).type(responseType).build();
    }

    private String extractResponseFormat(String format) {
        if (format != null && !format.isEmpty()) {
            if (format.equalsIgnoreCase("json")) {
                return MediaType.APPLICATION_JSON;

            } else if (format.contains("csv")) {
                return new MediaType("text", "csv").toString();

            }  else if (format.equalsIgnoreCase("tsv")) {
                return new MediaType("text", "tsv").toString();

            } else if (format.equalsIgnoreCase("html")) {
                return MediaType.TEXT_HTML;
            }
        }

        return MediaType.APPLICATION_XML;
    }

    private boolean isSpecialFormat(String format) {
        if (format != null && format.toLowerCase().contains("special")) {
            return true;
        }
        return false;
    }

    private boolean hasTitle(String format) {
        if (format != null && format.toLowerCase().contains("notitle")) {
            return false;
        }
        return true;
    }

    //TODO differentiate for html + add special controls for format support
    private Transformer extractTransformer(String format, RequestResponseHandler.Entity entity) {
        if (searchRegistry == null) { return  null;}

        if (format != null && (format.toLowerCase().contains("csv") || format.equalsIgnoreCase("html"))) {
            logger.debug("Extracting transformer...." + format + "_" + entity);
            return searchRegistry.getTransformer(format.replaceAll("-notitle","") + "_" + entity, Locale.getDefault());

        }

        return null;
    }

    //TODO old way of creating query see if needed.
    /*private String buildSimpleSearchQuery(RequestResponseHandler.Entity entity, String keywords) {
        StringBuilder queryBuilder = new StringBuilder();
        enhanceQueryWithEntityType(queryBuilder, entity);
        builtQueryKeywords(queryBuilder, cleanKeywords(keywords));
        return queryBuilder.toString();
    }*/

    private static String cleanKeywords(String keywords) {
        if (keywords != null) {
            return keywords.replaceAll("[\\\\/:*?<>|()]", "");
        }
        return keywords;
    }

    private String cleanId(String entityId) {
        return entityId.replaceAll("\"", "");
    }

    private String buildSearchRequest(RequestResponseHandler.Entity entity, String keywords, String doi, String sortBy, List<String> fieldQueries) {
        enhanceFieldQueryWithEntityType(entity, fieldQueries);

        StringBuilder queryBuilder = new StringBuilder();
        if (doi != null && !doi.trim().isEmpty()) {
            queryBuilder.append("(pidclassid exact \"doi\" and pid exact " + doi + ")");
        }

        builtQueryKeywords(queryBuilder, keywords);
        enhanceWithSortParameter(queryBuilder, sortBy);

        return queryBuilder.toString();
    }

    public static String buildSearchRequest(List<String> types, String keywords, String doi, String sortBy, List<String> fieldQueries) {

        List<RequestResponseHandler.Entity> entities = extractEntities(types);
        //all results are of oaftype result
        enhanceFieldQueryWithResultType(types, fieldQueries);
        
        StringBuilder queryBuilder = new StringBuilder();
        if (doi != null && !doi.trim().isEmpty()) {
            if (!queryBuilder.toString().isEmpty()) {
                queryBuilder.append(" and ");
            }
            queryBuilder.append("(pidclassid exact \"doi\" and pid exact " + doi + ")");
        }

        builtQueryKeywords(queryBuilder, keywords);
        enhanceWithSortParameter(queryBuilder, sortBy);

        return queryBuilder.toString();
    }

    private String buildSearchRequest(RequestResponseHandler.Entity entity, String keywords, List<String> fieldQueries) {
        enhanceFieldQueryWithEntityType(entity, fieldQueries);

        StringBuilder queryBuilder = new StringBuilder();
        builtQueryKeywords(queryBuilder, keywords);

        return queryBuilder.toString();
    }

    private String buildSearchRequest(List<String> types, String keywords, List<String> fieldQueries) {

        List<RequestResponseHandler.Entity> entities = extractEntities(types);
        enhanceFieldQueryWithEntityType(RequestResponseHandler.Entity.RESULT, fieldQueries);

        StringBuilder queryBuilder = new StringBuilder();
        builtQueryKeywords(queryBuilder, keywords);

        return queryBuilder.toString();
    }

    private static void enhanceFieldQueryWithEntityType(RequestResponseHandler.Entity entity, List<String> fieldQueries) {
        for (String fieldQuery: entity.getFieldQueries()) {
            logger.debug("adding " + fieldQuery);
            fieldQueries.add(fieldQuery);
        }
    }

    private static void enhanceFieldQueryWithResultType(final List<String> types, List<String> fieldQueries) {

        fieldQueries.add(RequestResponseHandler.Entity.RESULT.getSimpleQuery());
        List<RequestResponseHandler.Entity> entities = extractEntities(types);

        if (types!= null && !types.isEmpty()) {
            StringBuilder queryBuilder = new StringBuilder();
            for(RequestResponseHandler.Entity entity: entities) {
                    CQLQueryBuilder.appendSimpleTerm(queryBuilder, CQLQueryBuilder.Operator.OR, entity.getSimpleQuery());
            }
            fieldQueries.add(queryBuilder.toString());
        }
    }



    private static void builtQueryKeywords(StringBuilder queryBuilder, String keywords) {
        if (keywords != null && !keywords.trim().isEmpty()) {
            CQLQueryBuilder.appendKeywords(queryBuilder, cleanKeywords(keywords));
        }

        if (queryBuilder.toString().isEmpty()) {
            queryBuilder.append("(*)");
        }
    }

    private void builtEntityIdQuery(StringBuilder queryBuilder, RequestResponseHandler.Entity entity, String entityId, List<String> fieldQueries){
        enhanceFieldQueryWithEntityType(entity, fieldQueries);
        enhanceQueryWithEntityId(queryBuilder, entity, cleanId(entityId));
    }

    /* Queries with pids are only valid with deleted by inference marked as true */
    private void resultPidQuery(StringBuilder queryBuilder, RequestResponseHandler.Entity entity, String resultPid, List<String> fieldQueries){
        builtDeletedByInferenceQuery(queryBuilder);

        if (entity != RequestResponseHandler.Entity.NONE && entity != RequestResponseHandler.Entity.ORGANIZATION) {
            throw new IllegalArgumentException("Entity " + entity.toString() + " is not supported.");
        }
        enhanceQueryWithPid(queryBuilder, entity, cleanId(resultPid));
    }

    private void builtDeletedByInferenceQuery(StringBuilder queryBuilder) {
        CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.NONE, "deletedByInference", CQLQueryBuilder.Operator.EQUAL, "true");
    }

    private String builtEntity2EntityRelationQuery(RequestResponseHandler.Entity entity, String relationfield, String entityId, List<String> fieldQueries){
        StringBuilder queryBuilder = new StringBuilder();
        //enhanceQueryWithEntityType(queryBuilder, entity);
        enhanceFieldQueryWithEntityType(entity, fieldQueries);
        enhanceQueryWithRelationId(queryBuilder, relationfield, cleanId(entityId));
        return queryBuilder.toString();
    }

    private void enhanceQueryWithRelationId(StringBuilder queryBuilder, String relationfield, String entityId) {
        if (!queryBuilder.toString().isEmpty()) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.AND, " " + relationfield + " ", CQLQueryBuilder.Operator.EXACT, entityId);

        } else {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.NONE, relationfield + " ", CQLQueryBuilder.Operator.EXACT, entityId);
        }
    }

    private void enhanceQueryWithEntityId(StringBuilder queryBuilder, RequestResponseHandler.Entity entity, String entityId) {
        if (queryBuilder.toString().isEmpty()) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.NONE, "objidentifier", CQLQueryBuilder.Operator.EXACT, entityId);
        } else {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.AND, "objidentifier", CQLQueryBuilder.Operator.EXACT, entityId);
        }

        if (entity == RequestResponseHandler.Entity.PUBLICATION || entity == RequestResponseHandler.Entity.DATASET
                || entity == RequestResponseHandler.Entity.SOFTWARE || entity == RequestResponseHandler.Entity.OTHER) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.OR, "resultdupid", CQLQueryBuilder.Operator.EXACT, entityId);

        } else if (entity == RequestResponseHandler.Entity.ORGANIZATION) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.OR, "organizationdupid", CQLQueryBuilder.Operator.EXACT, entityId);
        }
    }



    private void enhanceQueryWithPid(StringBuilder queryBuilder, RequestResponseHandler.Entity entity, String entityId) {
        if (entity == RequestResponseHandler.Entity.NONE) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.AND, "resultdupid", CQLQueryBuilder.Operator.EXACT, entityId);

        } else if (entity == RequestResponseHandler.Entity.ORGANIZATION) {
            CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.AND, "organizationdupid", CQLQueryBuilder.Operator.EXACT, entityId);

        } else {
            throw new IllegalArgumentException("The entity " + entity.toString() + " is not supported.");
        }
    }

    private static void enhanceWithSortParameter(StringBuilder queryBuilder, String sortBy) {
        if (sortBy != null) {
            String[] sortParams = sortBy.split(",");

            if (sortParams.length != 2) {
                throw new IllegalArgumentException("Invalid sort paremeter. 'sortBy' parameter format is <fieldName>[,asc|,desc].");
            }

            String sortByField = sortParams[0];
            String order = sortParams[1];

            if (!sortByField.equals("resultdateofacceptance")){
                throw new IllegalArgumentException("'" + sortByField + "' is not a sortable field.");
            }

            if (!checkOrder(order)) {
                throw new IllegalArgumentException("'" + order + "' is not a valid ordering. Please use one of {ascending, descending}");
            }

            addSortParameter(sortByField, order, queryBuilder);
        }
    }

    private static boolean checkOrder(String order) {
        if (order.matches("ascending|descending")) {
            return true;
        }
        return false;
    }

    private static void addSortParameter(String indexField, String order, StringBuilder queryBuilder) {
        queryBuilder.append(" sortBy " + indexField + "/sort." + order);
    }

    private Response getCount(HttpServletRequest request, String query, String format, List<String> fieldQueries) {
        String responseFormat = extractResponseFormat(format);

        try {
            SearchResult searchResult = ((SearchServiceImpl)searchService).newSearch(query, Locale.getDefault().toString(), null, null, fieldQueries, 0, 0, responseFormat, null, null, false);
            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createCountMeta(request, query, searchResult.getTotal(), responseFormat)).type(responseFormat).build();

        } catch (SearchServiceException sse) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(APIResponseFormatter.compose500Message(responseFormat, "Fail to fetch count for query " + query + ".", sse.getMessage())).build();
        }
    }

    private Response getResponseByEntityId(HttpServletRequest request, RequestResponseHandler.Entity entity, String entityId, String format, List<String> fieldQueries) {
        String responseType = extractResponseFormat(format);

        try {
            StringBuilder queryBuilder = new StringBuilder();
            builtEntityIdQuery(queryBuilder, entity, entityId, fieldQueries);
            String fullQuery = queryBuilder.toString();

            if(fullQuery == null || fullQuery.isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST).
                        entity(APIResponseFormatter.compose400Message(format, "The 'query' parameter is required")).
                        type(responseType).build();
            }

            SearchResult searchResult = ((SearchServiceImpl)searchService).newSearch(fullQuery, Locale.getDefault().toString(), null, null, fieldQueries, 0, 1, responseType, null, null, false);

            if (searchResult.getSearchResults() == null || searchResult.getSearchResults().isEmpty() ||
                    searchResult.getSearchResults().size() == 0 || searchResult.getSearchResults().get(0) == null) {
                return Response.status(Response.Status.NOT_FOUND).entity(APIResponseFormatter.compose404Message(responseType, "404 - " + entity +" with id "
                        + entityId + " not found.")).type(responseType).build();
            }

            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createEntityResponse(request, entity, searchResult.getSearchResults().get(0).toString(), responseType)).type(responseType).build();

        } catch (SearchServiceException sse) {
            logger.error("Fail to fetch "+ entity + " with id " + entityId, sse);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(APIResponseFormatter.compose500Message(responseType, "Fail to fetch " + entity + " with id " + entityId, sse.getMessage())).build();
        }
    }

    private Response getResponseByEntity(String fullQuery, RequestResponseHandler.Entity entity, int offset, int limit, String format, HttpServletRequest request, boolean refine, List<String> refineFields, List<String> specialFacets, List<String> fieldQueries) {
        long startTime = System.nanoTime();
        Transformer transformer = extractTransformer(format, entity);
        boolean special = isSpecialFormat(format);
        boolean hasTitle = hasTitle(format);
        String responseType = extractResponseFormat(format);

        if(fullQuery == null || fullQuery.isEmpty()) {
            return Response.status(Response.Status.BAD_REQUEST).
                    entity(APIResponseFormatter.compose400Message(responseType, "The 'query' parameter is required")).
                    type(responseType).build();
        }

        try {
            SearchResult solrResults = ((SearchServiceImpl)searchService).newSearch(fullQuery, Locale.getDefault().toString(), refineFields, specialFacets, fieldQueries, offset, limit, responseType, transformer, null, false);
            long estimatedTime = System.nanoTime() - startTime;
            logger.debug("Publications search time " + estimatedTime/1000000 +  " milliseconds");
            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createEntitiesResponse(request, entity, fullQuery, fieldQueries, solrResults, refine, responseType, special, hasTitle)).type(responseType).build();

        } catch (SearchServiceException sse) {
            logger.error("Fail to get " + entity.getPlural() , sse);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).
                    entity(APIResponseFormatter.compose500Message(responseType, "Fail to fetch " + entity.getPlural(), sse.getMessage())).type(responseType).build();
        }
    }

    private Response getResponseByEntityTypes(String fullQuery, List<String> types, int offset, int limit, String format, HttpServletRequest request, boolean refine, List<String> refineFields, List<String> specialFacets, List<String> fieldQueries) {
        long startTime = System.nanoTime();

        boolean special = isSpecialFormat(format);
        boolean hasTitle = hasTitle(format);
        String responseType = extractResponseFormat(format);

        if(checkTypes(types)) {
            return Response.status(Response.Status.FORBIDDEN).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }

        RequestResponseHandler.Entity basicEntity = identifyBasicEntityType(types);
        if (basicEntity.equals(RequestResponseHandler.Entity.NONE)) {
            return Response.status(Response.Status.FORBIDDEN).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }
        Transformer transformer = extractTransformer(format, basicEntity);

        //List<RequestResponseHandler.Entity> entities = extractEntities(types);
        StringBuilder queryBuilder = new StringBuilder();
        if (basicEntity.equals(RequestResponseHandler.Entity.RESULT)) {
            enhanceFieldQueryWithResultType(types, fieldQueries);
        } else {
            enhanceFieldQueryWithEntityType(basicEntity, fieldQueries);
        }

        try {
            if (fullQuery==null || fullQuery.isEmpty()) {
                fullQuery = "*";
            }

            SearchResult solrResults = ((SearchServiceImpl)searchService).newSearch(fullQuery, Locale.getDefault().toString(), refineFields, specialFacets, fieldQueries, offset, limit, responseType, transformer, null, false);
            long estimatedTime = System.nanoTime() - startTime;
            logger.debug("Publications search time " + estimatedTime/1000000 +  " milliseconds");
            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createEntitiesResponse(request, basicEntity, fullQuery, fieldQueries, solrResults, refine, responseType, special, hasTitle)).type(responseType).build();

        } catch (SearchServiceException sse) {
            logger.error("Fail to get " + basicEntity.getPlural() , sse);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).
                    entity(APIResponseFormatter.compose500Message(responseType, "Fail to fetch " + basicEntity.getPlural(), sse.getMessage())).type(responseType).build();
        }
    }

    private Response getResponseByEntityTypes(String fullQuery, String pid, String pidType, List<String> types, int offset, int limit, String format, HttpServletRequest request, boolean refine, List<String> refineFields, List<String> specialFacets, List<String> fieldQueries) {
        long startTime = System.nanoTime();

        boolean special = isSpecialFormat(format);
        boolean hasTitle = hasTitle(format);
        String responseType = extractResponseFormat(format);

        if(checkTypes(types)) {
            return Response.status(Response.Status.FORBIDDEN).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }

        RequestResponseHandler.Entity basicEntity = identifyBasicEntityType(types);
        if (basicEntity.equals(RequestResponseHandler.Entity.NONE)) {
            return Response.status(Response.Status.FORBIDDEN).
                    entity(APIResponseFormatter.compose400Message(responseType, "The type parameter is missing or is invalid.")).type(responseType).build();
        }
        Transformer transformer = extractTransformer(format, basicEntity);

        //List<RequestResponseHandler.Entity> entities = extractEntities(types);

        if (basicEntity.equals(RequestResponseHandler.Entity.RESULT)) {
            enhanceFieldQueryWithResultType(types, fieldQueries);
        } else {
            enhanceFieldQueryWithEntityType(basicEntity, fieldQueries);
        }

        StringBuilder queryBuilder = new StringBuilder();
        logger.debug("pid " +pid);
        logger.debug("pidtype" +pidType);
        
        if (pid != null && !pid.isEmpty()) {
            if ((pidType != null) && !pidType.isEmpty()) {
                enhanceQueryWithPersistentIdentifier(queryBuilder, pid, pidType);
            } else {
                return Response.status(Response.Status.BAD_REQUEST).
                        entity(APIResponseFormatter.compose400Message(responseType, "The pid type is missing.")).type(responseType).build();
            }
        }
        try {
            logger.debug("Query builder " + queryBuilder.toString());
            logger.debug("Query " + fullQuery);

            if (!queryBuilder.toString().isEmpty()){
                if (fullQuery != null && !fullQuery.isEmpty()) {
                    queryBuilder.append(" ").append(CQLQueryBuilder.Operator.AND).append(" ").append(fullQuery);
                }
            } else {
                if (fullQuery != null && !fullQuery.isEmpty()) {
                    queryBuilder.append(fullQuery);
                } else {
                    queryBuilder.append("*");
                }
            }

            SearchResult solrResults = ((SearchServiceImpl)searchService).newSearch(queryBuilder.toString(), Locale.getDefault().toString(), refineFields, specialFacets, fieldQueries, offset, limit, responseType, transformer, null, false);
            long estimatedTime = System.nanoTime() - startTime;
            logger.debug("Publications search time " + estimatedTime/1000000 +  " milliseconds");
            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createEntitiesResponse(request, basicEntity, queryBuilder.toString(), fieldQueries, solrResults, refine, responseType, special, hasTitle)).type(responseType).build();

        } catch (SearchServiceException sse) {
            logger.error("Fail to get " + basicEntity.getPlural() , sse);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).
                    entity(APIResponseFormatter.compose500Message(responseType, "Fail to fetch " + basicEntity.getPlural(), sse.getMessage())).type(responseType).build();
        }
    }

    private void enhanceQueryWithPersistentIdentifier(StringBuilder queryBuilder, String pid, String pidType) {
        queryBuilder.append("(pidclassid exact "+ pidType + " and pid exact \"" + pid + "\")");
    }

    private void enhanceQueryWithEntities(StringBuilder queryBuilder, List<String> types) {

    }


    private static List<RequestResponseHandler.Entity> extractEntities(final List<String> types) {
        List<RequestResponseHandler.Entity> entities = null;
        if (types!= null || types.isEmpty()) {
            logger.debug("types " + types);
            entities = new ArrayList<>();
            for (String type: types) {
                if (!entities.contains(type)) {
                    logger.debug("extracting for type " + type);
                    entities.add(extractEntity(type));
                }
            }
        }
        return entities;
    }

    private static RequestResponseHandler.Entity extractEntity(String type) {

        if (type == null) {
            return RequestResponseHandler.Entity.NONE;
        }

        if (type.equalsIgnoreCase(RequestResponseHandler.Entity.RESULT.getPlural())) {
            return RequestResponseHandler.Entity.RESULT;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.PUBLICATION.getPlural())) {
            return RequestResponseHandler.Entity.PUBLICATION;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.DATASET.getPlural())) {
            return RequestResponseHandler.Entity.DATASET;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.SOFTWARE.getPlural())) {
            return RequestResponseHandler.Entity.SOFTWARE;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.OTHER.getPlural())) {
            return RequestResponseHandler.Entity.OTHER;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.PROJECT.getPlural())) {
            return RequestResponseHandler.Entity.PROJECT;

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.DATASOURCE.getPlural())) {
            return RequestResponseHandler.Entity.DATASOURCE;

        /*} else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.PERSON.getPlural())) {
            return RequestResponseHandler.Entity.PERSON;
        */

        } else if (type.equalsIgnoreCase(RequestResponseHandler.Entity.ORGANIZATION.getPlural())) {
            return RequestResponseHandler.Entity.ORGANIZATION;
        }

        return RequestResponseHandler.Entity.NONE;
    }


    //This follows the newSearchWithoutFieldQueries comment. I wish to remove it!!! This was only made
    // (quick and dirty - only the getResponseByEntity version with newSearchWithoutFieldQueries is used
    // after a last time request for the portal to show all the publications and the deletedbyinference ones.
    // I did not want to pass a parameter since I do not know if we are going to keep it. This is for a tech meeting showcase.
    // If we want to keep this I need to redesign.
    @GET
    @Path("/api/deletedByInferencePublications/{publicationid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.requests.portal.publicationDuplicates", longTask = false)
    //TODO remove
    public Response fetchDeletedByInferencePublicationsById(@PathParam("publicationid") String publicationid,
                                                            @QueryParam("format") final String format,
                                                            @QueryParam("size") final int limit,
                                                            @QueryParam("fq") final List<String> fieldQueries,
                                                            @Context final HttpServletRequest request) {

        return getDeletedByInferenceResponseByPid(request, RequestResponseHandler.Entity.PUBLICATION, publicationid, format, limit, fieldQueries);
    }

    @GET
    @Path("/api/deletedByInferenceResults/{resultPid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.requests.portal.publicationDuplicates", longTask = false)
    public Response fetchDeletedByInferenceEntitiesById(@PathParam("resultPid") String resultPid,
                                                        @QueryParam("format") final String format,
                                                        @QueryParam("size") final int limit,
                                                        @QueryParam("fq") final List<String> fieldQueries,
                                                        @Context final HttpServletRequest request) {

        return getDeletedByInferenceResponseByPid(request, RequestResponseHandler.Entity.NONE, resultPid, format, limit, fieldQueries);
    }


    @GET
    @Path("/api/deletedByInferenceOrganizations/{organizationPid}")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    @Timed(value = "http.requests.portal.organizationDuplicates", longTask = false)
    public Response fetchDeletedByInferenceOrganizationById(@PathParam("organizationPid") String organizationPid,
                                                            @QueryParam("format") final String format,
                                                            @QueryParam("size") final int limit,
                                                            @QueryParam("fq") final List<String> fieldQueries,
                                                            @Context final HttpServletRequest request) {

        return getDeletedByInferenceResponseByPid(request, RequestResponseHandler.Entity.ORGANIZATION, organizationPid, format, limit, fieldQueries);
    }


    //Please see comment above. I wish to remove it.
    private Response getDeletedByInferenceResponseByPid(HttpServletRequest request, RequestResponseHandler.Entity entity, String resultPid, String format, int limit, List<String> fieldQueries) {
        String responseType = extractResponseFormat(format);

        try {
            StringBuilder queryBuilder = new StringBuilder();
            resultPidQuery(queryBuilder, entity, resultPid, fieldQueries);
            String fullQuery = queryBuilder.toString();

            if(fullQuery == null || fullQuery.isEmpty()) {
                return Response.status(Response.Status.BAD_REQUEST).
                        entity(APIResponseFormatter.compose400Message(format, "The 'query' parameter is required")).
                        type(responseType).build();
            }

            SearchResult searchResults = ((SearchServiceImpl)searchService).newSearchWithoutFieldQueries(fullQuery, Locale.getDefault().toString(), null, null, fieldQueries, 0, limit, responseType, null, null, false);

            if (searchResults.getSearchResults() == null || searchResults.getSearchResults().isEmpty() ||
                    searchResults.getSearchResults().size() == 0 || searchResults.getSearchResults().get(0) == null) {
                return Response.status(Response.Status.NOT_FOUND).entity(APIResponseFormatter.compose404Message(responseType, "404 - " + entity +" with id "
                        + resultPid + " not found.")).type(responseType).build();
            }


            return Response.status(Response.Status.OK).entity(APIResponseFormatter.createEntitiesResponse(request, entity, fullQuery, null, searchResults, false, responseType, false, false)).type(responseType).build();

        } catch (SearchServiceException sse) {
            logger.error("Fail to fetch "+ entity + " with id " + resultPid, sse);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(APIResponseFormatter.compose500Message(responseType, "Fail to fetch " + entity + " with id " + resultPid, sse.getMessage())).build();
        }
    }

    public boolean checkTypes(List<String> types) {
        for (RequestResponseHandler.Entity entity: RequestResponseHandler.Entity.values()){
            if (!types.contains(entity.getPlural())) {
                return false;
            }
        }

        return true;
    }

    private RequestResponseHandler.Entity identifyBasicEntityType(final List<String> types) {
        if (types != null && !types.isEmpty() && types.get(0) != null && !types.get(0).isEmpty()) {
            String type = types.get(0);

            if (type.equals(RequestResponseHandler.Entity.RESULT.getPlural())) {
                return RequestResponseHandler.Entity.RESULT;

            } else if (type.equals(RequestResponseHandler.Entity.PUBLICATION.getPlural()) || type.equals(RequestResponseHandler.Entity.DATASET.getPlural()) ||
                    type.equals(RequestResponseHandler.Entity.SOFTWARE.getPlural()) || type.equals(RequestResponseHandler.Entity.OTHER.getPlural())) {
                return RequestResponseHandler.Entity.RESULT;

            } else if (type.equals(RequestResponseHandler.Entity.PROJECT.getPlural())) {
                return RequestResponseHandler.Entity.PROJECT;

            } else if (type.equals(RequestResponseHandler.Entity.DATASOURCE.getPlural())) {
                return RequestResponseHandler.Entity.DATASOURCE;

            } else if (type.equals(RequestResponseHandler.Entity.ORGANIZATION.getPlural())) {
                return RequestResponseHandler.Entity.ORGANIZATION;
            }
        }
        return RequestResponseHandler.Entity.NONE;
    }

    /*
    public static void main(String[] args) {
        String json = " {\"result\":{\"xmlns:oaf\":\"http://namespace.openaire.eu/oaf\",\"xmlns:xsi\":\"http://www.w3.o" +
                "rg/2001/XMLSchema-instance\",\"header\":{\"dri:dateOfCollection\":\"2016-05-20T03:34:07Z\",\"dri:dateOfTransformation\":\"\",\"counters\":{\"counter_dedup\":{\"value\":2},\"counter_doi\":{\"value\":2},\"counter_authorship\":{\"va" +
                "lue\":1},\"counter_authorship_inferred\":{\"value\":1}},\"dri:objIdentifier\":\"dedup_wf_001::0f5604702b32f59d9789573113279e69\"},\"metadata\":{\"oaf:entity\":{\"oaf:result\":{\"datainfo\":{\"provenanceaction\":{\"schemename\"" +
                ":\"dnet:provenanceActions\",\"classname\":\"sysimport:dedup\",\"schemeid\":\"dnet:provenanceActions\",\"classid\":\"sysimport:dedup\"},\"trust\":0.9,\"inferenceprovenance\":\"dedup-similarity-result\",\"deletedbyinference\":fal" +
                "se,\"inferred\":true},\"rels\":{\"rel\":{\"to\":{\"content\":\"dedup_wf_001::884d8c367553a8de98ec90ece1b1be4a\",\"scheme\":\"dnet:person_result_relations\",\"class\":\"hasAuthor\",\"type\":\"person\"},\"provenanceaction\":\"sysimpor" +
                "t:crosswalk:repository\",\"trust\":0.9,\"inferenceprovenance\":\"\",\"ranking\":1,\"inferred\":true,\"fullname\":\"Krafft, Alexander\"}},\"subject\":[{\"content\":\"Clinic for Obstetrics\",\"schemename\":\"dnet:result_subject\",\"c" +
                "lassname\":\"keyword\",\"schemeid\":\"dnet:result_subject\",\"classid\":\"keyword\"},{\"content\":\"610 Medicine & health\",\"schemename\":\"dnet:result_subject\",\"classname\":\"keyword\",\"schemeid\":\"dnet:result_subject\",\"class" +
                "id\":\"keyword\"},{\"content\":\"News\",\"schemename\":\"dnet:result_subject\",\"classname\":\"keyword\",\"schemeid\":\"dnet:result_subject\",\"classid\":\"keyword\"}],\"collectedfrom\":[{\"id\":\"opendoar____::0efe32849d230d7f53049d" +
                "dc4a4b0c60\",\"name\":\"Zurich Open Repository and Archive\"},{\"id\":\"opendoar____::8b6dd7db9af49e67306feb59a8bdc52c\",\"name\":\"Europe PubMed Central\"}],\"children\":{\"result\":[{\"title\":{\"content\":\"Iron supplementat" +
                "ion in pregnancy\",\"schemename\":\"dnet:dataCite_title\",\"classname\":\"main title\",\"schemeid\":\"dnet:dataCite_title\",\"classid\":\"main title\"},\"resulttype\":{\"schemename\":\"dnet:result_typologies\",\"classname\":\"publi" +
                "cation\",\"schemeid\":\"dnet:result_typologies\",\"classid\":\"publication\"},\"dateofacceptance\":\"1981-09-01\",\"objidentifier\":\"od_______908::f27f9e68f0b9f510ebf572ba295344e1\"},{\"title\":{\"content\":\"Iron supplementat" +
                "ion in pregnancy\",\"schemename\":\"dnet:dataCite_title\",\"classname\":\"main title\",\"schemeid\":\"dnet:dataCite_title\",\"classid\":\"main title\"},\"resulttype\":{\"schemename\":\"dnet:result_typologies\",\"classname\":\"publi" +
                "cation\",\"schemeid\":\"dnet:result_typologies\",\"classid\":\"publication\"},\"dateofacceptance\":\"2013-01-01\",\"publisher\":\"BMJ Publishing Group\",\"objidentifier\":\"od_______885::0f5604702b32f59d9789573113279e69\"}],\"i" +
                "nstance\":[{\"webresource\":{\"url\":\"http://europepmc.org/articles/PMC2306130\"},\"id\":\"opendoar____::8b6dd7db9af49e67306feb59a8bdc52c\",\"licence\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"sche" +
                "meid\":\"dnet:access_modes\",\"classid\":\"OPEN\"},\"hostedby\":{\"id\":\"opendoar____::8b6dd7db9af49e67306feb59a8bdc52c\",\"name\":\"Europe PubMed Central\"},\"instancetype\":{\"schemename\":\"dnet:publication_resource\",\"class" +
                "name\":\"Article\",\"schemeid\":\"dnet:publication_resource\",\"classid\":\"0001\"}},{\"webresource\":{\"url\":\"http://www.zora.uzh.ch/87059/1/30_13_Krafft_2013bmj_f4399_full.pdf\"},\"id\":\"opendoar____::0efe32849d230d7f530" +
                "49ddc4a4b0c60\",\"licence\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"schemeid\":\"dnet:access_modes\",\"classid\":\"OPEN\"},\"hostedby\":{\"id\":\"opendoar____::0efe32849d230d7f53049ddc4a4b0c60\",\"name" +
                "\":\"Zurich Open Repository and Archive\"},\"instancetype\":{\"schemename\":\"dnet:publication_resource\",\"classname\":\"Article\",\"schemeid\":\"dnet:publication_resource\",\"classid\":\"0001\"}}]},\"pid\":[{\"content\":\"10.5167" +
                "/uzh-87059\",\"schemename\":\"dnet:pid_types\",\"classname\":\"doi\",\"schemeid\":\"dnet:pid_types\",\"classid\":\"doi\"},{\"content\":\"PMC2306130\",\"schemename\":\"dnet:pid_types\",\"classname\":\"pmc\",\"schemeid\":\"dnet:pid_types\"," +
                "\"classid\":\"pmc\"},{\"content\":\"10.1136/bmj.f4399\",\"schemename\":\"dnet:pid_types\",\"classname\":\"doi\",\"schemeid\":\"dnet:pid_types\",\"classid\":\"doi\"}],\"resourcetype\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"c" +
                "lassid\":\"\"},\"format\":\"\",\"dateofacceptance\":\"2013-01-01\",\"embargoenddate\":\"\",\"version\":\"\",\"size\":\"\",\"publisher\":\"BMJ Publishing Group\",\"country\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"classid\":\"\"},\"" +
                "metadataversionnumber\":\"\",\"title\":{\"content\":\"Iron supplementation in pregnancy\",\"schemename\":\"dnet:dataCite_title\",\"classname\":\"main title\",\"schemeid\":\"dnet:dataCite_title\",\"classid\":\"main title\"},\"result" +
                "type\":{\"schemename\":\"dnet:result_typologies\",\"classname\":\"publication\",\"schemeid\":\"dnet:result_typologies\",\"classid\":\"publication\"},\"source\":\"Krafft, Alexander (2013). Iron supplementation in pregnancy. Br" +
                "itish Medical Journal, 347:f4399.\",\"contributor\":\"\",\"description\":\"\",\"originalId\":[\"oai:europepmc.org:1052305\",\"oai:www.zora.uzh.ch:87059\"],\"lastmetadataupdate\":\"\",\"relevantdate\":{\"schemename\":\"\",\"classnam" +
                "e\":\"\",\"schemeid\":\"\",\"classid\":\"\"},\"device\":\"\",\"bestlicense\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"schemeid\":\"dnet:access_modes\",\"classid\":\"OPEN\"},\"language\":{\"schemename\":\"dnet:langu" +
                "ages\",\"classname\":\"English\",\"schemeid\":\"dnet:languages\",\"classid\":\"eng\"},\"storagedate\":\"\",\"fulltext\":\"\"},\"xsi:schemaLocation\":\"http://namespace.openaire.eu/oaf https://www.openaire.eu/schema/0.2/oaf-0.2.xs" +
                "d\"}},\"xmlns:dri\":\"http://www.driver-repository.eu/namespace/dri\"}}";

        String s = "{\"result\":{\"xmlns:oaf\":\"http://namespace.openaire.eu/oaf\",\"xmlns:xsi\":\"http://www.w3.org/2001" +
                "/XMLSchema-instance\",\"header\":{\"dri:dateOfCollection\":\"2014-04-10T13:35:33Z\",\"dri:dateOfTransformation\":\"\",\"counters\":{\"counter_dedup\":{\"value\":2},\"counter_authorship\":{\"value\":1},\"counter_authorship_infer" +
                "red\":{\"value\":1}},\"dri:objIdentifier\":\"dedup_wf_001::4d934888dee33a62dece6fa36192c710\"},\"metadata\":{\"oaf:entity\":{\"oaf:result\":{\"datainfo\":{\"provenanceaction\":{\"schemename\":\"dnet:provenanceActions\",\"classn" +
                "ame\":\"sysimport:dedup\",\"schemeid\":\"dnet:provenanceActions\",\"classid\":\"sysimport:dedup\"},\"trust\":0.9,\"inferenceprovenance\":\"dedup-similarity-result\",\"deletedbyinference\":false,\"inferred\":true},\"rels\":{\"rel\"" +
                ":{\"to\":{\"content\":\"od______1232::d4f9cea02c7762da24cd71ea0b9791de\",\"scheme\":\"dnet:person_result_relations\",\"class\":\"hasAuthor\",\"type\":\"person\"},\"provenanceaction\":\"sysimport:crosswalk:repository\",\"trust\":0" +
                ".9,\"inferenceprovenance\":\"\",\"ranking\":1,\"inferred\":true,\"fullname\":\"Anonymous\"}},\"subject\":{\"content\":\"Señoríos-Legislación\",\"schemename\":\"dnet:result_subject\",\"classname\":\"keyword\",\"schemeid\":\"dnet:result" +
                "_subject\",\"classid\":\"keyword\"},\"collectedfrom\":[{\"id\":\"opendoar____::e53a0a2978c28872a4505bdb51db06dc\",\"name\":\"Biblioteca Virtual del Patrimonio Bibliográfico (Virtual Library of Bibliographical Heritage)\"" +
                "},{\"id\":\"opendoar____::7aee26c309def8c5a2a076eb250b8f36\",\"name\":\"Biblioteca Virtual de Aragón\"}],\"children\":{\"result\":[{\"title\":{\"content\":\"Discurso sobre los señoríos : necesidad y justicia del decreto de" +
                " Cortes de 6 de Agosto de 1811 y ventaja que debe causar a la Nación\",\"schemename\":\"dnet:dataCite_title\",\"classname\":\"main title\",\"schemeid\":\"dnet:dataCite_title\",\"classid\":\"main title\"},\"resulttype\":{\"sch" +
                "emename\":\"dnet:result_typologies\",\"classname\":\"publication\",\"schemeid\":\"dnet:result_typologies\",\"classid\":\"publication\"},\"dateofacceptance\":\"1820-01-01\",\"objidentifier\":\"od______2719::bc2326a31763719dc5ebd" +
                "38f16a25404\"},{\"title\":{\"content\":\"Discurso sobre los señoríos : necesidad y justicia del decreto de Cortes de 6 de Agosto de 1811 y ventaja que debe causar a la Nación\",\"schemename\":\"dnet:dataCite_title\"," +
                "\"classname\":\"main title\",\"schemeid\":\"dnet:dataCite_title\",\"classid\":\"main title\"},\"resulttype\":{\"schemename\":\"dnet:result_typologies\",\"classname\":\"publication\",\"schemeid\":\"dnet:result_typologies\",\"classid\"" +
                ":\"publication\"},\"dateofacceptance\":\"1820-01-01\",\"objidentifier\":\"od______1232::4d934888dee33a62dece6fa36192c710\"}],\"instance\":[{\"webresource\":[{\"url\":\"http://bvpb.mcu.es/es/consulta/registro.cmd?id=403864\"" +
                "},{\"url\":\"http://bvpb.mcu.es/es/catalogo_imagenes/grupo.cmd?path=2755\"}],\"id\":\"opendoar____::e53a0a2978c28872a4505bdb51db06dc\",\"licence\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"schemei" +
                "d\":\"dnet:access_modes\",\"classid\":\"OPEN\"},\"hostedby\":{\"id\":\"opendoar____::e53a0a2978c28872a4505bdb51db06dc\",\"name\":\"Biblioteca Virtual del Patrimonio Bibliográfico (Virtual Library of Bibliographical Herita" +
                "ge)\"},\"instancetype\":{\"schemename\":\"dnet:publication_resource\",\"classname\":\"Book\",\"schemeid\":\"dnet:publication_resource\",\"classid\":\"0002\"}},{\"webresource\":{\"url\":\"http://bibliotecavirtual.aragon.es/bva/i18" +
                "n/consulta/registro.cmd?id=2878\"},\"id\":\"opendoar____::7aee26c309def8c5a2a076eb250b8f36\",\"licence\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"schemeid\":\"dnet:access_modes\",\"classid\":\"OPEN\"" +
                "},\"hostedby\":{\"id\":\"opendoar____::7aee26c309def8c5a2a076eb250b8f36\",\"name\":\"Biblioteca Virtual de Aragón\"},\"instancetype\":{\"schemename\":\"dnet:publication_resource\",\"classname\":\"Book\",\"schemeid\":\"dnet:publi" +
                "cation_resource\",\"classid\":\"0002\"}}]},\"pid\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"classid\":\"\"},\"resourcetype\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"classid\":\"\"},\"format\":\"\",\"dateofaccepta" +
                "nce\":\"1820-01-01\",\"embargoenddate\":\"\",\"version\":\"\",\"size\":\"\",\"country\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"classid\":\"\"},\"publisher\":\"\",\"metadataversionnumber\":\"\",\"title\":{\"content\":\"Discurso sob" +
                "re los señoríos : necesidad y justicia del decreto de Cortes de 6 de Agosto de 1811 y ventaja que debe causar a la Nación\",\"schemename\":\"dnet:dataCite_title\",\"classname\":\"main title\",\"schemeid\":\"dnet:dataC" +
                "ite_title\",\"classid\":\"main title\"},\"resulttype\":{\"schemename\":\"dnet:result_typologies\",\"classname\":\"publication\",\"schemeid\":\"dnet:result_typologies\",\"classid\":\"publication\"},\"source\":\"\",\"contributor\":\"Impr" +
                "enta de Heras, imp.\",\"description\":\"\",\"originalId\":[\"oai:bvpb.mcu.es:403864\",\"oai:bibliotecavirtual.aragon.es:2878\"],\"lastmetadataupdate\":\"\",\"relevantdate\":{\"schemename\":\"\",\"classname\":\"\",\"schemeid\":\"\",\"cl" +
                "assid\":\"\"},\"device\":\"\",\"bestlicense\":{\"schemename\":\"dnet:access_modes\",\"classname\":\"Open Access\",\"schemeid\":\"dnet:access_modes\",\"classid\":\"OPEN\"},\"language\":{\"schemename\":\"dnet:languages\",\"classname\":\"Span" +
                "ish\",\"schemeid\":\"dnet:languages\",\"classid\":\"esl/spa\"},\"storagedate\":\"\",\"fulltext\":\"\"},\"xsi:schemaLocation\":\"http://namespace.openaire.eu/oaf https://www.openaire.eu/schema/0.2/oaf-0.2.xsd\"}},\"xmlns:dri\":\"h" +
                "ttp://www.driver-repository.eu/namespace/dri\"}}\n";

        JSONObject finalOject = new JSONObject();
        finalOject.put("dri:objIdentifier", new JSONObject(json).getJSONObject("result").getJSONObject("header").getString("dri:objIdentifier"));
        finalOject.put("content", new JSONObject(new JSONObject(json).getJSONObject("result").getJSONObject("metadata").getJSONObject("oaf:entity").getJSONObject("oaf:result"),
                new String[]{"title", "description", "language", "bestlicense", "pid", "rels"}));

        System.out.println(finalOject);

        JSONObject jsonObject = new JSONObject(s).getJSONObject("result").getJSONObject("metadata").getJSONObject("oaf:entity").getJSONObject("oaf:result");
        System.out.println("title --> " + jsonObject.getJSONObject("title").getString("content"));
        System.out.println("pid --> " + jsonObject.getJSONArray("pid"));

        JSONObject newJsonObject1 = new JSONObject(jsonObject, new String[]{"title", "description", "language", "bestlicense", "pid", "rels"});

    }
*/
    /*

    public static void main(String[] args) throws IOException, CQLParseException {
        CqlTranslator translator = new CqlTranslatorImpl();
        System.out.println(translator.toLucene("relfundinglevel0_id = ec__________::EC::FP7"));
    }
    */

}