package eu.dnetlib.data.search.utils.cql;

import eu.dnetlib.data.search.utils.vocabulary.VocabularyManager;
import eu.dnetlib.domain.enabling.Vocabulary;
import org.apache.log4j.Logger;

import javax.servlet.http.HttpServletRequest;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * Created by kiatrop on 10/7/2014.
 */
public class ParameterQueryEnhancer {

    private static final Logger logger = Logger.getLogger(ParameterQueryEnhancer.class);

    /**
     * Enhance the given CQL query with FP7 specific index fields
     */
    public static void enhanceQueryWithFundingLevelParams(StringBuilder queryBuilder, HttpServletRequest request, VocabularyManager vocabularyManager, boolean isModelSygma) {
        logger.debug("enhance funding " + isModelSygma);
        
        String funder = request.getParameter("funder");
        String fundingStream = request.getParameter("fundingStream");
        String FP7scientificArea = request.getParameter("FP7scientificArea");

        if(isModelSygma) {
            if (funder != null) {
                if(!funder.equalsIgnoreCase("WT")) {
                    Vocabulary sygmaFundersVocabulary = vocabularyManager.getVocabulary("sygma_funders", Locale.ROOT);
                    addExactQueryTerm("relfundinglevel0_id", devocabularizedTerm(funder.toUpperCase(), sygmaFundersVocabulary), queryBuilder);
                } else {
                    Vocabulary funderVocabulary = vocabularyManager.getVocabulary("funders_simple", Locale.ROOT);
                    addExactQueryTerm("relfunderid", devocabularizedTerm(funder.toUpperCase(), funderVocabulary), queryBuilder);
                }
            }

            if (fundingStream != null && !fundingStream.trim().isEmpty()) {
                Vocabulary relfundinglevel1Vocabulary = vocabularyManager.getVocabulary("programmes_simple", Locale.ROOT);
                addVocabularizedQueryTerm("relfundinglevel1_id", fundingStream.toUpperCase(), relfundinglevel1Vocabulary, queryBuilder);
            }

            if (FP7scientificArea != null && !FP7scientificArea.trim().isEmpty()) {
                Vocabulary relfundinglevel2Vocabulary = vocabularyManager.getVocabulary("areas", Locale.ROOT);
                addVocabularizedQueryTerm("relfundinglevel2_id", FP7scientificArea, relfundinglevel2Vocabulary, queryBuilder);
            }

        } else {

            if (funder != null) {
                //Vocabulary funderVocabulary = vocabularyManager.getVocabulary("funders_simple", Locale.ROOT);
                addExactQueryTerm("relfundershortname", funder.toUpperCase(), queryBuilder);
            }

            if (fundingStream != null && !fundingStream.trim().isEmpty()) {
                if(fundingStream.equalsIgnoreCase("FP7") || fundingStream.equalsIgnoreCase("H2020")) {
                    addORQueryTermExact(new String[]{"relfundinglevel0_name", "relfundinglevel1_name", "relfundinglevel2_name"}, "\"" + fundingStream.toUpperCase() + "\"", queryBuilder);
                } else {
                    addORQueryTermExact(new String[]{"relfundinglevel0_name", "relfundinglevel1_name", "relfundinglevel2_name"}, "\"" + fundingStream + "\"", queryBuilder);
                }
            }

            if (FP7scientificArea != null && !FP7scientificArea.trim().isEmpty()) {
                //Vocabulary relfundinglevel2Vocabulary = vocabularyManager.getVocabulary("areas", Locale.ROOT);
                queryBuilder.append(" and (relfundershortname exact \"EC\")").append(" and (relfundinglevel2_name exact \"").append(FP7scientificArea).append("\")");
            }
        }

    }

    public static void enhanceProjectQueryWithFundingLevelParams(StringBuilder queryBuilder, HttpServletRequest request, VocabularyManager vocabularyManager) {
        String funder = request.getParameter("funder");
        String fundingStream = request.getParameter("fundingStream");
        String FP7scientificArea = request.getParameter("FP7scientificArea");

        if (funder != null) {
            //addExactQueryTerm("fundinglevel0_name", funder.toUpperCase(), queryBuilder);
            //Vocabulary funderVocabulary = vocabularyManager.getVocabulary("funders_simple", Locale.ROOT);
            addExactQueryTerm("fundershortname", funder.toUpperCase(), queryBuilder);
        }

        if (fundingStream != null && !fundingStream.trim().isEmpty()) {
            //addExactQueryTerm("fundinglevel1_name", fundingStream.toUpperCase(), queryBuilder);
            if(fundingStream.equalsIgnoreCase("FP7") || fundingStream.equalsIgnoreCase("H2020")) {
                addORQueryTermExact(new String[]{"fundinglevel0_name", "fundinglevel1_name", "fundinglevel2_name"}, "\"" + fundingStream.toUpperCase() + "\"", queryBuilder);
            } else {
                addORQueryTermExact(new String[]{"fundinglevel0_name", "fundinglevel1_name", "fundinglevel2_name"}, "\"" + fundingStream + "\"", queryBuilder);
            }
        }

        if (FP7scientificArea != null && !FP7scientificArea.trim().isEmpty()) {
            //Vocabulary relfundinglevel2Vocabulary = vocabularyManager.getVocabulary("areas", Locale.ROOT);
            queryBuilder.append("and (fundershortname exact \"EC\")").append(" and (fundinglevel2_name exact \"").append(FP7scientificArea).append("\")");
        }
    }

    public static void enhanceQueryWithFundingParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String hasECFunding = request.getParameter("hasECFunding");
        String hasWTFunding = request.getParameter("hasWTFunding");

        addBooleanQueryTerm("contextid", hasECFunding, "EC", queryBuilder);
        addBooleanQueryTerm("contextid", hasWTFunding, "WT", queryBuilder);
    }

    public static void enhanceQueryWithCommunityParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String community = request.getParameter("community");

        if (community!= null) {
            addExactQueryTerm("communityId", community, queryBuilder);
        }

    }

    public static void enhanceQueryWithProjectFundingParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String hasECFunding = request.getParameter("hasECFunding");
        String hasWTFunding = request.getParameter("hasWTFunding");

        addBooleanQueryTerm("funderid", hasECFunding, "ec__________::EC", queryBuilder);
        addBooleanQueryTerm("funderid", hasWTFunding, "wt__________::WT", queryBuilder);

    }

    public static void enhanceQueryWithRelProjectParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String hasProject = request.getParameter("hasProject");
        String projectID = request.getParameter("projectID");
        String FP7ProjectID = request.getParameter("FP7ProjectID");

        if (hasProject != null && !hasProject.isEmpty()) {
            if (hasProject.equals("true")) {
                addEqualQueryTerm("relprojectid", "*", queryBuilder);
            } else {
                addNotEqualQueryTerm("relprojectid", "*", queryBuilder);
            }
        }

        if (FP7ProjectID != null && !FP7ProjectID.trim().isEmpty()) {
            addExactQueryTerm("relprojectcode", FP7ProjectID, queryBuilder);
            addExactQueryTerm(" relfundinglevel0_id", "ec__________::EC::FP7", queryBuilder);
        }

        if (projectID != null && !projectID.trim().isEmpty()) {
            queryBuilder.append(" and (relprojectcode exact \"").append(projectID).append("\")");
        }
    }

    public static void enhanceQueryWithAccessRights(StringBuilder queryBuilder, HttpServletRequest request) {
        String oa = request.getParameter("OA");
        addBooleanQueryTerm("resultbestaccessright", oa, "Open Access", queryBuilder);
    }

    public static void enhanceQueryWithDate(StringBuilder queryBuilder, HttpServletRequest request) throws IllegalArgumentException{
        String fromDateAccepted = request.getParameter("fromDateAccepted");
        String toDateAccepted = request.getParameter("toDateAccepted");
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setLenient(false);

        if (toDateAccepted != null && !toDateAccepted.isEmpty() && (fromDateAccepted == null || fromDateAccepted.isEmpty()) ) {
            if(!checkDate(toDateAccepted, simpleDateFormat)){
                //logger.debug("Format is " + !checkDate(toDateAccepted, simpleDateFormat));
                throw new IllegalArgumentException("toDateAccepted date must be formatted as YYYY-MM-DD");
            }
            fromDateAccepted = "*";
        }

        if (fromDateAccepted != null && !fromDateAccepted.isEmpty() && (toDateAccepted == null || toDateAccepted.isEmpty()) ) {
            if (!checkDate(fromDateAccepted, simpleDateFormat)){
                //logger.debug("Format from is " + fromDateAccepted != null && !fromDateAccepted.isEmpty());
                throw new IllegalArgumentException("fromDateAccepted date must be formatted as YYYY-MM-DD");
            }
            toDateAccepted = simpleDateFormat.format(new Date());
        }

        if (toDateAccepted != null && !toDateAccepted.isEmpty() && fromDateAccepted != null && !fromDateAccepted.isEmpty()) {
            queryBuilder.append(" and (resultdateofacceptance within \"").append(fromDateAccepted).append(" ").append(toDateAccepted).append("\")");
        }
    }


    public static void enhanceQueryWithYearParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String startYear = request.getParameter("startYear");
        String endYear = request.getParameter("endYear");

        int sYear = -1;
        if (startYear != null && !startYear.isEmpty()) {
            try {
                sYear = Integer.parseInt(startYear);

            } catch(NumberFormatException e) {
                throw new IllegalArgumentException("startYear parameter must be a numeric value.", e);
            }
        }

        int eYear = -1;
        if (endYear !=null && !endYear.isEmpty()) {
            try {
                eYear = Integer.parseInt(endYear);

            } catch(NumberFormatException e) {
                throw new IllegalArgumentException("endYear parameter must be a numeric value.",e);
            }
        }

        if (eYear != -1 && sYear != -1 && eYear < sYear) {
            throw new IllegalArgumentException("endYear must be greater than startYear.");
        }

        addExactQueryTerm("projectstartyear", startYear, queryBuilder);
        addExactQueryTerm("projectendyear", endYear, queryBuilder);
    }

    public static void enhanceQueryWithResultsSortParameters(StringBuilder queryBuilder, HttpServletRequest request) {
        String sortByParameter = request.getParameter("sortBy");

        if (sortByParameter != null) {
            String[] sortParams = sortByParameter.split(",");

            if (sortParams.length != 2) {
                throw new IllegalArgumentException("Invalid sort paremeter. 'sortBy' parameter format is <fieldName>[,ascending|,descending].");
            }

            String sortByField = sortParams[0];
            String order = sortParams[1];

            if (!checkPublicationSortParameterFields(sortByField)){
                throw new IllegalArgumentException("'" + sortByField + "' is not a sortable field.");
            }

            if (!checkOrder(order)) {
                throw new IllegalArgumentException("'" + order + "' is not a valid ordering. Please use one of {ascending, descending}");
            }

            addSortParameter(sortByField, order, queryBuilder);
        }
    }

    public static void enhanceQueryWithProjectSortParameters(StringBuilder queryBuilder, HttpServletRequest request) {
        String sortByParameter = request.getParameter("sortBy");

        if (sortByParameter != null) {
            String[] sortParams = sortByParameter.split(",");

            if (sortParams.length != 2) {
                throw new IllegalArgumentException("Invalid sort paremeter. 'sortBy' parameter format is <fieldName>[,ascending|,descending].");
            }

            String sortByField = sortParams[0];
            String order = sortParams[1];

            if (!checkProjectSortParameterFields(sortByField)){
                throw new IllegalArgumentException("'" + sortByField + "' is not a sortable field.");
            }

            if (!checkOrder(order)) {
                throw new IllegalArgumentException("'" + order + "' is not a valid ordering. Please use one of {ascending, descending}");
            }

            addSortParameter(sortByField, order, queryBuilder);
        }
    }

    private static boolean checkPublicationSortParameterFields(String sortField) {
        if ((sortField != null) && (!sortField.isEmpty()) &&
                sortField.matches("dateofcollection|resultstoragedate|resultembargoenddate|resultembargoendyear|" +
                "resulttypeid|resulttypename|resultlanguageid|resultlanguagename|resultbestaccessright|" +
                "resultbestlicenseid|resultdateofacceptance|resultacceptanceyear")) {
            return true;
        }
        return false;
    }

    private static boolean checkProjectSortParameterFields(String sortField) {
        if ((sortField != null) && (!sortField.isEmpty()) &&
                sortField.matches("dateofcollection|projectstartdate|projectstartyear|" +
                        "projectenddate|projectendyear|projectcallidentifier|projectduration|" +
                        "projectecsc39|projectcontracttypeid|projectcontracttypename")) {
            return true;
        }
        return false;
    }

    private static boolean checkOrder(String order) {
        if (order.matches("ascending|descending")) {
            return true;
        }
        return false;
    }

    public static boolean checkDate(String date, SimpleDateFormat simpleDateFormat) {
        try {
            simpleDateFormat.parse(date);

        } catch (ParseException pe) {
            logger.warn("Wrong date format.", pe);
            return false;
        }

        return true;
    }

    /**
     * Enhance the given CQL query with OpenAIRE specific ids
     * @param queryBuilder The string builder of the query
     * @param request the HTTP request
     */
    public static void enhanceQueryWithOpenAIREIds(StringBuilder queryBuilder, HttpServletRequest request) {
        String[] openairePublicationIDs = request.getParameterValues("openairePublicationID");
        String[] openaireDatasetIDs = request.getParameterValues("openaireDatasetID");
        String[] openaireSoftwareIDs = request.getParameterValues("openaireSoftwareID");
        String[] openaireOtherIDs = request.getParameterValues("openaireOtherID");
        String[] openaireProviderIDs  = request.getParameterValues("openaireProviderID");
        String[] openaireProjectIDs  = request.getParameterValues("openaireProjectID");

        enhanceQueryWithIds("objidentifier", openairePublicationIDs, queryBuilder);
        enhanceQueryWithIds("objidentifier", openaireDatasetIDs, queryBuilder);
        enhanceQueryWithIds("objidentifier", openaireSoftwareIDs, queryBuilder);
        enhanceQueryWithIds("objidentifier", openaireOtherIDs, queryBuilder);
        enhanceQueryWithIds("resulthostingdatasourceid", openaireProviderIDs, queryBuilder);
        enhanceQueryWithIds("relprojectid", openaireProjectIDs, queryBuilder);
    }

    public static void enhanceQueryWithMetadataKeywords(StringBuilder queryBuilder, HttpServletRequest request) {
        String keywords = request.getParameter("keywords");
        String title = request.getParameter("title");
        String author = request.getParameter("author");
        String country = request.getParameter("country");

        addMetadataQueryTerm(null, keywords, queryBuilder);
        addMetadataQueryTerm("resulttitle", title, queryBuilder);
        addMetadataQueryTerm("resultauthor", author, queryBuilder);
        addMetadataQueryTerm("country", country, queryBuilder);
    }

    public static void enhanceQueryWithProjectMetadataKeywords(StringBuilder queryBuilder, HttpServletRequest request) {
        String keywords = request.getParameter("keywords");
        String acronym = request.getParameter("acronym");
        String name = request.getParameter("name");
        String grantID = request.getParameter("grantID");
        String callID = request.getParameter("callID");

        addMetadataQueryTerm(null, keywords, queryBuilder);
        addMetadataQueryTerm("projectacronym", acronym, queryBuilder);
        addEqualQueryTerm("projecttitle", name, queryBuilder);
        addExactQueryTerm("projectcode", grantID, queryBuilder);
        addExactQueryTerm("projectcallidentifier", callID, queryBuilder);
    }

    public static void enhanceQueryWithParticipantsInfoParams(StringBuilder queryBuilder, HttpServletRequest request) {
        String[] participantCountries = request.getParameterValues("participantCountries");
        String participantAcronyms = request.getParameter("participantAcronyms");
        String openaireParticipantID = request.getParameter("openaireParticipantID");


        if(participantCountries != null) {
            enhanceQueryWithCommaSeparatedValues("relorganizationcountryid", participantCountries, queryBuilder);
        }

        addORQueryTerm("relorganizationname", "relorganizationshortname", participantAcronyms, queryBuilder);

        if(openaireParticipantID != null) {
            addEqualQueryTerm("relorganizationid", openaireParticipantID, queryBuilder);
        }

    }

    public static void enhanceQueryWithSC39Params(StringBuilder queryBuilder, HttpServletRequest request) {
        String sc39 = request.getParameter("sc39");
        addBooleanQueryTerm("projectecsc39", sc39, queryBuilder);
    }




    public static void enhanceQueryWithDoi(StringBuilder queryBuilder, HttpServletRequest request) {
        String[] dois = request.getParameterValues("doi");

        if (dois != null && !(dois.length==0)) {
            queryBuilder.append(" and ");
            for (int i = 0; i < dois.length; i++) {
                String[] commaSeparated = dois[i].split(",");
                for (int j = 0; j < commaSeparated.length; j++) {
                    queryBuilder.append("(pidclassid exact \"doi\" and pid exact \"").append(commaSeparated[j]).append("\")");
                    if (i < dois.length-1 || j < commaSeparated.length-1 ) {
                        queryBuilder.append(" or ");
                    }
                }
            }
        }
    }

    public static void enhanceQueryWithOriginalId(StringBuilder queryBuilder, HttpServletRequest request) {
        String[] originalIds = request.getParameterValues("originalId");

        if (originalIds != null && !(originalIds.length==0)) {
            queryBuilder.append(" and ");
            for (int i = 0; i < originalIds.length; i++) {
                String[] commaSeparated = originalIds[i].split(",");
                for (int j = 0; j < commaSeparated.length; j++) {
                    queryBuilder.append("(originalId exact \"").append(commaSeparated[j]).append("\")");
                    if (i < originalIds.length-1 || j < commaSeparated.length-1 ) {
                        queryBuilder.append(" or ");
                    }
                }
            }
        }
    }

    public static void enhanceProjectQueryWithOpenAIREIds(StringBuilder queryBuilder, HttpServletRequest request) {
        String openaireProjectID  = request.getParameter("openaireProjectID");

        if (openaireProjectID != null) {
            addExactQueryTerm("objIdentifier", openaireProjectID, queryBuilder);
        }
    }

    public static void enhanceQueryWithIds(String idIndexFieldName, String[] ids, StringBuilder queryBuilder) {
        if (ids != null && !(ids.length==0)) {
            queryBuilder.append(" and ");
            for (int i = 0; i < ids.length; i++) {
                String[] commaSeparated = ids[i].split(",");
                for (int j = 0; j < commaSeparated.length; j++) {
                    queryBuilder.append("(" + idIndexFieldName + " exact \"").append(commaSeparated[j]).append("\")");
                    if (i < ids.length-1 || j < commaSeparated.length-1 ) {
                        queryBuilder.append(" or ");
                    }
                }
            }
        }
    }

    public static void enhanceQueryWithCommaSeparatedValues(String indexFieldName, String[] fieldValues, StringBuilder queryBuilder) {
        if (fieldValues != null && !(fieldValues.length==0)) {
            queryBuilder.append(" and ");
            for (int i = 0; i < fieldValues.length; i++) {
                String[] commaSeparated = fieldValues[i].split(",");
                for (int j = 0; j < commaSeparated.length; j++) {
                    queryBuilder.append("(").append(indexFieldName).append(" exact \"").append(commaSeparated[j].toUpperCase()).append("\")");
                    if (i < fieldValues.length-1 || j < commaSeparated.length-1 ) {
                        queryBuilder.append(" and ");
                    }
                }
            }
        }
    }

    public static void addMetadataQueryTerm(String indexFieldName, String fieldValue, StringBuilder queryBuilder) {
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            if(indexFieldName != null) {
                for (String term: fieldValue.trim().split(" ")){
                    addEqualQueryTerm(indexFieldName, term, queryBuilder);
                }
            } else {
                queryBuilder.append(" and ( " );
                String[] keywords = fieldValue.trim().split(" ");
                for (int i = 0; i < keywords.length; i++) {
                    if (i == keywords.length -1) {
                        queryBuilder.append(keywords[i]);
                    } else {
                        queryBuilder.append(keywords[i]).append(" and ");
                    }
                }
                queryBuilder.append(")" );
            }
        }
    }

    public static void addORQueryTerm(String indexFieldName1, String indexFieldName2, String fieldValue, StringBuilder queryBuilder) {
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            for (String term: fieldValue.trim().split(" ")) {
                queryBuilder.append(" and (" + indexFieldName1 + " = " + term + " or  " + indexFieldName2 + " = " + term + ")");
            }
        }
    }

    public static void addORQueryTerm(String[] indexFields, String fieldValue, StringBuilder queryBuilder) {
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            queryBuilder.append(" and ( ");
            for (int i = 0; i < indexFields.length; i++) {
                for (String term : fieldValue.trim().split(",(?=(?:[^\"]*\"[^\"]*\")*[^\"]*$)", -1)) {
                    //logger.info("term: " + term);
                    queryBuilder.append(" (" + indexFields[i] + " = " + term + ")");
                }

                if (i == indexFields.length-1) {
                    queryBuilder.append(")");

                } else {
                    queryBuilder.append(" or ");
                }

            }
        }
    }

    public static void addORQueryTermExact(String[] indexFields, String fieldValue, StringBuilder queryBuilder) {
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            queryBuilder.append(" and ( ");
            for (int i = 0; i < indexFields.length; i++) {
                for (String term : fieldValue.trim().split(",(?=(?:[^\"]*\"[^\"]*\")*[^\"]*$)", -1)) {
                    //logger.info("term: " + term);
                    queryBuilder.append(" (" + indexFields[i] + " exact " + term + ")");
                }

                if (i == indexFields.length-1) {
                    queryBuilder.append(")");

                } else {
                    queryBuilder.append(" or ");
                }

            }
        }
    }

    private static void addBooleanQueryTerm(String indexFieldName, String requestValue, StringBuilder queryBuilder) {
        if (requestValue != null && !requestValue.trim().isEmpty()) {
            addExactQueryTerm(indexFieldName, requestValue, queryBuilder);
        }
    }

    public static void addBooleanQueryTerm(String indexFieldName, String requestValue, String fieldValue, StringBuilder queryBuilder){
        if (requestValue != null && !requestValue.trim().isEmpty()) {
            if (requestValue.trim().equals("true")) {
                addExactQueryTerm(indexFieldName, fieldValue, queryBuilder);
            } else {
                addDifferentQueryTerm(indexFieldName, fieldValue, queryBuilder);
            }
        }
    }

    public static void addExactQueryTerm(String indexFieldName, String fieldValue, StringBuilder queryBuilder){
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            CQLQueryBuilder.appendFieldQuotedTerm(queryBuilder, CQLQueryBuilder.Operator.AND, indexFieldName, CQLQueryBuilder.Operator.EXACT, fieldValue);
        }
    }

    public static void addEqualQueryTerm(String indexFieldName, String fieldValue, StringBuilder queryBuilder){
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            if (fieldValue.startsWith("\"") && fieldValue.endsWith("\"")) {
                CQLQueryBuilder.appendFieldTerm(queryBuilder, CQLQueryBuilder.Operator.AND, indexFieldName, CQLQueryBuilder.Operator.EQUAL, fieldValue);
            } else {
                CQLQueryBuilder.appendFieldQuotedTerm(queryBuilder, CQLQueryBuilder.Operator.AND, indexFieldName, CQLQueryBuilder.Operator.EQUAL, fieldValue);
            }
        }




    }

    public static void addNotEqualQueryTerm(String indexFieldName, String fieldValue, StringBuilder queryBuilder){
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            CQLQueryBuilder.appendFieldQuotedTerm(queryBuilder, CQLQueryBuilder.Operator.NOT, indexFieldName, CQLQueryBuilder.Operator.EQUAL, fieldValue);
        }
    }

    public static void addDifferentQueryTerm(String indexFieldName, String fieldValue, StringBuilder queryBuilder){
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            CQLQueryBuilder.appendFieldQuotedTerm(queryBuilder, CQLQueryBuilder.Operator.AND, indexFieldName, CQLQueryBuilder.Operator.NOTEQUAL, fieldValue);
        }
    }

    public static void addVocabularizedQueryTerm(String indexFieldName, String fieldValue, Vocabulary vocabulary, StringBuilder queryBuilder){
        if (fieldValue != null && !fieldValue.trim().isEmpty()) {
            addExactQueryTerm(indexFieldName, devocabularizedTerm(fieldValue, vocabulary), queryBuilder);
        }
    }

    private static void addSortParameter(String indexField, String order, StringBuilder queryBuilder) {
        queryBuilder.append(" sortBy " + indexField + "/sort." + order);
    }

    /**
     * Returns the encoding of the given value. If the encoding does not exist
     * in the given vocabulary the method returns the original value.|
     * @param value the value to be encoded
     * @param vocabulary the vocabulary containing the encoding - value mapping
     * @return the encoding of the given value taken from the given vocabulary
     */
    public static String devocabularizedTerm(String value, Vocabulary vocabulary) {
        if (vocabulary != null) {
            String term = vocabulary.getEncoding(value);
            if (term == null) {
                return value;
            }
            return term;
        }
        return value;
    }
}
