<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">

    <!-- Turn off auto-insertion of <?xml> tag and set indenting on -->
    <xsl:output method="text" encoding="utf-8" indent="yes"/>

    <!-- strip whitespace from whitespace-only nodes -->
    <xsl:strip-space elements="*"/>

    <!-- create a key for every element in the document using its name -->
    <xsl:key name="names" match="*" use="concat(generate-id(..),'/',name())"/>

    <!-- start with the root element -->
    <xsl:template match="/">
        <!-- first element needs brackets around it as template does not do that -->
        <xsl:text>{ </xsl:text>
        <!-- call the template for elements using one unique name at a time -->
        <xsl:apply-templates select="*[generate-id(.) = generate-id(key('names', concat(generate-id(..),'/',name()))[1])]" >
            <xsl:sort select="name()" data-type="number"/>
        </xsl:apply-templates>
        <xsl:text> }</xsl:text>
    </xsl:template>

    <xsl:template name="escape-string">
        <xsl:param name="s"/>
        <xsl:text>"</xsl:text>
        <xsl:call-template name="escape-bs-string">
            <xsl:with-param name="s" select="$s"/>
        </xsl:call-template>
        <xsl:text>"</xsl:text>
    </xsl:template>

    <xsl:template name="escape-bs-string">
        <xsl:param name="s"/>
        <xsl:choose>
            <xsl:when test="contains($s,'\')">
                <xsl:call-template name="escape-quot-string">
                    <xsl:with-param name="s" select="concat(substring-before($s,'\'),'\\')"/>
                </xsl:call-template>
                <xsl:call-template name="escape-bs-string">
                    <xsl:with-param name="s" select="substring-after($s,'\')"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="escape-quot-string">
                    <xsl:with-param name="s" select="$s"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="escape-quot-string">
        <xsl:param name="s"/>
        <xsl:choose>
            <xsl:when test="contains($s,'&quot;')">
                <xsl:call-template name="encode-string">
                    <xsl:with-param name="s" select="concat(substring-before($s,'&quot;'),'\&quot;')"/>
                </xsl:call-template>
                <xsl:call-template name="escape-quot-string">
                    <xsl:with-param name="s" select="substring-after($s,'&quot;')"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="encode-string">
                    <xsl:with-param name="s" select="$s"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="encode-string">
        <xsl:param name="s"/>
        <xsl:choose>
            <!-- tab -->
            <xsl:when test="contains($s,'&#x9;')">
                <xsl:call-template name="encode-string">
                    <xsl:with-param name="s" select="concat(substring-before($s,'&#x9;'),'\t',substring-after($s,'&#x9;'))"/>
                </xsl:call-template>
            </xsl:when>
            <!-- line feed -->
            <xsl:when test="contains($s,'&#xA;')">
                <xsl:call-template name="encode-string">
                    <xsl:with-param name="s" select="concat(substring-before($s,'&#xA;'),'\n',substring-after($s,'&#xA;'))"/>
                </xsl:call-template>
            </xsl:when>
            <!-- carriage return -->
            <xsl:when test="contains($s,'&#xD;')">
                <xsl:call-template name="encode-string">
                    <xsl:with-param name="s" select="concat(substring-before($s,'&#xD;'),'\r',substring-after($s,'&#xD;'))"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise><xsl:value-of select="$s"/></xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- this template handles elements -->
    <xsl:template match="*">
        <!-- count the number of elements with the same name -->
        <xsl:variable name="kctr" select="count(key('names', concat(generate-id(..),'/',name())))"/>
        <!-- iterate through by sets of elements with same name -->
        <xsl:for-each select="key('names', concat(generate-id(..),'/',name()))">
            <!-- deal with the element name and start of multiple element block -->
            <xsl:choose>
                <xsl:when test="($kctr > 1) and (position() = 1)">
                    <xsl:text>"</xsl:text>
                    <xsl:value-of select="name()"/>
                    <xsl:text>" : [ </xsl:text>
                </xsl:when>
                <xsl:when test="$kctr = 1 and name()!='result'">
                    <xsl:text>"</xsl:text>
                    <xsl:value-of select="name()"/>
                    <xsl:text>" : </xsl:text>
                </xsl:when>
                <xsl:when test="$kctr = 1 and name()='result'">
                    <xsl:text>"</xsl:text>
                    <xsl:value-of select="name()"/>
                    <xsl:text>" : [</xsl:text>
                </xsl:when>
            </xsl:choose>
            <!-- count number of elements, text nodes and attribute nodes -->
            <xsl:variable name="nctr" select="count(*|text()|@*)"/>
            <xsl:choose>
                <xsl:when test="$nctr = 0">
                    <!-- no contents at all -->
                    <xsl:text>null</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:variable name="ctr" select="count(*)"/>
                    <xsl:variable name="tctr" select="count(text())"/>
                    <xsl:variable name="actr" select="count(@*)"/>
                    <!-- there will be contents so start an object -->
                    <xsl:text>{ </xsl:text>
                    <!-- handle attribute nodes -->
                    <xsl:if test="$actr > 0">
                        <xsl:apply-templates select="@*"/>
                        <xsl:if test="($tctr > 0) or ($ctr > 0)">
                            <xsl:text>, </xsl:text>
                        </xsl:if>
                    </xsl:if>
                    <!-- call template for child elements one unique name at a time -->
                    <xsl:if test="$ctr > 0">
                        <xsl:apply-templates select="*[generate-id(.) = generate-id(key('names', concat(generate-id(..),'/',name()))[1])]">
                            <xsl:sort select="name()" data-type="number"/>
                        </xsl:apply-templates>
                        <xsl:if test="$tctr > 0">
                            <xsl:text>, </xsl:text>
                        </xsl:if>
                    </xsl:if>
                    <!-- handle text nodes -->
                    <xsl:choose>
                        <xsl:when test="$tctr = 1">
                            <xsl:text>"$" : </xsl:text>
                            <xsl:apply-templates select="text()"/>
                        </xsl:when>
                        <xsl:when test="$tctr > 1">
                            <xsl:text>"$" : [ </xsl:text>
                            <xsl:apply-templates select="text()"/>
                            <xsl:text> ]</xsl:text>
                        </xsl:when>
                    </xsl:choose>
                    <xsl:text> }</xsl:text>
                </xsl:otherwise>
            </xsl:choose>
            <!-- special processing if we are in multiple element block -->
            <xsl:if test="$kctr > 1">
                <xsl:choose>
                    <xsl:when test="position() = last()">
                        <xsl:text> ]</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:text>, </xsl:text>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:if>
            <xsl:if test="$kctr = 1 and name()='result'">
                <xsl:text> ]</xsl:text>
            </xsl:if>
        </xsl:for-each>
        <xsl:if test="position() != last()">
            <xsl:text>, </xsl:text>
        </xsl:if>
    </xsl:template>

    <!-- this template handle text nodes -->
    <xsl:template match="text()">
        <xsl:variable name="t" select="." />
        <xsl:choose>
            <!-- test to see if it is a number -->
            <xsl:when test="string(number($t)) != 'NaN' and not(starts-with($t,'+')) and (not(local-name(..)='code') and not(local-name(..)='subject'))">
                <xsl:value-of select="$t"/>
            </xsl:when>
            <!-- deal with any case booleans -->
            <xsl:when test="translate($t, 'TRUE', 'true') = 'true'">
                <xsl:text>true</xsl:text>
            </xsl:when>
            <xsl:when test="translate($t, 'FALSE', 'false') = 'false'">
                <xsl:text>false</xsl:text>
            </xsl:when>
            <!-- must be text -->
            <xsl:otherwise>
                <xsl:text>"</xsl:text>
                <xsl:call-template name="escape-bs-string">
                    <xsl:with-param name="s" select="$t"/>
                </xsl:call-template>
                <xsl:text>"</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
        <xsl:if test="position() != last()">
            <xsl:text>, </xsl:text>
        </xsl:if>
    </xsl:template>

    <!-- this template handles attribute nodes -->
    <xsl:template match="@*">
        <!-- attach prefix to attribute names -->
        <xsl:text>"@</xsl:text>
        <xsl:value-of select="name()"/>
        <xsl:text>" : </xsl:text>
        <xsl:variable name="t" select="." />
        <xsl:choose>
            <xsl:when test="translate($t, 'TRUE', 'true') = 'true'">
                <xsl:text>true</xsl:text>
            </xsl:when> 
            <xsl:when test="translate($t, 'FALSE', 'false') = 'false'">
                <xsl:text>false</xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:text>"</xsl:text>
                <xsl:value-of select="$t"/>
                <xsl:text>"</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
        <xsl:if test="position() != last()">
            <xsl:text>, </xsl:text>
        </xsl:if>
    </xsl:template>

</xsl:stylesheet>