package eu.dnetlib.oai.parser;

import java.io.StringReader;
import java.util.List;
import java.util.Map.Entry;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import eu.dnetlib.oai.PublisherField;
import eu.dnetlib.oai.conf.OAIConfigurationExistReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

/**
 * An instance of this class can parse an XML record and extract the information needed to store the record in a publisher store.
 *
 * @author alessia
 */
public class PublisherRecordParser {

	private static final Log log = LogFactory.getLog(PublisherRecordParser.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * List of the indices of the target store.
	 */
	private List<PublisherField> storeIndices;

	private final SAXReader saxReader = new SAXReader();

	/**
	 * Parses the record and returns a map where a key is the name of an index, the value is the value in the record at the xpath specificed
	 * in this.storeIndices.
	 *
	 * @param record
	 *            the XML string to parse.
	 * @param source
	 *             String identifying the source of the record. Can be null.
	 * @return a Multimap describing the values to be indexed for this record.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Multimap<String, String> parseRecord(final String record, final String source) {
		Multimap<String, String> recordProps = ArrayListMultimap.create();
		try {
			Document doc = this.saxReader.read(new StringReader(record));
			if(StringUtils.isNotBlank(source)) recordProps.put(OAIConfigurationExistReader.SET_FIELD, source);
			for (PublisherField field : this.storeIndices) {
				for (Entry<String, String> indexEntry : field.getSources().entries()) {
					// each xpath can return a list of nodes or strings, depending on the xpath
					List xPathResult = doc.selectNodes(indexEntry.getValue());
					if ((xPathResult != null) && !xPathResult.isEmpty()) {
						if (containsStrings(xPathResult)) {
							recordProps.putAll(field.getFieldName(), xPathResult);
						} else {
							if (containsNodes(xPathResult)) {

								recordProps.putAll(field.getFieldName(), Iterables.transform(xPathResult, obj -> {
									if (obj == null) return "";
									Node node = (Node) obj;
									return node.getText();
								}));
							}
						}
					}
				}
			}

		} catch (DocumentException e) {
			log.fatal("Can't parse record");
			recordProps = null;
		}
		return recordProps;
	}

	@SuppressWarnings("rawtypes")
	private boolean containsStrings(final List objects) {
		Object first = objects.get(0);
		return first instanceof String;
	}

	@SuppressWarnings("rawtypes")
	private boolean containsNodes(final List objects) {
		Object first = objects.get(0);
		return first instanceof Node;
	}

	public List<PublisherField> getStoreIndices() {
		return storeIndices;
	}

	public void setStoreIndices(final List<PublisherField> storeIndices) {
		this.storeIndices = storeIndices;
	}

	public SAXReader getSaxReader() {
		return saxReader;
	}

	public PublisherRecordParser(final List<PublisherField> storeIndices) {
		this.storeIndices = storeIndices;
	}

	public PublisherRecordParser() {
		super();
		// TODO Auto-generated constructor stub
	}


}
