package eu.dnetlib.oai.mongo;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.Date;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import com.google.common.collect.Sets;
import com.mongodb.DBObject;
import eu.dnetlib.oai.conf.OAIConfigurationReader;
import eu.dnetlib.oai.info.RecordInfo;
import eu.dnetlib.rmi.provision.OaiPublisherRuntimeException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Helper class to generate a RecordInfo object from a Mongo DBObject.
 *
 * @author alessia
 */
public class RecordInfoGenerator {

	@Autowired
	private MetadataExtractor metadataExtractor;
	@Autowired
	private ProvenanceExtractor provenanceExtractor;

	@SuppressWarnings("unchecked")
	public RecordInfo transformDBObject(final DBObject object, final boolean includeBody) {
		if ((object == null) || object.keySet().isEmpty()) { return null; }
		String id = (String) object.get(OAIConfigurationReader.ID_FIELD);
		// need to escape the identifier, otherwise the XML breaks
		id = StringEscapeUtils.escapeXml11(id);
		final boolean deleted = (Boolean) object.get("deleted");
		final RecordInfo record = new RecordInfo();
		record.setIdentifier(id);
		record.setInternalId(object.get("_id").toString());
		record.setDatestamp((Date) object.get(OAIConfigurationReader.DATESTAMP_FIELD));
		record.setDeleted(deleted);
		final List<String> sets = (List<String>) object.get(OAIConfigurationReader.SET_FIELD);
		if (sets != null) {
			record.setSetspecs(Sets.newHashSet(sets));
		}
		if (includeBody && !deleted) {
			final String body = decompressRecord((byte[]) object.get(OAIConfigurationReader.BODY_FIELD));
			final SAXReader reader = new SAXReader();
			Document doc;
			try {
				doc = reader.read(new StringReader(body));
				record.setMetadata(this.metadataExtractor.apply(doc));
				record.setProvenance(this.provenanceExtractor.apply(doc));
			} catch (final DocumentException e) {
				throw new OaiPublisherRuntimeException(e);
			}
		}
		return record;

	}

	public String decompressRecord(final byte[] input) {

		try {
			final ByteArrayInputStream bis = new ByteArrayInputStream(input);
			final ZipInputStream zis = new ZipInputStream(bis);
			ZipEntry ze;
			ze = zis.getNextEntry();
			if (ze == null) { throw new OaiPublisherRuntimeException("cannot decompress null zip entry "); }
			if (!ze.getName()
					.equals(OAIConfigurationReader.BODY_FIELD)) { throw new OaiPublisherRuntimeException("cannot decompress zip entry name :" + ze.getName()); }
			return IOUtils.toString(zis);
		} catch (final IOException e) {
			throw new OaiPublisherRuntimeException(e);
		}

	}

	public MetadataExtractor getMetadataExtractor() {
		return this.metadataExtractor;
	}

	public void setMetadataExtractor(final MetadataExtractor metadataExtractor) {
		this.metadataExtractor = metadataExtractor;
	}

	public ProvenanceExtractor getProvenanceExtractor() {
		return this.provenanceExtractor;
	}

	public void setProvenanceExtractor(final ProvenanceExtractor provenanceExtractor) {
		this.provenanceExtractor = provenanceExtractor;
	}
}
