package eu.dnetlib.oai.core;

import java.util.List;

import eu.dnetlib.oai.CannotDisseminateFormatException;
import eu.dnetlib.oai.conf.OAIConfigurationExistReader;
import eu.dnetlib.oai.utils.OAIISLookUpClient;
import eu.dnetlib.oai.info.*;
import eu.dnetlib.oai.sets.SetCollection;
import eu.dnetlib.rmi.provision.MDFInfo;
import eu.dnetlib.rmi.provision.OaiPublisherException;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

/**
 * Core OAI servlet implementation.
 *
 * @author michele, alessia
 */

public abstract class AbstractOAICore {

	private static final Log log = LogFactory.getLog(AbstractOAICore.class); // NOPMD by marko on 11/24/08 5:02 PM
	/**
	 * page size.
	 */
	protected int pageSize = 100;
	private String currentDBName;

	@Autowired
	private SetCollection setCollection;
	@Autowired
	private OAIConfigurationExistReader oaiConfigurationExistReader;

	@Autowired
	private OAIISLookUpClient lookupClient;

	private Cache mdFormatsCache;

	/**
	 * Returns informations about a record. It contains basic information about a record, such as "prefix", "identifier", "datestamp",
	 * "setspec", "metadata".
	 *
	 * @param identifier record identifier
	 * @param prefix     metadata prefix
	 * @return a recordInfo instance
	 * @throws OaiPublisherException could happen
	 */
	public RecordInfo getInfoRecord(final String identifier, final String prefix) throws OaiPublisherException {
		MDFInfo mdf = obtainMDFInfo(prefix);
		return getRecordById(mdf, identifier);
	}

	/**
	 * List records.
	 *
	 * @param onlyIdentifiers only return record identifiers
	 * @param metadataPrefix  metadata prefix
	 * @param set             set name
	 * @param from            from date
	 * @param until           to date
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String metadataPrefix, final String set, final String from, final String until)
			throws OaiPublisherException {
		final ListRecordsInfo res = new ListRecordsInfo();

		if (from != null) {
			res.setFrom(from);
			res.setWithSize(false);
		}
		if (until != null) {
			res.setUntil(until);
			res.setWithSize(false);
		}
		if (metadataPrefix != null) {
			res.setMetadataprefix(metadataPrefix);
		}
		if (set != null) {
			res.setSet(set);
		}

		ListDocumentsInfo documents = getDocuments(onlyIdentifiers, set, metadataPrefix, from, until);
		ResumptionToken resumptionToken = documents.getResumptionToken();
		int cursor = documents.getCursor();
		int nMaxElements = documents.getnMaxElements();

		if (onlyIdentifiers) {
			res.setIdentifiers(documents.getDocs());
		} else {
			res.setDocuments(documents.getDocs());
		}

		if ((resumptionToken != null)) {
			res.setResumptiontoken(resumptionToken.serialize());
			res.setCursor(cursor);
			res.setSize(nMaxElements);
		}
		return res;
	}

	/**
	 * List records.
	 *
	 * @param onlyIdentifiers only resource identifiers.
	 * @param resumptionToken resumption token
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException {
		ListDocumentsInfo docs = getDocuments(onlyIdentifiers, resumptionToken);

		final ListRecordsInfo res = new ListRecordsInfo();

		if (docs.getMetadataPrefix() != null) {
			res.setMetadataprefix(docs.getMetadataPrefix());
		}

		if (onlyIdentifiers) {
			res.setIdentifiers(docs.getDocs());
		} else {
			res.setDocuments(docs.getDocs());
		}

		ResumptionToken newResumptionToken = docs.getResumptionToken();
		if ((newResumptionToken != null)) {
			res.setResumptiontoken(newResumptionToken.serialize());
			res.setCursor(docs.getCursor());
			res.setSize(docs.getnMaxElements());
			res.setWithSize(!newResumptionToken.hasDateRange());
		}
		return res;
	}

	public List<? extends SetInfo> listSets() throws OaiPublisherException {
		return this.setCollection.getAllSets(true, getCurrentDBName());
	}

	public List<MDFInfo> listMetadataFormats() throws OaiPublisherException {
		return this.oaiConfigurationExistReader.getMetadataFormatInfo(true);
	}

	public boolean existSet(final String setSpec) {
		return this.setCollection.containEnabledSet(setSpec, getCurrentDBName());
	}

	protected MDFInfo obtainMDFInfo(final String metadataPrefix) throws OaiPublisherException {

		MDFInfo mdf = null;
		final Element element = this.mdFormatsCache.get(metadataPrefix);
		if (element == null) {
			log.fatal("Not using cache for " + metadataPrefix);
			mdf = oaiConfigurationExistReader.getMetadataFormatInfo(metadataPrefix);
			if (mdf == null) throw new CannotDisseminateFormatException("Invalid metadataPrefix " + metadataPrefix);
			this.mdFormatsCache.put(new Element(metadataPrefix, mdf));
		} else {
			mdf = (MDFInfo) element.getObjectValue();
		}
		return mdf;
	}

	/**
	 * Set the current DB from the configuration on the IS.
	 */
	public void setCurrentDBFromIS() throws OaiPublisherException {
		try {
			currentDBName = getLookupClient().getCurrentDB();
		} catch (Exception e) {
			throw new OaiPublisherException(e);
		}
	}

	protected abstract RecordInfo getRecordById(final MDFInfo mdf, final String id) throws OaiPublisherException;

	protected abstract ListDocumentsInfo getDocuments(final boolean onlyIdentifiers,
			final String set,
			final String metadataPrefix,
			final String from,
			final String until) throws OaiPublisherException;

	protected abstract ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException;

	public String getCurrentDBName() {
		return currentDBName;
	}

	public void setCurrentDBName(final String currentDBName) {
		this.currentDBName = currentDBName;
	}

	public SetCollection getSetCollection() {
		return setCollection;
	}

	public void setSetCollection(final SetCollection setCollection) {
		this.setCollection = setCollection;
	}

	public OAIConfigurationExistReader getOaiConfigurationExistReader() {
		return oaiConfigurationExistReader;
	}

	public void setOaiConfigurationExistReader(final OAIConfigurationExistReader oaiConfigurationExistReader) {
		this.oaiConfigurationExistReader = oaiConfigurationExistReader;
	}

	public OAIISLookUpClient getLookupClient() {
		return lookupClient;
	}

	public void setLookupClient(final OAIISLookUpClient lookupClient) {
		this.lookupClient = lookupClient;
	}

	public Cache getMdFormatsCache() {
		return mdFormatsCache;
	}

	@Required
	public void setMdFormatsCache(final Cache mdFormatsCache) {
		this.mdFormatsCache = mdFormatsCache;
	}
}
