package eu.dnetlib.data.objectstore;

import java.util.List;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardServerHandler;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * The Class CreateObjectStoreAction is responsible to execute the blacboard action of type CREATE.
 */
public class CreateObjectStoreAction extends AbstractObjectStoreAction {

	private static final Log log = LogFactory.getLog(CreateObjectStoreAction.class);

	@Autowired
	private UniqueServiceLocator locator;


	/**
	 * The profile creator.
	 */
	private ObjectStoreProfileCreator profileCreator;

	/**
	 * Gets the profile creator.
	 *
	 * @return the profile creator
	 */
	public ObjectStoreProfileCreator getProfileCreator() {
		return profileCreator;
	}

	/**
	 * Sets the profile creator.
	 *
	 * @param profileCreator the new profile creator
	 */
	public void setProfileCreator(final ObjectStoreProfileCreator profileCreator) {
		this.profileCreator = profileCreator;
	}

	@Override
	protected void executeAsync(final BlackboardServerHandler handler, final BlackboardJob job) {
		try {
			final String interpretation = job.getParameters().get("interpretation");
			final String basePath = job.getParameters().get("basePath");
			final boolean createOnlyIfMissing = BooleanUtils.toBoolean(job.getParameters().get("createOnlyIfMissing"));
			if (createOnlyIfMissing) {
				final String objID = findObjectStoreByInterpretation(interpretation);
				if (objID != null) {
					job.getParameters().put("objectStoreId", objID);
					completeWithSuccess(handler, job);
					return;
				}
			}

			final String objID = profileCreator.registerProfile(interpretation);
			getDao().createObjectStore(objID, interpretation, basePath);
			job.getParameters().put("objectStoreId", objID);
			completeWithSuccess(handler, job);
		} catch (Exception e) {
			completeWithFail(handler, job, e);
		}

	}

	private String findObjectStoreByInterpretation(final String interpretation) {
		final ISLookUpService service = locator.getService(ISLookUpService.class);
		final String query =
				"for $x in collection('/db/DRIVER/ObjectStoreDSResources/ObjectStoreDSResourceType') where $x//OBJECTSTORE_INTERPRETATION='%s' return $x//RESOURCE_IDENTIFIER/@value/string()";
		try {
			final List<String> strings = service.quickSearchProfile(String.format(query, interpretation));
			if (strings != null && strings.size() > 0)
				return strings.get(0);
		} catch (ISLookUpException e) {
			log.error("error on executing query", e);

		}
		return null;

	}
}
