package eu.dnetlib.data.objectstore.gridFS;

import com.mongodb.*;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import eu.dnetlib.enabling.resultset.listener.ResultSetListener;
import eu.dnetlib.rmi.common.ResultSetException;
import eu.dnetlib.rmi.data.ObjectStoreFile;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * The listener interface for receiving gridFSObjectstoreResultSet events. The class that is interested in processing a
 * gridFSObjectstoreResultSet event implements this interface, and the object created with that class is registered with a component using
 * the component's <code>addGridFSObjectstoreResultSetListener<code> method. When
 * the gridFSObjectstoreResultSet event occurs, that object's appropriate
 * method is invoked.
 */
public class GridFSObjectstoreResultSetListener implements ResultSetListener<ObjectStoreFile> {

	/**
	 * The Constant log.
	 */
	private static final Log log = LogFactory.getLog(GridFSObjectstoreResultSetListener.class);

    private ObjectStoreFile nextElement = null;

	/**
	 * The from date.
	 */
	private Long fromDate;

	/**
	 * The until date.
	 */
	private Long untilDate;

	/**
	 * The records.
	 */
	private Iterable<String> records;

	/**
	 * The object store id.
	 */
	private String objectStoreID;

	/**
	 * The collection.
	 */
	private GridFS collection;

	/**
	 * The base uri.
	 */
	private String baseURI;

	/**
	 * The current cursor.
	 */
	private DBCursor currentCursor;

	public GridFSObjectstoreResultSetListener(final Long fromDate,
			final Long untilDate,
			final String objectStoreID,
			final GridFS collection,
			final String baseURI,
			final Iterable<String> records) {
		this.fromDate = fromDate;
		this.untilDate = untilDate;
		this.objectStoreID = objectStoreID;
		this.collection = collection;
		this.baseURI = baseURI;
		this.records = records;
		createCurrentCursor();
        constructNextElement();
    }


    private void constructNextElement() {
        if (currentCursor.hasNext()) {
            nextElement = ObjectStoreFileUtility.build((GridFSDBFile) currentCursor.next(), baseURI, objectStoreID);
        } else
            nextElement = null;
    }

	@Override
	public boolean hasNext() {
        return nextElement != null;
    }

	@Override
	public ObjectStoreFile next() throws ResultSetException {
        final ObjectStoreFile tmp = nextElement;
        constructNextElement();
        return tmp;
    }

	@Override
	public int getCount() {
		return currentCursor.itcount();
	}

	@Override
	public int getTotal() {
		return currentCursor.count();
	}

	/**
	 * Creates the current cursor.
	 */
	private void createCurrentCursor() {
		if (log.isDebugEnabled()) {
			log.debug(String.format("Staring query "));
		}

		DBObject query;
		if (records != null) {
			QueryBuilder qBuilder = QueryBuilder.start("metadata.id").in(records);
			query = qBuilder.get();
		} else if ((fromDate != null) && (untilDate != null)) {
			query = new BasicDBObject();
			query.put("$gt", fromDate.doubleValue());
			query.put("$lt", untilDate.doubleValue());
			if (currentCursor != null) {
				currentCursor.close();
			}
		} else
			throw new IllegalArgumentException("Missing parameters on Delivery must provide either from, to, or ObjectStoreIDs");
        currentCursor = collection.getFileList(new BasicDBObject("metadata.timestamp", query)).sort(new BasicDBObject("_id", 1)).addOption(Bytes.QUERYOPTION_NOTIMEOUT);
    }

	/**
	 * Gets the from date.
	 *
	 * @return the from date
	 */
	public Long getFromDate() {
		return fromDate;
	}

	/**
	 * Sets the from date.
	 *
	 * @param fromdate the new from date
	 */
	public void setFromDate(final Long fromdate) {
		this.fromDate = fromdate;
	}

	/**
	 * Gets the until date.
	 *
	 * @return the until date
	 */
	public Long getUntilDate() {
		return untilDate;
	}

	/**
	 * Sets the until date.
	 *
	 * @param untilDate the new until date
	 */
	public void setUntilDate(final Long untilDate) {
		this.untilDate = untilDate;
	}

	/**
	 * Gets the records.
	 *
	 * @return the records
	 */
	public Iterable<String> getRecords() {
		return records;
	}

	/**
	 * Sets the records.
	 *
	 * @param records the new records
	 */
	public void setRecords(final Iterable<String> records) {
		this.records = records;
	}

	/**
	 * Gets the collection.
	 *
	 * @return the collection
	 */
	public GridFS getCollection() {
		return collection;
	}

	/**
	 * Sets the collection.
	 *
	 * @param collection the new collection
	 */
	public void setCollection(final GridFS collection) {
		this.collection = collection;
	}

	/**
	 * Gets the base uri.
	 *
	 * @return the base uri
	 */
	public String getBaseURI() {
		return baseURI;
	}

	/**
	 * Sets the base uri.
	 *
	 * @param baseURI the new base uri
	 */
	public void setBaseURI(final String baseURI) {
		this.baseURI = baseURI;
	}

	/**
	 * Gets the object store id.
	 *
	 * @return the object store id
	 */
	public String getObjectStoreID() {
		return objectStoreID;
	}

	/**
	 * Sets the object store id.
	 *
	 * @param objectStoreID the new object store id
	 */
	public void setObjectStoreID(final String objectStoreID) {
		this.objectStoreID = objectStoreID;
	}

}
