package eu.dnetlib.data.transform.xml;

import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message.Builder;
import eu.dnetlib.data.graph.model.DNGFRowKeyDecoder;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.DNGFProtos.DNGFEntity;
import eu.dnetlib.data.proto.DNGFProtos.DNGFRel;
import eu.dnetlib.data.proto.DatasetProtos.Dataset;
import eu.dnetlib.data.proto.DatasourceProtos.Datasource;
import eu.dnetlib.data.proto.FieldTypeProtos.DataInfo;
import eu.dnetlib.data.proto.FieldTypeProtos.Instance;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import eu.dnetlib.data.proto.OrganizationProtos.Organization;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.ProjectProtos.Project;
import eu.dnetlib.data.proto.PublicationProtos.Publication;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DbmfToHbaseXsltFunctions extends CommonDNetXsltFunctions {

	public static String oafEntity(
			final String id,
			final String collectedFromId,
			final String collectedFromName,
			final String originalId,
			final String dateOfCollection,
			final String dateOfTransformation,
			final NodeList nodeList) {

		final DNGFRowKeyDecoder d = DNGFRowKeyDecoder.decode(id);
		final String entityId = d.getKey();
		switch (d.getType()) {
		case datasource:
			return serializeOafEntity(nodeList, Type.datasource, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Datasource.newBuilder());
		case organization:
			return serializeOafEntity(nodeList, Type.organization, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Organization.newBuilder());
		case person:
			return serializeOafEntity(nodeList, Type.person, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Person.newBuilder());
		case project:
			return serializeOafEntity(nodeList, Type.project, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Project.newBuilder());
		case publication:
			return serializeOafEntity(nodeList, Type.publication, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Publication.newBuilder());
		case dataset:
			return serializeOafEntity(nodeList, Type.dataset, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection,
					dateOfTransformation, Dataset.newBuilder());
		default:
			throw new IllegalArgumentException("Invalid entity type: " + d.getType());
		}
	}

	public static String rel(
			final String source,
			final String target,
			final String relClass,
			final String relScheme) {
		return rel(source, target, null, relClass, relScheme);
	}

	public static String rel(
			final String source,
			final String target,
			final NodeList nodeList,
			final String relClass,
			final String relScheme) {

		final DNGFRowKeyDecoder sd = DNGFRowKeyDecoder.decode(source);
		final DNGFRowKeyDecoder td = DNGFRowKeyDecoder.decode(target);
		final Qualifier relType = getSimpleQualifier(relClass, relScheme).build();

		return serializeOafRel(nodeList, sd.getKey(), sd.getType(), td.getKey(), td.getType(), relType, false);
	}

	// ////////////////////////////////////////////////////////

	protected static String serializeOafEntity(final NodeList nodelist,
			final Type type,
			final String id,
			final KeyValue collectedFrom,
			final String originalId,
			final String dateOfCollection,
			final String dateOfTransformation,
			final Builder entity) {
		try {
			final FieldDescriptor md = entity.getDescriptorForType().findFieldByName("metadata");

			final DNGFEntity.Builder parent = getEntity(type, id, collectedFrom, Lists.newArrayList(originalId), dateOfCollection, dateOfTransformation, null);
			final Builder metadata = entity.newBuilderForField(md);
			final DataInfo.Builder dataInfo = DataInfo.newBuilder();

			if (type.equals(Type.publication)) {
				final Instance.Builder instance = Instance.newBuilder();
				parseNodelist(nodelist, instance);
				FieldDescriptor instanceDescriptor = Publication.getDescriptor().findFieldByName(Instance.getDescriptor().getName());
				if (instanceDescriptor != null) {
					entity.setField(instanceDescriptor, instance);
				}
			}
			if (type.equals(Type.dataset)) {
				final Instance.Builder instance = Instance.newBuilder();
				parseNodelist(nodelist, instance);
				FieldDescriptor instanceDescriptor = Dataset.getDescriptor().findFieldByName(Instance.getDescriptor().getName());
				if (instanceDescriptor != null) {
					entity.setField(instanceDescriptor, instance);
				}
			}
			parseNodelist(nodelist, parent, entity, metadata, dataInfo);

			final FieldDescriptor entityDescriptor = DNGFEntity.getDescriptor().findFieldByName(type.toString());

			final DNGF build = getOaf(parent.setField(entityDescriptor, entity.setField(md, metadata.build()).build()), dataInfo);

			return base64(build.toByteArray());
		} catch (Exception e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}
	}

	protected static String serializeOafRel(
			final NodeList nodeList,
			final String sourceId,
			final Type sourceType,
			final String targetId,
			final Type targetType,
			final Qualifier relType,
			final boolean isChild) {
		try {
			final DataInfo.Builder dataInfo = DataInfo.newBuilder();

			parseNodelist(nodeList, dataInfo);

			DNGFRel.Builder builder = getRel(sourceId, sourceType, targetId, targetType, relType, isChild);

			DNGF build = getOaf(builder, dataInfo);
			return base64(build.toByteArray());
		} catch (Exception e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}
	}

	private static void parseNodelist(final NodeList nodeList, final Builder... builders) {

		if (nodeList == null) return;

		for (int i = 0; i < nodeList.getLength(); i++) {

			final Node fieldNode = nodeList.item(i);
			final Node attr = fieldNode.getAttributes().getNamedItem("name");

			final String fieldName = attr.getNodeValue();
			final NodeList chilidren = fieldNode.getChildNodes();

			for (int j = 0; j < chilidren.getLength(); j++) {

				final Node child = chilidren.item(j);
				final String childName = child.getLocalName();
				if ("ITEM".equals(childName) || StringUtils.isBlank(childName)) {
					for (Builder builder : builders) {
						FieldDescriptor desc = builder.getDescriptorForType().findFieldByName(fieldName);
						if (desc != null) {
							String text = getText((StringUtils.isBlank(childName)) ? fieldNode : child);
							if (!StringUtils.isBlank(text)) {
								addField(builder, desc, text);
							}
						}
					}
				}
			}
		}
	}

	private static String getText(final Node node) {
		StringBuffer result = new StringBuffer();
		if (!node.hasChildNodes()) { return ""; }

		NodeList list = node.getChildNodes();
		for (int i = 0; i < list.getLength(); i++) {
			Node subnode = list.item(i);
			if (subnode.getNodeType() == Node.TEXT_NODE) {
				result.append(subnode.getNodeValue());
			} else if (subnode.getNodeType() == Node.CDATA_SECTION_NODE) {
				result.append(subnode.getNodeValue());
			} else if (subnode.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
				result.append(getText(subnode));
			}
		}
		return result.toString().trim();
	}

}
