package eu.dnetlib.data.transform.xml;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.protobuf.Message.Builder;
import com.google.protobuf.ProtocolMessageEnum;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.DNGFProtos.DNGFEntity;
import eu.dnetlib.data.proto.DNGFProtos.DNGFRel;
import eu.dnetlib.data.proto.FieldTypeProtos.*;
import eu.dnetlib.data.proto.FieldTypeProtos.OAIProvenance.OriginDescription;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.miscutils.collections.Pair;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public abstract class AbstractDNetXsltFunctions {

	public static final String URL_REGEX = "^(http|https|ftp)\\://.*";
	private static final int MAX_NSPREFIX_LEN = 12;


	public static Predicate<String> urlFilter = s -> s.trim().matches(URL_REGEX);

	// Builder for Entities
	protected static DNGF getOaf(final DNGFEntity.Builder entity, final DataInfo.Builder info) {
		return _getOaf(DNGF.newBuilder(), info).setKind(Kind.entity).setEntity(entity).build();
	}

	// Builder for Rels
	protected static DNGF getOaf(final DNGFRel.Builder rel, final DataInfo.Builder info) {
		return _getOaf(DNGF.newBuilder(), info).setKind(Kind.relation).setRel(rel).build();
	}

	private static DNGF.Builder _getOaf(final DNGF.Builder oaf, final DataInfo.Builder info) {
		if (info != null) {
			return oaf.setDataInfo(ensureDataInfo(info));
		} else return oaf;
	}

	protected static DataInfo.Builder ensureDataInfo(final DataInfo.Builder info) {
		if (info.isInitialized()) return info;
		return getDataInfo(null, "UNKNOWN", "0.9", false, false);
	}

	protected static KeyValue getKV(final String id, final String name) {
		return KeyValue.newBuilder().setKey(id).setValue(name).build();
	}

	protected static DNGFRel.Builder getRel(
			final String sourceId,
			final Type sourceType,
			final String targetId,
			final Type targetType,
			final Qualifier relType,
			final boolean isChild) {
		return DNGFRel.newBuilder().setSource(sourceId).setTarget(targetId)
				.setSourceType(sourceType)
				.setTargetType(targetType).setRelType(relType)
				.setChild(isChild);
	}

	protected static DNGFEntity.Builder getEntity(final Type type,
			final String id,
			final KeyValue collectedFrom,
			final List<String> originalIds,
			final String dateOfCollection,
			final String dateOfTransformation,
			final List<StructuredProperty> pids) {
		final DNGFEntity.Builder builder = DNGFEntity.newBuilder().setType(type).setId(id);
		if (collectedFrom != null) builder.addCollectedfrom(collectedFrom);
		builder.setDateoftransformation(StringUtils.isBlank(dateOfTransformation) ? "" : dateOfTransformation);
		builder.setDateofcollection(StringUtils.isBlank(dateOfCollection) ? "" : dateOfCollection);

		if ((originalIds != null) && !originalIds.isEmpty()) {
			builder.addAllOriginalId(Iterables.filter(originalIds, getPredicateNotBlankString()));
		}

		if ((pids != null) && !pids.isEmpty()) {
			builder.addAllPid(Iterables.filter(pids, Predicates.notNull()));
		}

		return builder;
	}

    public static Predicate<String> getPredicateNotBlankString() {
        return s -> StringUtils.isNotBlank(s);
	}

	public static DataInfo.Builder getDataInfo(final NodeList about,
			final String provenanceaction,
			final String trust,
			final boolean deletedbyinference,
			final boolean inferred) {

		final DataInfo.Builder dataInfoBuilder = DataInfo.newBuilder();
		dataInfoBuilder.setInferred(Boolean.valueOf(inferred));
		dataInfoBuilder.setDeletedbyinference(Boolean.valueOf(deletedbyinference));
		dataInfoBuilder.setTrust(trust);
		dataInfoBuilder.setProvenanceaction(getSimpleQualifier(provenanceaction, "dnet:provenanceActions").build());

		// checking instanceof because when receiving an empty <oaf:datainfo> we don't want to parse it.
		if (((about != null) && (about.getLength() > 0)) /* && (dataInfo instanceof org.w3c.dom.Element) */) {

			final org.w3c.dom.Element dataInfoElement = getDirectChild((org.w3c.dom.Element) about.item(0), "datainfo");
			if (dataInfoElement != null) {
				org.w3c.dom.Element elem = getDirectChild(dataInfoElement, "inferred");
				dataInfoBuilder.setInferred(Boolean.valueOf(getStringValue(elem, String.valueOf(inferred))));

				elem = getDirectChild(dataInfoElement, "deletedbyinference");
				dataInfoBuilder.setDeletedbyinference(Boolean.valueOf(getStringValue(elem, String.valueOf(deletedbyinference))));

				elem = getDirectChild(dataInfoElement, "trust");
				dataInfoBuilder.setTrust(getStringValue(elem, trust));

				elem = getDirectChild(dataInfoElement, "inferenceprovenance");
				dataInfoBuilder.setInferenceprovenance(getStringValue(elem));

				elem = getDirectChild(dataInfoElement, "provenanceaction");
				final Qualifier.Builder pBuilder = Qualifier.newBuilder();
				if (elem != null && elem.hasAttributes()) {
					final NamedNodeMap attributes = elem.getAttributes();
					pBuilder.setClassid(getAttributeValue(attributes, "classid"));
					pBuilder.setClassname(getAttributeValue(attributes, "classname"));
					pBuilder.setSchemeid(getAttributeValue(attributes, "schemeid"));
					pBuilder.setSchemename(getAttributeValue(attributes, "schemename"));
				} else {
					pBuilder.mergeFrom(getSimpleQualifier(provenanceaction, "dnet:provenanceActions").build());
				}
				dataInfoBuilder.setProvenanceaction(pBuilder);
			}
		}

		return dataInfoBuilder;
	}

	protected static OAIProvenance getOAIProvenance(final NodeList about) {

		OAIProvenance.Builder oaiProv = OAIProvenance.newBuilder();

		if (((about != null) && (about.getLength() > 0))) {

			final org.w3c.dom.Element provenance = getDirectChild((org.w3c.dom.Element) about.item(0), "provenance");

			if (provenance != null) {
				final org.w3c.dom.Element origDesc = getDirectChild(provenance, "originDescription");
				oaiProv.setOriginDescription(buildOriginDescription(origDesc, OriginDescription.newBuilder()));
			}
		}

		return oaiProv.build();
	}

	private static OriginDescription buildOriginDescription(final org.w3c.dom.Element origDesc, final OriginDescription.Builder od) {
		od.setHarvestDate(origDesc.getAttribute("harvestDate")).setAltered(Boolean.valueOf(origDesc.getAttribute("altered")));

		org.w3c.dom.Element elem = getDirectChild(origDesc, "baseURL");
		od.setBaseURL(getStringValue(elem));

		elem = getDirectChild(origDesc, "identifier");
		od.setIdentifier(getStringValue(elem));

		elem = getDirectChild(origDesc, "datestamp");
		od.setDatestamp(getStringValue(elem));

		elem = getDirectChild(origDesc, "metadataNamespace");
		od.setMetadataNamespace(getStringValue(elem));

		elem = getDirectChild(origDesc, "originDescription");

		if (elem != null) {

			od.setOriginDescription(buildOriginDescription(elem, OriginDescription.newBuilder()));
		}

		return od.build();
	}

	protected static String getStringValue(final org.w3c.dom.Element elem, final String defaultValue) {
		return (elem != null && elem.getTextContent() != null) ? elem.getTextContent() : defaultValue;
	}

	protected static String getStringValue(final org.w3c.dom.Element elem) {
		return getStringValue(elem, "");
	}

	protected static String getAttributeValue(final NamedNodeMap attributes, final String name) {
		final Node attr = attributes.getNamedItem(name);
		if (attr == null) return "";
		final String value = attr.getNodeValue();
		return value != null ? value : "";
	}

	protected static org.w3c.dom.Element getDirectChild(final org.w3c.dom.Element parent, final String name) {
		for (Node child = parent.getFirstChild(); child != null; child = child.getNextSibling()) {
			if ((child instanceof org.w3c.dom.Element) && name.equals(child.getLocalName())) return (org.w3c.dom.Element) child;
		}
		return null;
	}

	protected static Qualifier.Builder getSimpleQualifier(final String classname, final String schemename) {
		return getQualifier(classname, classname, schemename, schemename);
	}

	protected static Qualifier.Builder getSimpleQualifier(final ProtocolMessageEnum classname, final String schemename) {
		return getQualifier(classname.toString(), classname.toString(), schemename, schemename);
	}

	protected static Qualifier.Builder getQualifier(final String classid, final String classname, final String schemeid, final String schemename) {
		return Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemeid).setSchemename(schemename);
	}

	protected static Qualifier.Builder setQualifier(final Qualifier.Builder qualifier, final List<String> fields) {
		if ((fields == null) || fields.isEmpty() || fields.get(0).isEmpty()) return null;

		if ((fields != null) && !fields.isEmpty() && (fields.get(0) != null)) {
			qualifier.setClassid(fields.get(0));
			qualifier.setClassname(fields.get(0));
		}
		return qualifier;
	}

	protected static void addStructuredProps(final Builder builder,
			final FieldDescriptor fd,
			final List<String> values,
			final String classid,
			final String schemeid) {
		if (values != null) {
			for (final String s : values) {
				addField(builder, fd, getStructuredProperty(s, classid, classid, schemeid, schemeid));
			}
		}
	}

	protected static List<StructuredProperty> parsePids(final NodeList nodelist) {

		final List<StructuredProperty> pids = Lists.newArrayList();

		for (int i = 0; i < nodelist.getLength(); i++) {
			final Node node = nodelist.item(i);
			Node pidType = null;
			if (node.getNodeType() == Node.ELEMENT_NODE) {
				if (node.getLocalName().equalsIgnoreCase("identifier")) {
					pidType = node.getAttributes().getNamedItem("identifierType");
				}
				//this is to handle dataset pids
				if (node.getLocalName().equalsIgnoreCase("alternateIdentifier")) {
					pidType = node.getAttributes().getNamedItem("alternateIdentifierType");
				}

				for (int j = 0; j < node.getChildNodes().getLength(); j++) {
					final Node child = node.getChildNodes().item(j);

					if ((child.getNodeType() == Node.TEXT_NODE) && (pidType != null) && (pidType.getNodeValue() != null) && !pidType.getNodeValue().isEmpty()
							&& !pidType.getNodeValue().equalsIgnoreCase("url")) {

						final String type = pidType.getNodeValue().toLowerCase();

						final String value = child.getTextContent();

						pids.add(getStructuredProperty(value, type, type, "dnet:pid_types", "dnet:pid_types"));
						break;
					}
				}
			}
		}
		return pids;
	}

	@SuppressWarnings("unchecked")
	protected static void addField(final Builder builder, final FieldDescriptor descriptor, Object value) {

		if (value == null) return;

		if (value instanceof List<?>) {
			for (final Object o : (List<Object>) value) {
				addField(builder, descriptor, o);
			}
		} else {
			Object fieldValue = value;
			switch (descriptor.getType()) {
			case BOOL:
				fieldValue = Boolean.valueOf(value.toString());
				break;
			case BYTES:
				fieldValue = value.toString().getBytes(Charset.forName("UTF-8"));
				break;
			case DOUBLE:
				fieldValue = Double.valueOf(value.toString());
				break;
			case FLOAT:
				fieldValue = Float.valueOf(value.toString());
				break;
			case INT32:
			case INT64:
			case SINT32:
			case SINT64:
				fieldValue = Integer.valueOf(value.toString());
				break;
			case MESSAGE:
				final Builder q = builder.newBuilderForField(descriptor);

				if (value instanceof Builder) {
					value = ((Builder) value).build();
					final byte[] b = ((Message) value).toByteArray();
					try {
						q.mergeFrom(b);
					} catch (final InvalidProtocolBufferException e) {
						throw new IllegalArgumentException("Unable to merge value: " + value + " with builder: " + q.getDescriptorForType().getName());
					}
				} else if (Qualifier.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof Qualifier) {
						q.mergeFrom((Qualifier) value);
					} else {
						parseMessage(q, Qualifier.getDescriptor(), value.toString(), "@@@");
					}
				} else if (StructuredProperty.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof StructuredProperty) {
						q.mergeFrom((StructuredProperty) value);
					} else {
						parseMessage(q, StructuredProperty.getDescriptor(), value.toString(), "###");
					}
				} else if (KeyValue.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof KeyValue) {
						q.mergeFrom((KeyValue) value);
					} else {
						parseMessage(q, KeyValue.getDescriptor(), value.toString(), "&&&");
					}
				} else if (StringField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof StringField) {
						q.mergeFrom((StringField) value);
					} else {
						q.setField(StringField.getDescriptor().findFieldByName("value"), value);
					}
				} else if (BoolField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof BoolField) {
						q.mergeFrom((BoolField) value);
					} else if (value instanceof String) {
						q.setField(BoolField.getDescriptor().findFieldByName("value"), Boolean.valueOf((String) value));
					} else {
						q.setField(BoolField.getDescriptor().findFieldByName("value"), value);
					}
				} else if (IntField.getDescriptor().getName().equals(q.getDescriptorForType().getName())) {
					if (value instanceof IntField) {
						q.mergeFrom((IntField) value);
					} else if (value instanceof String) {
						q.setField(IntField.getDescriptor().findFieldByName("value"), NumberUtils.toInt((String) value));
					} else {
						q.setField(IntField.getDescriptor().findFieldByName("value"), value);
					}
				}

				fieldValue = q.buildPartial();
				break;
			default:
				break;
			}

			doAddField(builder, descriptor, fieldValue);
		}

	}

	protected static void doAddField(final Builder builder, final FieldDescriptor fd, final Object value) {
		if (value != null) {
			if (fd.isRepeated()) {
				builder.addRepeatedField(fd, value);
			} else if (fd.isOptional() || fd.isRequired()) {
				builder.setField(fd, value);
			}
		}
	}

	protected static void parseMessage(final Builder builder, final Descriptor descriptor, final String value, final String split) {

		Iterable<Pair> iterablePair = () -> {

			final Iterator<FieldDescriptor> fields = descriptor.getFields().iterator();
			final Iterator<String> values = Lists.newArrayList(Splitter.on(split).trimResults().split(value)).iterator();

			return new Iterator<Pair>() {
				@Override
				public boolean hasNext() {
					return fields.hasNext() && values.hasNext();
				}

				@Override
				public Pair next() {
					final FieldDescriptor field = fields.next();
					final String value1 = values.next();
					return new Pair(field, value1);
				}

				@Override
				public void remove() {
					throw new UnsupportedOperationException("cannot remove");
				}
			};
		};

//		final IterablePair<FieldDescriptor, String> iterablePair =
//				new IterablePair<FieldDescriptor, String>(descriptor.getFields(), Lists.newArrayList(Splitter
//						.on(split).trimResults().split(value)));

		for (final Pair<FieldDescriptor, String> p : iterablePair) {
			addField(builder, p.getKey(), p.getValue());
		}
	}

	protected static String base64(final byte[] data) {
		final byte[] bytes = Base64.encodeBase64(data);
		return new String(bytes);
	}

	public static String replace(final String s, final String regex, final String replacement) {
		return s.replaceAll(regex, replacement);
	}

	public static String trim(final String s) {
		return s.trim();
	}

	protected static String removePrefix(final Type type, final String s) {
		return removePrefix(type.toString(), s);
	}

	private static String removePrefix(final String prefix, final String s) {
		return StringUtils.removeStart("" + s, prefix + "|");
	}

	protected static Qualifier.Builder getDefaultQualifier(final String scheme) {
		final Qualifier.Builder qualifier = Qualifier.newBuilder().setSchemeid(scheme).setSchemename(scheme);
		return qualifier;
	}

	protected static StructuredProperty getStructuredProperty(final String value,
			final String classid,
			final String classname,
			final String schemeid,
			final String schemename) {
		if ((value == null) || value.isEmpty()) return null;
		return StructuredProperty.newBuilder().setValue(value).setQualifier(getQualifier(classid, classname, schemeid, schemename)).build();
	}

	protected static StringField.Builder sf(final String s) {
		return StringField.newBuilder().setValue(s);
	}

	public static String generateNsPrefix(final String prefix, final String externalId) {
		return StringUtils.substring(prefix + StringUtils.leftPad(externalId, MAX_NSPREFIX_LEN - prefix.length(), "_"), 0, MAX_NSPREFIX_LEN);
	}

	public static String md5(final String s) {
		try {
			final MessageDigest md = MessageDigest.getInstance("MD5");
			md.update(s.getBytes("UTF-8"));
			return new String(Hex.encodeHex(md.digest()));
		} catch (final Exception e) {
			System.err.println("Error creating id");
			return null;
		}
	}

	public static String oafId(final String entityType, final String prefix, final String id) {
		if (id.isEmpty() || prefix.isEmpty()) return "";
		return oafSimpleId(entityType, prefix + "::" + md5(id));
	}

	public static String oafSimpleId(final String entityType, final String id) {
		String returnValue = (Type.valueOf(entityType).getNumber() + "|" + id).replaceAll("\\s|\\n", "");
		return returnValue;
	}

	public static String oafSplitId(final String entityType, final String fullId) {
		return oafId(entityType, StringUtils.substringBefore(fullId, "::"), StringUtils.substringAfter(fullId, "::"));
	}

	/**
	 * Utility method, allows to perform param based map lookups in xsl
	 *
	 * @param map
	 * @param key
	 * @return value associated to the key.
	 */
	public static Object lookupValue(final Map<String, Object> map, final String key) {
		return map.get(key);
	}

	/**
	 * Utility method, allows to perform param based map lookups in xsl
	 *
	 * @param map
	 * @param key
	 * @return value associated to the key.
	 */
	public static int mustMerge(final Map<String, Object> map, final String key) {
		final Object val = lookupValue(map, key);
		return (val != null) && (val instanceof String) && val.equals("true") ? 1 : 0;
	}

}
