package eu.dnetlib.data.transform;

import java.io.StringReader;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import javax.xml.transform.Transformer;
import javax.xml.transform.stream.StreamSource;

import com.google.common.base.Function;
import com.ximpleware.AutoPilot;
import com.ximpleware.VTDGen;
import com.ximpleware.VTDNav;
import eu.dnetlib.data.graph.model.DNGFDecoder;
import eu.dnetlib.data.transform.VtdUtilityParser.Node;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.io.DocumentResult;

public class XsltRowTransformer implements Function<String, List<Row>> {

	private static final Log log = LogFactory.getLog(XsltRowTransformer.class); // NOPMD by marko on 11/24/08 5:02 PM

	public static final String DATA_XPATH = "/ROWS/ROW";

	public static final List<String> ATTRS = Arrays.asList("value", "mode");

	private Transformer transformer;

	public XsltRowTransformer(final Transformer transformer) {
		this.transformer = transformer;
		log.info(String.format("using trasformer: '%s'", getTransformerClassName()));
	}

	private List<Row> transform(final String xml) {
		final DocumentResult result = new DocumentResult();
		try {
			transformer.transform(new StreamSource(new StringReader(xml)), result);

			final VTDGen parser = new VTDGen();
			parser.setDoc(result.getDocument().asXML().getBytes());
			parser.parse(true);

			final VTDNav nav = parser.getNav();
			final AutoPilot autoPilot = new AutoPilot(nav);
			return VtdUtilityParser.getTextValuesWithAttributes(autoPilot, nav, DATA_XPATH, ATTRS)
					.stream()
					.map(this::asRow)
					.collect(Collectors.toList());
		} catch (Exception e) {
			log.error("Error parsing xml:\n" + xml, e);
			throw new RuntimeException("Unable to parse document:\n" + xml, e);
		}
	}

	private Row asRow(final Node base64Node) {
		final String base64 = base64Node.getAttributes().get("value");
		if (StringUtils.isBlank(base64))
			throw new IllegalArgumentException("empty protocolbuffer value, check xslt");

		final DNGFDecoder d = DNGFDecoder.decode(Base64.decodeBase64(base64));
		final String qualifier = getQualifier(base64Node, d);
		return new Row(d.getColumnFamily(), d.getRowkey(), Arrays.asList(new Column<>(qualifier, d.getDNGF().toByteArray())));
	}

	private String getQualifier(final Node base64Node, final DNGFDecoder d) {
		if("update".equals(base64Node.getAttributes().get("mode"))) {
			return "update_" + System.nanoTime();
		}
		return d.getQualifier();
	}

	public String getTransformerClassName() {
		return transformer != null ? transformer.getClass().getName() : "unknown";
	}

	@Override
	public List<Row> apply(final String xml) {
		return transform(xml);
	}

}
