package eu.dnetlib.data.graph.utils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import org.apache.commons.lang3.StringUtils;

public class RelDescriptor {

    public static final String ONTOLOGY_SEPARATOR = "_";

	public static final String QUALIFIER_SEPARATOR = ":";

	public static final String RELTYPE_REGEX = "(\\w+_\\w+)_(\\w+)";
	public static final String ID_REGEX = "([0-9][0-9]\\|.{12}::[a-zA-Z0-9]{32})";

	public static final Pattern QUALIFIER_PATTERN = Pattern.compile("^" + RELTYPE_REGEX + "$");
	public static final Pattern FULL_QUALIFIER_PATTERN = Pattern.compile("^" + RELTYPE_REGEX + QUALIFIER_SEPARATOR + ID_REGEX + "$");

    private String code;

	private String ontologyCode;

	private String termCode;

	private String targetId = null;

    public RelDescriptor(final String ontologyCode, final String termCode) {
        this.ontologyCode = ontologyCode;
        this.termCode = termCode;
        this.code = ontologyCode + ONTOLOGY_SEPARATOR + termCode;
    }

	public RelDescriptor(final String ontologyCode, final String termCode, final String targetId) {
		this(ontologyCode, termCode);
		this.targetId = targetId;
	}

	public RelDescriptor(final String value) {
		if (StringUtils.isBlank(value)) {
			throw new IllegalArgumentException("empty value is not permitted");
		}

		final Matcher matcher = QUALIFIER_PATTERN.matcher(value);
		if (matcher.matches()) {
			this.code = value;
			this.ontologyCode = matcher.group(1);
			this.termCode = matcher.group(2);
		} else {
			final Matcher fullMatcher = FULL_QUALIFIER_PATTERN.matcher(value);
			if (fullMatcher.matches()) {
				this.code = fullMatcher.group(1) + ONTOLOGY_SEPARATOR + fullMatcher.group(2);
				this.ontologyCode = fullMatcher.group(1);
				this.termCode = fullMatcher.group(2);
				this.targetId = fullMatcher.group(3);
			} else {
				throw new IllegalArgumentException(String.format("invalid qualifier format: '%s'", value));
			}
		}
    }

    public String shortQualifier() {
    	return getOntologyCode() + ONTOLOGY_SEPARATOR + getTermCode();
    }

    public String qualifier() {
		return qualifier(getTargetId());
    }

	public String qualifier(final String targetId) {
		if (StringUtils.isBlank(targetId)) {
			throw new IllegalStateException("incomplete qualifier");
		}

		return shortQualifier() + QUALIFIER_SEPARATOR + targetId;
	}

	public static String asString(final Qualifier qualifier) {
    	if (qualifier == null) {
		    throw new IllegalArgumentException("invalid qualifier");
	    }
    	return qualifier.getSchemeid() + ONTOLOGY_SEPARATOR + qualifier.getClassid();
	}

	public Qualifier asQualifier() {
		return Qualifier.newBuilder()
				.setClassid(getTermCode()).setClassname(getTermCode())
				.setSchemeid(getOntologyCode()).setSchemename(getOntologyCode()).build();
	}

	public String getCode() {
		return code;
	}

	public String getOntologyCode() {
		return ontologyCode;
	}

	public String getTermCode() {
		return termCode;
	}

	@Override
	public String toString() {
		return getCode();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + ((code == null) ? 0 : code.hashCode());
		return result;
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) return true;
		if (obj == null) return false;
		if (getClass() != obj.getClass()) return false;
		RelDescriptor other = (RelDescriptor) obj;
		if (code == null) {
			if (other.code != null) return false;
		} else if (!code.equals(other.code)) return false;
		return true;
	}

	public String getTargetId() {
		return targetId;
	}

	public void setTargetId(final String targetId) {
		this.targetId = targetId;
	}
}
