/**
 * (c) 2014 FAO / UN (project: fi-security-server)
 */
package org.fao.fi.security.common.support.token.impl;

import javax.inject.Inject;

import org.fao.fi.security.common.encryption.pgp.PGPDecryptorBean;
import org.fao.fi.security.common.encryption.pgp.PGPEncryptorBean;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringAccessException;
import org.fao.fi.security.common.encryption.pgp.exceptions.MissingOrBadKeyringException;
import org.fao.fi.security.common.services.exceptions.token.TokenProcessingException;
import org.fao.fi.security.common.utilities.EncryptionUtils;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
public class PGPEncryptedTokenProcessor extends AbstractTokenProcessor {
	private PGPDecryptorBean _pgpDecryptor;
	private PGPEncryptorBean _pgpEncryptor;
	
	/**
	 * Class constructor
	 *
	 * @param pgpDecryptor
	 * @param pgpEncryptor
	 */
	@Inject public PGPEncryptedTokenProcessor(PGPDecryptorBean pgpDecryptor, PGPEncryptorBean pgpEncryptor) {
		super();
		this._pgpDecryptor = pgpDecryptor;
		this._pgpEncryptor = pgpEncryptor;
	}

	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.token.spi.TokenProcessor#process(java.lang.String)
	 */
	@Override
	public String processAfterCreation(String token) throws TokenProcessingException {
		try {
			return token == null ? null : new String(EncryptionUtils.toBase64(this._pgpEncryptor.encryptBytes(token.getBytes("UTF-8"))));
		} catch(KeyringAccessException KAe) {
			this._log.error("Processing token after creation yield a keyring access error: {}", KAe.getMessage(), KAe);
			
			throw new TokenProcessingException("Token can't be processed due to encryption / decryption problems (see log for further details)");
		} catch(MissingOrBadKeyringException MOBKe) {
			this._log.error("Processing token after creation yield a bad keyring error: {}", MOBKe.getMessage(), MOBKe);
			
			throw new TokenProcessingException("Token can't be processed due to keyring problems (see log for further details)");
		} catch(Throwable t) {
			this._log.error("Processing token after creation yield an error: {}", t.getMessage(), t);
			
			throw new TokenProcessingException("Token can't be processed correctly after creation");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.providers.token.spi.TokenProcessor#processBeforeConsumption(java.lang.String)
	 */
	@Override
	public String processBeforeConsumption(String token) throws TokenProcessingException {
		try {
			return token == null ? null : new String(this._pgpDecryptor.decryptBytes(EncryptionUtils.fromBase64(token)), "UTF-8");
		} catch(Throwable t) {
			this._log.error("Processing token before consumption yield an error: {}", t.getMessage(), t);
			
			throw new TokenProcessingException("Token can't be processed correctly before consumption");
		}
	}
}
