/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment.file;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.plugins.blob.BlobStoreBlob;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.Segment;
import org.apache.jackrabbit.oak.plugins.segment.SegmentId;
import org.apache.jackrabbit.oak.plugins.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.plugins.segment.SegmentStore;
import org.apache.jackrabbit.oak.plugins.segment.SegmentTracker;
import org.apache.jackrabbit.oak.plugins.segment.file.TarReader;
import org.apache.jackrabbit.oak.plugins.segment.file.TarWriter;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
implements SegmentStore {
    private static final Logger log = LoggerFactory.getLogger(FileStore.class);
    private static final int MB = 0x100000;
    private static final Pattern FILE_NAME_PATTERN = Pattern.compile("(data|bulk)((0|[1-9][0-9]*)[0-9]{4})([a-z])?.tar");
    private static final String FILE_NAME_FORMAT = "data%05d%s.tar";
    private static final String JOURNAL_FILE_NAME = "journal.log";
    private static final boolean MEMORY_MAPPING_DEFAULT = "64".equals(System.getProperty("sun.arch.data.model", "32"));
    private final SegmentTracker tracker;
    private final File directory;
    private final BlobStore blobStore;
    private final int maxFileSize;
    private final boolean memoryMapping;
    private volatile List<TarReader> readers;
    private int writeNumber;
    private File writeFile;
    private TarWriter writer;
    private final RandomAccessFile journalFile;
    private final FileLock journalLock;
    private final AtomicReference<RecordId> head;
    private final AtomicReference<RecordId> persistedHead;
    private final Thread flushThread;
    private final AtomicBoolean cleanupNeeded = new AtomicBoolean(false);
    private final LinkedList<File> toBeRemoved = Lists.newLinkedList();
    private final CountDownLatch timeToClose = new CountDownLatch(1);

    public FileStore(BlobStore blobStore, File directory, int maxFileSizeMB, boolean memoryMapping) throws IOException {
        this(blobStore, directory, EmptyNodeState.EMPTY_NODE, maxFileSizeMB, 0, memoryMapping);
    }

    public FileStore(File directory, int maxFileSizeMB, boolean memoryMapping) throws IOException {
        this(null, directory, maxFileSizeMB, memoryMapping);
    }

    public FileStore(File directory, int maxFileSizeMB) throws IOException {
        this(null, directory, maxFileSizeMB, MEMORY_MAPPING_DEFAULT);
    }

    public FileStore(File directory, int maxFileSizeMB, int cacheSizeMB, boolean memoryMapping) throws IOException {
        this(null, directory, EmptyNodeState.EMPTY_NODE, maxFileSizeMB, cacheSizeMB, memoryMapping);
    }

    public FileStore(BlobStore blobStore, final File directory, NodeState initial, int maxFileSizeMB, int cacheSizeMB, boolean memoryMapping) throws IOException {
        Preconditions.checkNotNull(directory).mkdirs();
        this.tracker = cacheSizeMB > 0 ? new SegmentTracker(this, cacheSizeMB) : new SegmentTracker(this);
        this.blobStore = blobStore;
        this.directory = directory;
        this.maxFileSize = maxFileSizeMB * 0x100000;
        this.memoryMapping = memoryMapping;
        this.journalFile = new RandomAccessFile(new File(directory, JOURNAL_FILE_NAME), "rw");
        this.journalLock = this.journalFile.getChannel().lock();
        Map<Integer, Map<Character, File>> map = FileStore.collectFiles(directory);
        this.readers = Lists.newArrayListWithCapacity(map.size());
        Object[] indices = map.keySet().toArray(new Integer[map.size()]);
        Arrays.sort(indices);
        for (int i = indices.length - 1; i >= 0; --i) {
            this.readers.add(TarReader.open(map.get(indices[i]), memoryMapping));
        }
        this.writeNumber = indices.length > 0 ? (Integer)indices[indices.length - 1] + 1 : 0;
        this.writeFile = new File(directory, String.format(FILE_NAME_FORMAT, this.writeNumber, "a"));
        this.writer = new TarWriter(this.writeFile);
        LinkedList<RecordId> heads = Lists.newLinkedList();
        String line = this.journalFile.readLine();
        while (line != null) {
            int space = line.indexOf(32);
            if (space != -1) {
                heads.add(RecordId.fromString(this.tracker, line.substring(0, space)));
            }
            line = this.journalFile.readLine();
        }
        RecordId id = null;
        while (id == null && !heads.isEmpty()) {
            RecordId last = (RecordId)heads.removeLast();
            SegmentId segmentId = last.getSegmentId();
            if (this.containsSegment(segmentId.getMostSignificantBits(), segmentId.getLeastSignificantBits())) {
                id = last;
                continue;
            }
            log.warn("Unable to access revision {}, rewinding...", (Object)last);
        }
        if (id != null) {
            this.head = new AtomicReference<Object>(id);
            this.persistedHead = new AtomicReference<Object>(id);
        } else {
            NodeBuilder builder = EmptyNodeState.EMPTY_NODE.builder();
            builder.setChildNode("root", initial);
            this.head = new AtomicReference<RecordId>(this.tracker.getWriter().writeNode(builder.getNodeState()).getRecordId());
            this.persistedHead = new AtomicReference<Object>(null);
        }
        this.flushThread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    FileStore.this.timeToClose.await(1L, TimeUnit.SECONDS);
                    while (FileStore.this.timeToClose.getCount() > 0L) {
                        long start = System.nanoTime();
                        try {
                            FileStore.this.flush();
                        }
                        catch (IOException e) {
                            log.warn("Failed to flush the TarMK at" + directory, (Throwable)e);
                        }
                        long time = TimeUnit.SECONDS.convert(System.nanoTime() - start, TimeUnit.NANOSECONDS);
                        FileStore.this.timeToClose.await(Math.max(5L, 2L * time), TimeUnit.SECONDS);
                    }
                }
                catch (InterruptedException e) {
                    log.warn("TarMK flush thread interrupted");
                }
            }
        });
        this.flushThread.setName("TarMK flush thread: " + directory);
        this.flushThread.setDaemon(true);
        this.flushThread.setPriority(1);
        this.flushThread.start();
        log.info("TarMK opened: {} (mmap={})", (Object)directory, (Object)memoryMapping);
    }

    static Map<Integer, Map<Character, File>> collectFiles(File directory) throws IOException {
        Map<Character, File> files;
        HashMap<Integer, Map<Character, File>> dataFiles = Maps.newHashMap();
        HashMap<Integer, File> bulkFiles = Maps.newHashMap();
        for (File file : directory.listFiles()) {
            Matcher matcher = FILE_NAME_PATTERN.matcher(file.getName());
            if (!matcher.matches()) continue;
            Object index = Integer.parseInt(matcher.group(2));
            if ("data".equals(matcher.group(1))) {
                files = (HashMap<Character, File>)dataFiles.get(index);
                if (files == null) {
                    files = Maps.newHashMap();
                    dataFiles.put((Integer)index, files);
                }
                Character generation = Character.valueOf('a');
                if (matcher.group(4) != null) {
                    generation = Character.valueOf(matcher.group(4).charAt(0));
                }
                Preconditions.checkState(files.put(generation, file) == null);
                continue;
            }
            Preconditions.checkState(bulkFiles.put((Integer)index, file) == null);
        }
        if (!bulkFiles.isEmpty()) {
            Integer newIndex;
            int position;
            Object[] indices;
            log.info("Upgrading TarMK file names in {}", (Object)directory);
            if (!dataFiles.isEmpty()) {
                indices = dataFiles.keySet().toArray(new Integer[dataFiles.size()]);
                Arrays.sort(indices);
                position = Math.max((Integer)indices[indices.length - 1] + 1, bulkFiles.size());
                for (Object index : indices) {
                    files = (Map)dataFiles.remove(index);
                    newIndex = position++;
                    for (Character generation : Sets.newHashSet(files.keySet())) {
                        File file = (File)files.get(generation);
                        File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, generation));
                        log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                        file.renameTo(newFile);
                        files.put(generation, newFile);
                    }
                    dataFiles.put(newIndex, files);
                }
            }
            indices = bulkFiles.keySet().toArray(new Integer[bulkFiles.size()]);
            Arrays.sort(indices);
            position = 0;
            for (Object index : indices) {
                File file = (File)bulkFiles.remove(index);
                newIndex = position++;
                File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, "a"));
                log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                file.renameTo(newFile);
                dataFiles.put(newIndex, Collections.singletonMap(Character.valueOf('a'), newFile));
            }
        }
        return dataFiles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush() throws IOException {
        AtomicReference<RecordId> atomicReference = this.persistedHead;
        synchronized (atomicReference) {
            RecordId before = this.persistedHead.get();
            RecordId after = this.head.get();
            boolean cleanup = this.cleanupNeeded.getAndSet(false);
            if (cleanup || !after.equals(before)) {
                this.tracker.getWriter().flush();
                this.writer.flush();
                FileStore fileStore = this;
                synchronized (fileStore) {
                    log.debug("TarMK journal update {} -> {}", (Object)before, (Object)after);
                    this.journalFile.writeBytes(after + " root\n");
                    this.journalFile.getChannel().force(false);
                    this.persistedHead.set(after);
                    if (cleanup) {
                        long start = System.nanoTime();
                        HashSet<UUID> ids = Sets.newHashSet();
                        for (SegmentId id : this.tracker.getReferencedSegmentIds()) {
                            ids.add(new UUID(id.getMostSignificantBits(), id.getLeastSignificantBits()));
                        }
                        this.writer.cleanup(ids);
                        ArrayList<TarReader> list = Lists.newArrayListWithCapacity(this.readers.size());
                        for (TarReader reader : this.readers) {
                            TarReader cleaned = reader.cleanup(ids);
                            if (cleaned == reader) {
                                list.add(reader);
                                continue;
                            }
                            if (cleaned != null) {
                                list.add(cleaned);
                            }
                            File file = reader.close();
                            log.info("TarMK GC: Cleaned up file {}", (Object)file);
                            this.toBeRemoved.addLast(file);
                        }
                        this.readers = list;
                        log.debug("TarMK GC: Completed in {}ms", (Object)TimeUnit.MILLISECONDS.convert(System.nanoTime() - start, TimeUnit.NANOSECONDS));
                    }
                }
                Iterator iterator = this.toBeRemoved.iterator();
                while (iterator.hasNext()) {
                    File file = (File)iterator.next();
                    if (file.exists() && !file.delete()) continue;
                    log.debug("TarMK GC: Removed old file {}", (Object)file);
                    iterator.remove();
                }
            }
        }
    }

    public synchronized Iterable<SegmentId> getSegmentIds() {
        ArrayList<SegmentId> ids = Lists.newArrayList();
        for (UUID uuid : this.writer.getUUIDs()) {
            ids.add(this.tracker.getSegmentId(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits()));
        }
        for (TarReader reader : this.readers) {
            for (UUID uuid : reader.getUUIDs()) {
                ids.add(this.tracker.getSegmentId(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits()));
            }
        }
        return ids;
    }

    @Override
    public SegmentTracker getTracker() {
        return this.tracker;
    }

    @Override
    public SegmentNodeState getHead() {
        return new SegmentNodeState(this.head.get());
    }

    @Override
    public boolean setHead(SegmentNodeState base, SegmentNodeState head) {
        RecordId id = this.head.get();
        return id.equals(base.getRecordId()) && this.head.compareAndSet(id, head.getRecordId());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        try {
            this.timeToClose.countDown();
            try {
                this.flushThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.warn("Interrupted while joining the TarMK flush thread", (Throwable)e);
            }
            FileStore e = this;
            synchronized (e) {
                this.flush();
                this.writer.close();
                List<TarReader> list = this.readers;
                this.readers = Lists.newArrayList();
                for (TarReader reader : list) {
                    reader.close();
                }
                this.journalLock.release();
                this.journalFile.close();
                System.gc();
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to close the TarMK at " + this.directory, e);
        }
        log.info("TarMK closed: {}", (Object)this.directory);
    }

    @Override
    public boolean containsSegment(SegmentId id) {
        if (id.getTracker() == this.tracker) {
            return true;
        }
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        return this.containsSegment(msb, lsb);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean containsSegment(long msb, long lsb) {
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        Iterator<TarReader> i$ = this;
        synchronized (i$) {
            if (this.writer.containsEntry(msb, lsb)) {
                return true;
            }
        }
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Segment readSegment(SegmentId id) {
        ByteBuffer buffer;
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        for (TarReader reader : this.readers) {
            try {
                buffer = reader.readEntry(msb, lsb);
                if (buffer == null) continue;
                return new Segment(this.tracker, id, buffer);
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + reader, (Throwable)e);
            }
        }
        Iterator<TarReader> i$ = this;
        synchronized (i$) {
            try {
                ByteBuffer buffer2 = this.writer.readEntry(msb, lsb);
                if (buffer2 != null) {
                    return new Segment(this.tracker, id, buffer2);
                }
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + this.writer, (Throwable)e);
            }
        }
        for (TarReader reader : this.readers) {
            try {
                buffer = reader.readEntry(msb, lsb);
                if (buffer == null) continue;
                return new Segment(this.tracker, id, buffer);
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + reader, (Throwable)e);
            }
        }
        throw new IllegalStateException("Segment " + id + " not found");
    }

    @Override
    public synchronized void writeSegment(SegmentId id, byte[] data, int offset, int length) {
        try {
            long size = this.writer.writeEntry(id.getMostSignificantBits(), id.getLeastSignificantBits(), data, offset, length);
            if (size >= (long)this.maxFileSize) {
                this.writer.close();
                ArrayList<TarReader> list = Lists.newArrayListWithCapacity(1 + this.readers.size());
                list.add(TarReader.open(this.writeFile, this.memoryMapping));
                list.addAll(this.readers);
                this.readers = list;
                this.cleanupNeeded.set(true);
                ++this.writeNumber;
                this.writeFile = new File(this.directory, String.format(FILE_NAME_FORMAT, this.writeNumber, "a"));
                this.writer = new TarWriter(this.writeFile);
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Blob readBlob(String blobId) {
        if (this.blobStore != null) {
            return new BlobStoreBlob(this.blobStore, blobId);
        }
        throw new IllegalStateException("Attempt to read external blob with blobId [" + blobId + "] " + "without specifying BlobStore");
    }

    @Override
    public BlobStore getBlobStore() {
        return this.blobStore;
    }

    @Override
    public void gc() {
        System.gc();
        this.cleanupNeeded.set(true);
    }
}

