/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.security.authentication.token;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.jcr.AccessDeniedException;
import javax.jcr.Credentials;
import javax.jcr.RepositoryException;
import javax.jcr.SimpleCredentials;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Root;
import org.apache.jackrabbit.oak.api.Tree;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.namepath.NamePathMapper;
import org.apache.jackrabbit.oak.plugins.identifier.IdentifierManager;
import org.apache.jackrabbit.oak.plugins.name.NamespaceConstants;
import org.apache.jackrabbit.oak.spi.security.ConfigurationParameters;
import org.apache.jackrabbit.oak.spi.security.authentication.ImpersonationCredentials;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenInfo;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenProvider;
import org.apache.jackrabbit.oak.spi.security.user.UserConfiguration;
import org.apache.jackrabbit.oak.spi.security.user.util.PasswordUtil;
import org.apache.jackrabbit.oak.util.NodeUtil;
import org.apache.jackrabbit.oak.util.TreeUtil;
import org.apache.jackrabbit.util.ISO8601;
import org.apache.jackrabbit.util.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TokenProviderImpl
implements TokenProvider {
    private static final Logger log = LoggerFactory.getLogger(TokenProviderImpl.class);
    private static final String TOKEN_ATTRIBUTE = ".token";
    private static final String TOKEN_ATTRIBUTE_EXPIRY = "rep:token.exp";
    private static final String TOKEN_ATTRIBUTE_KEY = "rep:token.key";
    private static final String TOKENS_NODE_NAME = ".tokens";
    private static final String TOKENS_NT_NAME = "rep:Unstructured";
    private static final String TOKEN_NT_NAME = "rep:Token";
    private static final long DEFAULT_TOKEN_EXPIRATION = 0x6DDD00L;
    private static final int DEFAULT_KEY_SIZE = 8;
    private static final char DELIM = '_';
    private static final Set<String> RESERVED_ATTRIBUTES = new HashSet<String>(3);
    private final Root root;
    private final ConfigurationParameters options;
    private final long tokenExpiration;
    private final UserManager userManager;
    private final IdentifierManager identifierManager;

    TokenProviderImpl(Root root, ConfigurationParameters options, UserConfiguration userConfiguration) {
        this.root = root;
        this.options = options;
        this.tokenExpiration = options.getConfigValue("tokenExpiration", 0x6DDD00L);
        this.userManager = userConfiguration.getUserManager(root, NamePathMapper.DEFAULT);
        this.identifierManager = new IdentifierManager(root);
    }

    @Override
    public boolean doCreateToken(Credentials credentials) {
        SimpleCredentials sc = TokenProviderImpl.extractSimpleCredentials(credentials);
        if (sc == null) {
            return false;
        }
        Object attr = sc.getAttribute(TOKEN_ATTRIBUTE);
        return attr != null && "".equals(attr.toString());
    }

    @Override
    public TokenInfo createToken(Credentials credentials) {
        SimpleCredentials sc = TokenProviderImpl.extractSimpleCredentials(credentials);
        TokenInfo tokenInfo = null;
        if (sc != null) {
            String[] attrNames = sc.getAttributeNames();
            HashMap<String, String> attributes = new HashMap<String, String>(attrNames.length);
            for (String attrName : sc.getAttributeNames()) {
                attributes.put(attrName, sc.getAttribute(attrName).toString());
            }
            tokenInfo = this.createToken(sc.getUserID(), attributes);
            if (tokenInfo != null) {
                sc.setAttribute(TOKEN_ATTRIBUTE, tokenInfo.getToken());
            }
        }
        return tokenInfo;
    }

    @Override
    public TokenInfo createToken(String userId, Map<String, ?> attributes) {
        String error = "Failed to create login token. ";
        NodeUtil tokenParent = this.getTokenParent(userId);
        if (tokenParent != null) {
            try {
                long creationTime = new Date().getTime();
                Calendar creation = GregorianCalendar.getInstance();
                creation.setTimeInMillis(creationTime);
                String tokenName = Text.replace(ISO8601.format(creation), ":", ".");
                NodeUtil tokenNode = tokenParent.addChild(tokenName, TOKEN_NT_NAME);
                tokenNode.setString("jcr:uuid", IdentifierManager.generateUUID());
                String key = TokenProviderImpl.generateKey(this.options.getConfigValue("tokenLength", 8));
                String nodeId = IdentifierManager.getIdentifier(tokenNode.getTree());
                String token = nodeId + '_' + key;
                String keyHash = PasswordUtil.buildPasswordHash(TokenProviderImpl.getKeyValue(key, userId));
                tokenNode.setString(TOKEN_ATTRIBUTE_KEY, keyHash);
                long exp = attributes.containsKey("tokenExpiration") ? Long.parseLong(attributes.get("tokenExpiration").toString()) : this.tokenExpiration;
                long expTime = TokenProviderImpl.createExpirationTime(creationTime, exp);
                tokenNode.setDate(TOKEN_ATTRIBUTE_EXPIRY, expTime);
                for (String name : attributes.keySet()) {
                    if (RESERVED_ATTRIBUTES.contains(name)) continue;
                    String attr = attributes.get(name).toString();
                    tokenNode.setString(name, attr);
                }
                this.root.commit();
                return new TokenInfoImpl(tokenNode, token, userId);
            }
            catch (NoSuchAlgorithmException e) {
                log.error(error, (Object)e.getMessage());
            }
            catch (UnsupportedEncodingException e) {
                log.error(error, (Object)e.getMessage());
            }
            catch (CommitFailedException e) {
                log.warn(error, (Object)e.getMessage());
            }
            catch (AccessDeniedException e) {
                log.warn(error, (Object)e.getMessage());
            }
        } else {
            log.warn("Unable to get/create token store for user " + userId);
        }
        return null;
    }

    @Override
    public TokenInfo getTokenInfo(String token) {
        int pos = token.indexOf(95);
        String nodeId = pos == -1 ? token : token.substring(0, pos);
        Tree tokenTree = this.identifierManager.getTree(nodeId);
        String userId = this.getUserId(tokenTree);
        if (userId == null || !TokenProviderImpl.isValidTokenTree(tokenTree)) {
            return null;
        }
        return new TokenInfoImpl(new NodeUtil(tokenTree), token, userId);
    }

    private static long createExpirationTime(long creationTime, long tokenExpiration) {
        return creationTime + tokenExpiration;
    }

    private static long getExpirationTime(NodeUtil tokenNode, long defaultValue) {
        return tokenNode.getLong(TOKEN_ATTRIBUTE_EXPIRY, defaultValue);
    }

    @CheckForNull
    private static SimpleCredentials extractSimpleCredentials(Credentials credentials) {
        Credentials base;
        if (credentials instanceof SimpleCredentials) {
            return (SimpleCredentials)credentials;
        }
        if (credentials instanceof ImpersonationCredentials && (base = ((ImpersonationCredentials)credentials).getBaseCredentials()) instanceof SimpleCredentials) {
            return (SimpleCredentials)base;
        }
        return null;
    }

    @Nonnull
    private static String generateKey(int size) {
        SecureRandom random = new SecureRandom();
        byte[] key = new byte[size];
        random.nextBytes(key);
        StringBuilder res = new StringBuilder(key.length * 2);
        for (byte b : key) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    @Nonnull
    private static String getKeyValue(String key, String userId) {
        return key + userId;
    }

    private static boolean isValidTokenTree(Tree tokenTree) {
        if (tokenTree == null || !tokenTree.exists()) {
            return false;
        }
        return TOKENS_NODE_NAME.equals(tokenTree.getParent().getName()) && TOKEN_NT_NAME.equals(TreeUtil.getPrimaryTypeName(tokenTree));
    }

    @CheckForNull
    private Tree getTokenTree(TokenInfo tokenInfo) {
        if (tokenInfo instanceof TokenInfoImpl) {
            return this.root.getTree(((TokenInfoImpl)tokenInfo).tokenPath);
        }
        return null;
    }

    @CheckForNull
    private String getUserId(Tree tokenTree) {
        if (tokenTree != null && tokenTree.exists()) {
            try {
                String userPath = Text.getRelativeParent(tokenTree.getPath(), 2);
                Authorizable authorizable = this.userManager.getAuthorizableByPath(userPath);
                if (authorizable != null && !authorizable.isGroup() && !((User)authorizable).isDisabled()) {
                    return authorizable.getID();
                }
            }
            catch (RepositoryException e) {
                log.debug("Cannot determine userID from token: ", (Object)e.getMessage());
            }
        }
        return null;
    }

    @CheckForNull
    private NodeUtil getTokenParent(String userId) {
        NodeUtil tokenParent;
        block6: {
            tokenParent = null;
            String parentPath = null;
            try {
                Authorizable user = this.userManager.getAuthorizable(userId);
                if (user != null && !user.isGroup()) {
                    String userPath = user.getPath();
                    NodeUtil userNode = new NodeUtil(this.root.getTree(userPath));
                    tokenParent = userNode.getChild(TOKENS_NODE_NAME);
                    if (tokenParent == null) {
                        tokenParent = userNode.addChild(TOKENS_NODE_NAME, TOKENS_NT_NAME);
                        parentPath = userPath + '/' + TOKENS_NODE_NAME;
                        this.root.commit();
                    }
                } else {
                    log.debug("Cannot create login token: No corresponding node for User " + userId + '.');
                }
            }
            catch (RepositoryException e) {
                log.debug("Error while accessing user " + userId + '.', (Throwable)e);
            }
            catch (CommitFailedException e) {
                Tree parentTree;
                log.debug("Conflict while creating token store -> retrying", (Object)e.getMessage());
                this.root.refresh();
                if (parentPath == null || !(parentTree = this.root.getTree(parentPath)).exists()) break block6;
                tokenParent = new NodeUtil(parentTree);
            }
        }
        return tokenParent;
    }

    static {
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE);
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE_EXPIRY);
        RESERVED_ATTRIBUTES.add(TOKEN_ATTRIBUTE_KEY);
    }

    private final class TokenInfoImpl
    implements TokenInfo {
        private final String token;
        private final String tokenPath;
        private final String userId;
        private final long expirationTime;
        private final String key;
        private final Map<String, String> mandatoryAttributes;
        private final Map<String, String> publicAttributes;

        private TokenInfoImpl(NodeUtil tokenNode, String token, String userId) {
            this.token = token;
            this.tokenPath = tokenNode.getTree().getPath();
            this.userId = userId;
            this.expirationTime = TokenProviderImpl.getExpirationTime(tokenNode, Long.MIN_VALUE);
            this.key = tokenNode.getString(TokenProviderImpl.TOKEN_ATTRIBUTE_KEY, null);
            this.mandatoryAttributes = new HashMap<String, String>();
            this.publicAttributes = new HashMap<String, String>();
            for (PropertyState propertyState : tokenNode.getTree().getProperties()) {
                String name = propertyState.getName();
                String value = propertyState.getValue(Type.STRING);
                if (RESERVED_ATTRIBUTES.contains(name)) continue;
                if (this.isMandatoryAttribute(name)) {
                    this.mandatoryAttributes.put(name, value);
                    continue;
                }
                if (!this.isInfoAttribute(name)) continue;
                this.publicAttributes.put(name, value);
            }
        }

        @Override
        public String getUserId() {
            return this.userId;
        }

        @Override
        public String getToken() {
            return this.token;
        }

        @Override
        public boolean isExpired(long loginTime) {
            return this.expirationTime < loginTime;
        }

        @Override
        public boolean resetExpiration(long loginTime) {
            Tree tokenTree = TokenProviderImpl.this.getTokenTree(this);
            if (tokenTree != null && tokenTree.exists()) {
                NodeUtil tokenNode = new NodeUtil(tokenTree);
                if (this.isExpired(loginTime)) {
                    log.debug("Attempt to reset an expired token.");
                    return false;
                }
                if (this.expirationTime - loginTime <= TokenProviderImpl.this.tokenExpiration / 2L) {
                    try {
                        long expTime = TokenProviderImpl.createExpirationTime(loginTime, TokenProviderImpl.this.tokenExpiration);
                        tokenNode.setDate(TokenProviderImpl.TOKEN_ATTRIBUTE_EXPIRY, expTime);
                        TokenProviderImpl.this.root.commit();
                        log.debug("Successfully reset token expiration time.");
                        return true;
                    }
                    catch (CommitFailedException e) {
                        log.debug("Failed to reset token expiration", (Object)e.getMessage());
                        TokenProviderImpl.this.root.refresh();
                    }
                }
            }
            return false;
        }

        @Override
        public boolean remove() {
            Tree tokenTree = TokenProviderImpl.this.getTokenTree(this);
            if (tokenTree != null && tokenTree.exists()) {
                try {
                    if (tokenTree.remove()) {
                        TokenProviderImpl.this.root.commit();
                        return true;
                    }
                }
                catch (CommitFailedException e) {
                    log.debug("Error while removing expired token", (Object)e.getMessage());
                }
            }
            return false;
        }

        @Override
        public boolean matches(TokenCredentials tokenCredentials) {
            String tk = tokenCredentials.getToken();
            int pos = tk.lastIndexOf(95);
            if (pos > -1) {
                tk = tk.substring(pos + 1);
            }
            if (this.key == null || !PasswordUtil.isSame(this.key, TokenProviderImpl.getKeyValue(tk, this.userId))) {
                return false;
            }
            for (String name : this.mandatoryAttributes.keySet()) {
                String expectedValue = this.mandatoryAttributes.get(name);
                if (expectedValue.equals(tokenCredentials.getAttribute(name))) continue;
                return false;
            }
            List<String> attrNames = Arrays.asList(tokenCredentials.getAttributeNames());
            for (String name : this.publicAttributes.keySet()) {
                if (attrNames.contains(name)) continue;
                tokenCredentials.setAttribute(name, this.publicAttributes.get(name).toString());
            }
            return true;
        }

        @Override
        public Map<String, String> getPrivateAttributes() {
            return Collections.unmodifiableMap(this.mandatoryAttributes);
        }

        @Override
        public Map<String, String> getPublicAttributes() {
            return Collections.unmodifiableMap(this.publicAttributes);
        }

        private boolean isMandatoryAttribute(String attributeName) {
            return attributeName != null && attributeName.startsWith(TokenProviderImpl.TOKEN_ATTRIBUTE);
        }

        private boolean isInfoAttribute(String attributeName) {
            String prefix = Text.getNamespacePrefix(attributeName);
            return !NamespaceConstants.RESERVED_PREFIXES.contains(prefix);
        }
    }
}

