package org.gcube.accounting.messaging.producer;


import java.util.Random;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.messaging.ConnectionsManager;
import org.gcube.common.messaging.endpoints.BrokerNotConfiguredInScopeException;




/**
 * JMS Client that sends message to ActiveMQ Broker
 * 
 * @author Andrea Manzi (CERN)
 * @author Ermanno Travaglino (E-IIS)
 *
 */
public class Producer implements Runnable{

	private static int ackMode;
	private static boolean transacted;
	
	private static String queueName = "resource-accounting";
	private static Producer singleton; 
	private static Integer MAX_ACCOUNTING_QUEUE_SIZE = 1000;
	private static  ConcurrentLinkedQueue<RawUsageRecord> messageForQueue = null;
	private static QueueConnection connection;
	private static Logger logger = LoggerFactory.getLogger(Producer.class);
	
	static {
		ackMode = Session.AUTO_ACKNOWLEDGE;
		transacted = false;
		singleton = new Producer();
		messageForQueue = new ConcurrentLinkedQueue<RawUsageRecord>();
		//starting thread
		Thread t = new Thread(singleton);
		t.start();
	}

	/**
	 * no object instantiation possible
	 */
	private Producer() {}


	/**
	 * Sends message to  a QUEUE destination
	 * 	@param message the RawUsageRecord to send
	 */
	public synchronized void sendMessageToQueue(RawUsageRecord message) {

		String scope = message.getResourceScope();
		//logger.debug(scope, scopeBean.enclosingScope());
		logger.debug("current scope = "+scope);
		try {
			ConnectionsManager.addScope(scope);
		} catch (BrokerNotConfiguredInScopeException e) {
			logger.debug("Exception sending message to the Broker");
			enqueueMessageForQueue(message);
		} catch (Exception e1) {
			logger.debug("Exception sending message to the Broker 2");
			enqueueMessageForQueue(message);
		}
		connection = ConnectionsManager.getQueueConnection(scope);



		logger.debug("CONNECTION = "+connection);
		if (connection == null)
			ConnectionsManager.reloadConnection(scope);
		else{
			
			try {
				QueueSession session = connection.createQueueSession(transacted, ackMode);
				logger.debug("session created");
				Queue queue = session.createQueue(scope+"."+queueName);
				logger.debug(connection.getClientID());
				logger.debug("accounting queue has been created");
				QueueSender sender = session.createSender(queue);
				sender.setDeliveryMode(DeliveryMode.PERSISTENT);
				ObjectMessage objMsg = session.createObjectMessage();
				objMsg.setObject(message);
				objMsg.setJMSMessageID(createRandomString());
				sender.send(objMsg);
				logger.debug("Message "+message.toString()+ " SENT");
			} catch (JMSException e) {
				logger.debug("Exception sending message to the Broker");
				enqueueMessageForQueue(message);
			} catch (Exception e) {
				logger.debug("Exception sending message to the Broker");
				enqueueMessageForQueue(message);

			}
		}

	}

	/**
	 * Create a random Long
	 * @return a Random Long-String
	 */
	private  String createRandomString() {
		Random random = new Random(System.currentTimeMillis());
		long randomLong = random.nextLong();
		return Long.toHexString(randomLong);
	}

	public static Producer getSingleton() {
		return singleton;
	}


	/**
	 * 
	 * enqueue the message
	 * @param message the message
	 */
	private void enqueueMessageForQueue(RawUsageRecord message){
		try {
			synchronized (messageForQueue) {
				if (messageForQueue.size() >=MAX_ACCOUNTING_QUEUE_SIZE){
					logger.debug("Reached Maximum queue size, message discarded");
					logger.debug(message.toString());
				} else messageForQueue.add(message);	
			}
		}catch  (Exception e) {
			logger.debug("Error enqueuing Message : "+ message.toString());
		}

	}

	public void run() {
		int sizeQueue = 0;
		while (true){

			synchronized(messageForQueue){
				sizeQueue = messageForQueue.size();
			}


			if ((sizeQueue ==0)){
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) 
				{
					e.printStackTrace();
				}
			}
			else {
				if (sizeQueue >0){
					RawUsageRecord message = null;
					synchronized (messageForQueue) 
					{
						message = messageForQueue.poll();
					}
					this.sendMessageToQueue(message);
					try {
						Thread.sleep(100);
					} catch (InterruptedException e) 
					{
						e.printStackTrace();
					}
				}


			}
		}

	}


}