package org.gcube.application.enm.service;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.resumes.ExperimentResumes;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.service.concurrent.JobExecutionPool;
import org.gcube.application.enm.service.concurrent.JobMonitor;
import org.gcube.application.enm.service.concurrent.JobScheduler;
import org.gcube.application.enm.service.concurrent.ShutdownHook;
import org.gcube.application.enm.service.conn.InformationSystemClient;
import org.gcube.application.enm.service.conn.PersistenceClient;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Entry point of the Environmental Niche Modelling service. 
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class ENMService {	

	protected GCUBELog logger = new GCUBELog(ENMService.class);

	private final InformationSystemClient isClient = 
			new InformationSystemClient();
	private final PersistenceClient persistenceClient = new PersistenceClient();
	private final PluginLoader pluginLoader = new PluginLoader();

	public ENMService() {
		// Setup logging
		logger.trace("Constructor...");
		// Place a shutdown hook to terminate the thread pool and the scheduler 
		final ShutdownHook shutdownHook = new ShutdownHook();
		shutdownHook.attachShutDownHook();
		// Retrieve the state and initialize the job monitor
		JobMonitor.get().init(loadPendingJobs());
		// Start the thread pool and the scheduler
		JobExecutionPool.get().start();		
		JobScheduler.get().start();		
	}

	public String sumbitExperiment(final ExperimentRequest request) 
			throws RemoteException {
		String experimentId = "";
		// Create a new job depending on the request		
		try {
			final String pluginName = isClient.getJobPluginName(request);
			final GenericJob job = pluginLoader.getNewJobInstance(createUUID(), 
					request, pluginName);
			if (job != null) {
				if (JobMonitor.get().registerNewJob(job)) {
					JobExecutionPool.get().submit(job);
					experimentId = job.getUUID().toString();
				} else throw new IOException(
						"Cannot create a job for the experiment");
			} else throw new IllegalArgumentException(
					"Cannot find an execution plugin for the experiment");
		} catch (Exception e) {
			experimentId = "";
			throw new RemoteException(e.getLocalizedMessage());		
		}
		return experimentId;
	}

	public ExperimentStatus getStatus(final String experimentId) 
			throws RemoteException {
		ExperimentStatus status = null;
		final UUID uuid = getUUIDFromString(experimentId);
		final GenericJob job = JobMonitor.get().getJob(uuid);
		if (job != null)
			status = job.getStatus();
		return status;
	}

	public ExperimentResults getResults(final String experimentId) 
			throws RemoteException {
		ExperimentResults results = null;
		final UUID uuid = getUUIDFromString(experimentId);
		final GenericJob job = JobMonitor.get().getJob(uuid);
		if (job != null)
			results = job.getResults();
		return results;
	}

	public ExperimentLogs getLogs(final String experimentId) 
			throws RemoteException {
		ExperimentLogs logs = null;
		final UUID uuid = getUUIDFromString(experimentId);
		final GenericJob job = JobMonitor.get().getJob(uuid);
		if (job != null)
			logs = job.getLogs();
		return logs;
	}

	public void cancelExperiment(final String experimentId) 
			throws RemoteException {
		final UUID uuid = getUUIDFromString(experimentId);
		JobMonitor.get().cancelJob(uuid);
	}

	public ExperimentResumes getUserExperimentResumes(final String credentials) 
			throws RemoteException {
		try {
			return persistenceClient.getUserExperimentResumes(credentials);
		} catch (Exception e) {
			throw new RemoteException(e.getLocalizedMessage());
		}
	}

	public void removeExperiment(final String experimentId) 
			throws RemoteException {
		final UUID uuid = getUUIDFromString(experimentId);
		JobMonitor.get().removeJob(uuid);
	}

	private ConcurrentMap<UUID, GenericJob> loadPendingJobs() {
		final ConcurrentMap<UUID, GenericJob> jobsMap = 
				new ConcurrentHashMap<UUID, GenericJob>();
		persistenceClient.loadJobs(jobsMap);
		return jobsMap;
	}

	private UUID createUUID() {
		return UUID.randomUUID();
	}

	private UUID getUUIDFromString(final String experimentId) 
			throws RemoteException {
		UUID uuid = null;
		try {
			uuid = UUID.fromString(experimentId);
		} catch (Exception e) {
			throw new RemoteException(e.getLocalizedMessage());
		}
		return uuid;
	}

}
