package org.gcube.application.enm.service.concurrent;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.gcube.application.enm.service.GenericJob;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Schedules jobs to execute periodically.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class JobScheduler {

	protected GCUBELog logger = new GCUBELog(JobScheduler.class);

	private static final int NUM_WORKERS = 1;

	private static final int TIMEOUT_SECS = 20;

	public static final long INITIAL_DELAY = 10l;
	public static final long JOB_EXEC_DELAY = 30l;

	public static final int MAX_JOBS_PER_RUN = 1000;

	public static final long CLEAUNP_TIME_MILLIS = 3600000l;

	private long lastClean = System.currentTimeMillis() + CLEAUNP_TIME_MILLIS;

	private final ScheduledExecutorService pool;

	private AtomicBoolean isStarted = new AtomicBoolean(false);
	private AtomicBoolean shouldRun = new AtomicBoolean(true);

	/**
	 * The singleton instance of the scheduler.
	 */
	private static JobScheduler instance;

	/**
	 * Get the singleton instance of the scheduler.
	 * @return The singleton instance.
	 */
	public static JobScheduler get() {
		if (instance == null) {
			instance = new JobScheduler();
		}
		return instance;
	}

	/**
	 * Construct a new scheduler.
	 */
	private JobScheduler() {
		// Setup logging
		logger.trace("Constructor with " + NUM_WORKERS 
				+ " thread workers");
		// Create the scheduler
		pool = Executors.newScheduledThreadPool(NUM_WORKERS);
	}

	public void start() {
		if (!isStarted.get()) {
			isStarted.set(true);
			pool.scheduleWithFixedDelay(new Runnable() {			
				@Override
				public void run() {
					// Cleanup
					if (System.currentTimeMillis() - lastClean 
							> CLEAUNP_TIME_MILLIS) {
						JobMonitor.get().clean();
						lastClean = System.currentTimeMillis();
					}
					// Run next round
					final int jobCount = Math.min(JobMonitor.get()
							.numPendingJobs(), MAX_JOBS_PER_RUN);
					for (int i = 0; i < jobCount; i++) {
						// Run next job
						final GenericJob job = JobMonitor.get().nextPendingJob();
						JobExecutionPool.get().submit(job);
						logger.trace("Job '" + job.getUUID() + "' re-scheduled");
					}
				}
			}, INITIAL_DELAY, JOB_EXEC_DELAY, TimeUnit.SECONDS);
		}
	}

	public final void shutdownAndAwaitTermination() {
		shouldRun.set(false);
		pool.shutdown(); // Disable new jobs from being scheduled
		try {
			// Wait a while for existing jobs to terminate
			if (!pool.awaitTermination(TIMEOUT_SECS, TimeUnit.SECONDS)) {
				pool.shutdownNow(); // Cancel currently executing jobs
				// Wait a while for jobs to respond to being cancelled
				if (!pool.awaitTermination(TIMEOUT_SECS, TimeUnit.SECONDS))
					logger.warn("Scheduler did not terminate");
			}
		} catch (InterruptedException ie) {
			// (Re-)Cancel if current thread also interrupted
			pool.shutdownNow();
			// Preserve interrupt status
			Thread.currentThread().interrupt();
		} finally {
			logger.trace("Exit Scheduler");			
		}
	}	

}
