package org.gcube.application.enm.service.conn;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Simple FTP client.
 * 
 * @author Erik Torres <ertorser@upv.es>
 * 
 */
public class FtpClient {

	protected GCUBELog logger = new GCUBELog(FtpClient.class);

	private final FTPClient ftp = new FTPClient();

	public FtpClient() {
		// Setup logging
		logger.trace("Constructor...");
	}

	public void connect(final String server, final int port, 
			final String username, final String password) throws IOException {
		final int effectivePort = (port > 0 ? port : ftp.getDefaultPort());
		ftp.connect(server, effectivePort);
		logger.trace("Connected to " + server + " on " + effectivePort);
		// After connection attempt, you should check the reply code to 
		// verify success
		final int reply = ftp.getReplyCode();
		if (!FTPReply.isPositiveCompletion(reply)) {
			ftp.disconnect();				
			throw new IOException("FTP server " + server 
					+ " refused connection");
		}
		// Login
		ftp.login(username, password);		
	}

	public boolean changeDirectory(final String dirname) throws IOException {
		final boolean res = ftp.changeWorkingDirectory(dirname);
		if (res) logger.trace("New working directory is: " + dirname);
		return res;
	}

	public String[] listFiles(final String dirName) throws IOException {
		final FTPFile[] files = ftp.listFiles(dirName);
		String[] filenames = null;
		if (files != null) {
			filenames = new String[files.length];
			for (int i = 0; i < files.length; i++)
				filenames[i] = files[i].getName();
		}
		return filenames;
	}

	public InputStream retrieveFileAsInputStream(final String filename) 
			throws IOException {
		return ftp.retrieveFileStream(filename);
	}

	public String retrieveFileAsString(final String filename) 
			throws IOException {
		final InputStream is = retrieveFileAsInputStream(filename);
		final BufferedReader br = new BufferedReader(new InputStreamReader(is));	 
		String line, content = "";
		while ((line = br.readLine()) != null) {
			content += line;
		}
		br.close();
		logger.trace("File retrieved: " + filename + "' from '" 
				+ ftp.printWorkingDirectory() + "'");
		return content;
	}

	public void makeDirectory(final String dirname) throws IOException {
		ftp.makeDirectory(dirname);
		logger.trace("New directory created: '" + dirname + "' on '" 
				+ ftp.printWorkingDirectory() + "'");
	}

	public void uploadFile(final String localFilename, 
			final String remoteFilename) throws IOException {
		final InputStream input = new FileInputStream(localFilename);
		ftp.storeFile(remoteFilename, input);
		input.close();
		logger.trace("File uploaded: " + remoteFilename + "' to '" 
				+ ftp.printWorkingDirectory() + "'");
	}

	public void uploadString(final String content, final String remoteFilename) 
			throws IOException {
		final InputStream input = new ByteArrayInputStream(content.getBytes());
		ftp.storeFile(remoteFilename, input);
		input.close();
		logger.trace("File uploaded: " + remoteFilename + "' to '" 
				+ ftp.printWorkingDirectory() + "'");
	}

	public void deleteDir(final String dirname) throws IOException {
		deleteDir(ftp, dirname);
	}

	public void disconnect() {
		try {
			if (ftp.isConnected()) {
				try {
					ftp.logout();
					ftp.disconnect();
				} catch(IOException ioe) {
					logger.error(ioe.getLocalizedMessage());
				}
			}
		} catch (Exception e) {
			// nothing to do
		}
	}

	private void deleteDir(final FTPClient ftp, String dirName) 
			throws IOException {
		final FTPFile[] files = ftp.listFiles(dirName);
		// Go over entries
		for (FTPFile file : files) {
			// If the file is a directory and the recurse flag is set, recurse 
			// into the directory
			if (file.isDirectory()) {  
				// System.out.println("Inside dir   : " + dirName + "/" + file.getName());
				deleteDir(ftp, dirName + "/" + file.getName());
			} else {	 
				// System.out.println("Removing file: " + dirName + "/" + file.getName());
				ftp.deleteFile(dirName + "/" + file.getName());
			}			
		}
		// Delete directory
		ftp.removeDirectory(dirName);
		logger.trace("Directory deleted: " + dirName + "' from '" 
				+ ftp.printWorkingDirectory() + "'");
	}

}
