package org.gcube.application.enm.service.plugins.comps;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import com.bsc.venusbes.jobdescription.VENUSCJobGenerator;
import com.bsc.venusbes.venusclient.VENUSClient;
import com.bsc.venusbes.venusclient.credentials.BESCredential;
import com.bsc.venusbes.venusclient.credentials.StorageCredential;
import com.bsc.venusbes.venusclient.types.JobProgress;
import com.bsc.venusbes.venusclient.types.JobStatus;

/**
 * COMPSs client for the execution of openModeller experiments.
 * 
 * @author Roger Rafanell <roger.rafanell@bsc.es>
 */
public class COMPSsClient {

	/** The service wsdl. */
	private String address;

	/**
	 * Instantiates a new OModeller client.
	 *
	 * @param endpoint The endpoint of the COMPSs enactment service.
	 * @param tempDir the temp dir
	 */
	public COMPSsClient(final String endpoint){
		this.address = endpoint;
	}

	/**
	 * Submit a OModeller Job to the COMPSs enactment service.
	 * @param besCred the BES credential.
	 * @param cred the FTP Storage credential.
	 * @param appLocation the CDMI URI of the application.
	 * @param diskSize Set the size of the needed disk size in GB.
	 * @param minOfVMs the minimum number of VMs allowed in the execution.
	 * @param maxOfVMs the maximum number of VMs allowed in the execution.
	 * @param inputSequence Path of the sequence file in the storage (relative to the user home).
	 * @param dataBase Path of the database file in the storage (relative to the user home).
	 * @param numFragments Number of sequence fragments.
	 * @param outputFile Path of the OModeller output in the storage (relative to the user home).
	 * @param commandArgs Command line arguments for blastall binary.
	 * @return A string containing the JobID of the execution.
	 */
	public String submitModel(BESCredential besCred, StorageCredential cred, Integer wallClockTime, String VMImage, 
			Double diskSize, Integer cores, Double memory, Integer minOfVMs, Integer maxOfVMs,
			String jobName, String appLocation, String logsDir,
			String requests,  String testPoints, String outputs){

		String jobId = null;
		String executable = "modeller.Modeller";
		try {

			VENUSClient venusClient = new VENUSClient(this.address, besCred);  
			VENUSCJobGenerator VENUSCJob = new VENUSCJobGenerator();

			VENUSCJob.createJobIdentification(jobName, "OpenBIO OModeller");
			VENUSCJob.createApplication(new URL(appLocation), cred);
			VENUSCJob.setExecutable(executable);

			//Application Debug
			VENUSCJob.createArgument("true");


			//Requests
			StringTokenizer st = new StringTokenizer(requests, "/");
			String requestsName = null;	

			while (st.hasMoreElements()){
				requestsName = st.nextToken();       	
			}

			//Test Points
			st = new StringTokenizer(testPoints, "/");
			String tPoints = null;	

			while (st.hasMoreElements()){
				tPoints = st.nextToken();       	
			}

			//outputs
			st = new StringTokenizer(outputs, "/");
			String outputDir = null;
			while (st.hasMoreElements()){
				outputDir= st.nextToken();       	
			}

			//============ Setting Arguments ==============	  	
			VENUSCJob.createArgument("/sharedDisk/"+ requestsName+"/");		  	
			VENUSCJob.createArgument("/sharedDisk/"+ tPoints);
			VENUSCJob.createArgument("/home/user/IT/"+executable+"/"+ outputDir+"/");	

			//Setting Resources	
			VENUSCJob.setVMInfo(VMImage, diskSize);
			VENUSCJob.setVMCores(cores);
			VENUSCJob.setVMMemory(memory);
			VENUSCJob.setMinVMs(minOfVMs);
			VENUSCJob.setMaxVMs(maxOfVMs);
			VENUSCJob.setWallClockTime(wallClockTime); //Minutes


			//============ Setting Data Staging ==============

			//Request directory
			VENUSCJob.createStageIn(requestsName+"/", new URL(requests), cred);
			VENUSCJob.createStageIn(tPoints, new URL(testPoints), cred);

			String outputPath = outputs.substring(0, (outputs.length() - (outputDir+1).length()));

			//Output
			VENUSCJob.createStageOut(outputDir+"/", new URL(outputPath), cred);

			//Logs
			VENUSCJob.createLogs(new URL(logsDir), cred);

			//Generating job description
			VENUSCJob.generateJobDescription();

			//Submitting job
			jobId = venusClient.createActivity(VENUSCJob.getJobDefinition());


		} catch(Exception e) {
			e.printStackTrace();
		}

		return jobId;
	}

	/**
	 * Retrieves the status of a list of submitted jobs.
	 *
	 * @param besCred the BES credential.
	 * @param jobIds Array of job IDs. One job is supported in this version.
	 * @return Array of ActivityStatus containing the status of the job.
	 */
	public List<JobStatus> OModellerStatus(BESCredential besCred, String[] jobIds) {

		VENUSClient venusClient = new VENUSClient(this.address, besCred); 
		List<JobStatus>  status = null;

		try {
			status = venusClient.getActivityStatus(jobIds);

		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return status;
	}

	/**
	 * Retrieves the progress of concrete jobs.
	 *
	 * @param besCred the BES credential.
	 * @param jobIds The IDs of the retrieved jobs.
	 * @return List of JobProgress objects of each job.
	 */
	public List<JobProgress> getJobsProgress(BESCredential besCred, String[] jobIds) {

		VENUSClient venusClient = new VENUSClient(this.address, besCred); 
		List<JobProgress> progressList = new ArrayList<JobProgress>();

		try {	
			progressList = venusClient.getJobsProgress(jobIds);

		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return progressList;
	}

	/**
	 * Terminate a list of submitted jobs.
	 *
	 * @param besCred the BES credential.
	 * @param jobIds Array of job IDs
	 */
	public List<String> terminateOModeller(BESCredential besCred, String[] jobIds){
		VENUSClient venusClient = new VENUSClient(this.address, besCred); 
		List<String> messageList = null;
		try {
			messageList = venusClient.terminateActivity(jobIds);

		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return messageList;
	}

	// % of job's completeness
	public int getCompletness(BESCredential besCred, String[] jobIds) {
		if (jobIds != null && jobIds.length > 0) {
			List<JobProgress> progress = getJobsProgress(besCred, jobIds);
			if (progress != null) {
				if (progress.get(0).getTotal() > 0)
					return (int) (((float) progress.get(0).getCompleted()/progress.get(0).getTotal())*100); //% of job's completness
			}
		}
		return 0;
	} 

}
