package org.gcube.application.enm.service;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;

import javax.xml.datatype.DatatypeConfigurationException;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.common.xml.status.OperationType;
import org.gcube.application.enm.common.xml.status.OperationTypeType;
import org.gcube.application.enm.common.xml.status.PastOperationsType;
import org.gcube.application.enm.common.xml.status.StatusType;
import org.gcube.application.enm.service.concurrent.JobMonitor;
import org.gcube.application.enm.service.util.XmlHelper;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Represents a job. Execution resource providers must extend this class with 
 * the implementation details specific to how the experiments are executed in 
 * their platforms. They must override the method {@link GenericJob#call()} with 
 * the logic that allows the management of the complete life-cycle of the job,
 * and the constructors {@link GenericJob#GenericJob(UUID, ExperimentRequest)}
 * and {@link GenericJob#GenericJob(UUID, ExperimentRequest, ExperimentStatus, ExperimentResults, ExperimentLogs)}, 
 * which are used by {@link PluginLoader} to load and inject new providers into
 * the service.
 * Class {@link JobMonitor} uses this logic to manage the execution of the job. 
 * <b>Note:</b> This class has a natural ordering that is inconsistent with 
 * equals.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public abstract class GenericJob implements Callable<String>, 
Comparable<GenericJob> {

	protected GCUBELog logger = new GCUBELog(GenericJob.class);

	public static int MINIMUM_PRIORITY = 20;
	public static int MAXIMUM_PRIORITY = 0;

	protected final UUID uuid;
	protected final ExperimentRequest request;
	protected final ExperimentStatus status;
	protected final ExperimentResults results;
	protected final ExperimentLogs logs;

	private int priority;
	private boolean cancelled = false;

	public GenericJob(final UUID uuid, final ExperimentRequest request) {
		this(uuid, request, initialStatus(), initialResults(), initialLogs());
	}

	public GenericJob(final UUID uuid, final ExperimentRequest request, 
			final ExperimentStatus status, final ExperimentResults results, 
			final ExperimentLogs logs) {
		// Setup logging
		logger.trace("Constructor...");
		// Setup the job
		this.uuid = uuid;
		this.request = request;
		this.status = status;
		this.results = results;
		this.logs = logs;
		setPriority(MINIMUM_PRIORITY);
	}

	private static ExperimentStatus initialStatus() {
		final org.gcube.application.enm.common.xml.status.ObjectFactory factory = 
				new org.gcube.application.enm.common.xml.status.ObjectFactory();
		final ExperimentStatus initialStatus = factory.createExperimentStatus();
		initialStatus.setStatus(StatusType.PENDING);
		initialStatus.setCompletenessPercentage(new BigDecimal(0));
		try {
			initialStatus.setStartDate(XmlHelper.now());
		} catch (DatatypeConfigurationException e) {
			// nothing to do			
		}
		initialStatus.setEndDate(null);
		initialStatus.setExecutionTrace(factory.createExecutionTraceType());
		final OperationType currentOperation = factory.createOperationType();
		final OperationTypeType operation = factory.createOperationTypeType();
		operation.setOperationId(BigInteger.valueOf(-1));		
		currentOperation.setOperation(operation);
		initialStatus.getExecutionTrace().setCurrentOperation(currentOperation);
		final PastOperationsType pastOperations = factory.createPastOperationsType();
		initialStatus.getExecutionTrace().setPastOperations(pastOperations);
		return initialStatus;
	}

	private static ExperimentResults initialResults() {
		final ExperimentResults initialResults = (new org.gcube.application.enm
				.common.xml.results.ObjectFactory()).createExperimentResults();
		return initialResults;
	}

	private static ExperimentLogs initialLogs() {
		final ExperimentLogs initialLogs = (new org.gcube.application.enm
				.common.xml.logs.ObjectFactory()).createExperimentLogs();
		return initialLogs;
	}	

	protected void next(final int step, final String jobId) {
		final String jobIds[] = { jobId };
		next(step, jobIds);
	}

	protected void next(final int step, final List<String> jobIds) {
		next(step, jobIds.toArray(new String[jobIds.size()]));
	}

	protected void next(final int step, final String[] jobIds) {
		final org.gcube.application.enm.common.xml.status.ObjectFactory factory = 
				new org.gcube.application.enm.common.xml.status.ObjectFactory();		
		final OperationType newOperation = factory.createOperationType();	
		final OperationTypeType operation = factory.createOperationTypeType();		
		operation.setOperationId(BigInteger.valueOf(step));
		if (jobIds != null && jobIds.length > 0)
			operation.getRemoteJobId().addAll(Arrays.asList(jobIds));		
		newOperation.setOperation(operation);
		if (status.getExecutionTrace().getCurrentOperation() != null) {
			status.getExecutionTrace().getPastOperations().getPastOperation().add(
					status.getExecutionTrace().getCurrentOperation());
		}
		status.getExecutionTrace().setCurrentOperation(newOperation);
	}

	protected int currentStep() {
		return currenOperation().getOperationId().intValue();
	}

	protected List<String> currentJobIds() {
		return (currenOperation().getRemoteJobId() != null) 
				? currenOperation().getRemoteJobId() : new ArrayList<String>();
	}

	private OperationTypeType currenOperation() {
		return status.getExecutionTrace().getCurrentOperation().getOperation();
	}

	public final UUID getUUID() {
		return uuid;
	}

	public final ExperimentRequest getRequest() {
		return request;
	}

	public final ExperimentStatus getStatus() {
		return status;
	}

	public final ExperimentResults getResults() {
		return results;
	}

	public final ExperimentLogs getLogs() {
		return logs;
	}

	public int getPriority() {
		return priority;
	}

	public void setPriority(int priority) {
		this.priority = priority;
	}

	public final boolean isCanceled() {
		return cancelled;
	}

	public final void cancel() {
		priority = MAXIMUM_PRIORITY;
		cancelled = true;
	}

	public abstract ExecutionResource getExecutionResource();

	@Override
	public int compareTo(final GenericJob other) {
		return (other == null || other.uuid == null 
				|| other.priority < MINIMUM_PRIORITY 
				|| other.priority > MAXIMUM_PRIORITY) ? -1
						: -Integer.valueOf(other.priority).compareTo(
								Integer.valueOf(priority));		
	}

	@Override
	public boolean equals(final Object other) {
		if (other instanceof GenericJob) {
			return uuid == ((GenericJob) other).uuid;
		}
		return false;
	}

}
