package org.gcube.application.geoportal.client;

import java.io.InputStream;
import java.util.ArrayList;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.gcube.application.geoportal.client.utils.Serialization;
import org.gcube.application.geoportal.common.model.legacy.Concessione;
import org.gcube.application.geoportal.common.rest.AddSectionToConcessioneRequest;
import org.gcube.application.geoportal.common.rest.InterfaceConstants;
import org.gcube.application.geoportal.common.rest.MongoConcessioni;
import org.gcube.common.clients.Call;
import org.gcube.common.clients.delegates.ProxyDelegate;

import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
public class DefaultMongoConcessioni implements MongoConcessioni{

	@NonNull
	private final ProxyDelegate<WebTarget> delegate;
	
	private ObjectMapper mapper=Serialization.mapper;
	
	@Override
	public Concessione createNew(Concessione c) throws Exception {
		log.debug("Serializing {} ",c);
		final String serialized=mapper.writeValueAsString(c);
		Call<WebTarget,Concessione> call= new Call<WebTarget,Concessione>(){
			@Override
			public Concessione call(WebTarget endpoint) throws Exception {				
				Response resp= endpoint.request(MediaType.APPLICATION_JSON).post(Entity.entity(serialized, MediaType.APPLICATION_JSON));
				return check(resp,Concessione.class);
			}
		};
		Concessione toReturn = delegate.make(call);
		log.info("Registered {}",toReturn);
		return toReturn;
	}

	@Override
	public void deleteById(String id) throws Exception {
		log.debug("Deleting by id {}",id);
		delegate.make(new Call<WebTarget,String>(){
			@Override
			public String call(WebTarget endpoint) throws Exception {
				check(endpoint.path(id).request(MediaType.APPLICATION_JSON).delete(),null);
				log.info("Deleted by id {}",id);
				return null;
			}
		});
	}

	@Override
	public Concessione getById(String id) throws Exception {
		log.debug("Loading by id {}",id);
		Call<WebTarget,Concessione> call= new Call<WebTarget,Concessione>(){
			@Override
			public Concessione call(WebTarget endpoint) throws Exception {				
				Response resp= endpoint.path(id).request(MediaType.APPLICATION_JSON).get();
				return check(resp,Concessione.class);
			}
		};
		Concessione toReturn = delegate.make(call);
		log.debug("Loaded {}",toReturn);
		return toReturn;
	}

	@Override
	public Iterable<Concessione> getList() throws Exception {
		log.debug("Getting list");
		Call<WebTarget,Iterable<Concessione>> call=new Call<WebTarget,Iterable<Concessione>>(){
			@Override
			public Iterable<Concessione> call(WebTarget endpoint) throws Exception {
				Iterable<Concessione> toReturn=new ArrayList<Concessione>();
				return (Iterable<Concessione>) mapper.readerFor(Concessione.class).readValues(
						endpoint.request(MediaType.APPLICATION_JSON).get(InputStream.class));
			}
		};
		return delegate.make(call);
	}

	@Override
	public Concessione publish(String id) throws Exception {
		log.info("Publishing {} ",id);
		Call<WebTarget,Concessione> call= new Call<WebTarget,Concessione>(){
			@Override
			public Concessione call(WebTarget endpoint) throws Exception {				
				Response resp= endpoint.path(InterfaceConstants.Methods.PUBLISH_PATH).
						path(id).request(MediaType.APPLICATION_JSON).put(Entity.entity(id, MediaType.APPLICATION_JSON));
				return check(resp,Concessione.class);
			}
		};
		Concessione toReturn = delegate.make(call);
		log.debug("Published {} ",toReturn);
		return toReturn;
	}

	@Override
	public Concessione registerFile(String id, AddSectionToConcessioneRequest request) throws Exception {
		log.info("Registering {} in {}",request,id);
		Call<WebTarget,Concessione> call= new Call<WebTarget,Concessione>(){
			@Override
			public Concessione call(WebTarget endpoint) throws Exception {				
				Response resp= endpoint.path(InterfaceConstants.Methods.REGISTER_FILES_PATH).
						path(id).request(MediaType.APPLICATION_JSON).post(Entity.entity(mapper.writeValueAsString(request), 
								MediaType.APPLICATION_JSON));
				return check(resp,Concessione.class);
			}
		};
		Concessione toReturn = delegate.make(call);
		log.debug("Published {} ",toReturn);
		return toReturn;
	}

	@Override
	public Concessione update(String id, String jsonUpdate) throws Exception {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Concessione replace(Concessione replacement) throws Exception {
		log.info("Replacing {}",replacement);
		Call<WebTarget,Concessione> call= new Call<WebTarget,Concessione>(){
			@Override
			public Concessione call(WebTarget endpoint) throws Exception {				
				Response resp= endpoint.
						request(MediaType.APPLICATION_JSON).
						put(Entity.entity(mapper.writeValueAsString(replacement), 
								MediaType.APPLICATION_JSON));
				return check(resp,Concessione.class);
			}
		};
		Concessione toReturn = delegate.make(call);
		log.debug("Reloaded {} ",toReturn);
		return toReturn;
	}

	
	protected static<T> T check(Response resp, Class<T> clazz) throws Exception {
		String resString=resp.readEntity(String.class);
		if(resp.getStatus()<200||resp.getStatus()>=300)
			throw new Exception("RESP STATUS IS "+resp.getStatus()+". Message : "+resString);
		System.out.println("Resp String is "+resString);
		if(clazz!=null)
			return Serialization.read(resString, clazz);
		else return null;
	}
}
