package org.gcube.application.geoportalcommon;

import static org.gcube.application.geoportal.client.GeoportalAbstractPlugin.mongoConcessioni;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.bson.Document;
import org.gcube.application.geoportal.common.model.legacy.Concessione;
import org.gcube.application.geoportal.common.model.rest.QueryRequest;
import org.gcube.application.geoportal.common.model.rest.QueryRequest.OrderedRequest;
import org.gcube.application.geoportal.common.model.rest.QueryRequest.OrderedRequest.Direction;
import org.gcube.application.geoportal.common.model.rest.QueryRequest.PagedRequest;
import org.gcube.application.geoportal.common.rest.MongoConcessioni;
import org.gcube.application.geoportalcommon.shared.ItemField;
import org.gcube.application.geoportalcommon.shared.ResultSetPaginatedData;
import org.gcube.application.geoportalcommon.shared.SearchingFilter;
import org.gcube.application.geoportalcommon.shared.SearchingFilter.LOGICAL_OP;
import org.gcube.application.geoportalcommon.shared.SearchingFilter.ORDER;
import org.gcube.application.geoportalcommon.shared.WhereClause;
import org.gcube.application.geoportalcommon.shared.products.ConcessioneDV;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;

/**
 * The Class MongoServiceCommon.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR francesco.mangiacrapa@isti.cnr.it
 * 
 *         Dec 3, 2021
 */
public class MongoServiceCommon {

	private static Logger LOG = LoggerFactory.getLogger(MongoServiceCommon.class);

	/**
	 * Gets the instance mongo concessioni.
	 *
	 * @return the instance mongo concessioni
	 */
	public MongoConcessioni getInstanceMongoConcessioni() {
		return mongoConcessioni().build();
	}

	/**
	 * Gets the list of concessioni.
	 *
	 * @param reloadFromService the reload from service
	 * @return the list of concessioni
	 * @throws Exception the exception
	 */
	public List<Concessione> getListOfConcessioni() throws Exception {
		LOG.info("called getListOfConcessioni");

		List<Concessione> listOfConcessioni = new ArrayList<Concessione>();
		LOG.info("Loading list of concessioni from client mongo");
		MongoConcessioni clientMongo = getInstanceMongoConcessioni();

		Iterator<Concessione> concessioni = clientMongo.getList();
		if (concessioni != null) {
			while (concessioni.hasNext()) {
				Concessione concessione = (Concessione) concessioni.next();
				listOfConcessioni.add(concessione);

			}
		}

		LOG.info("read list of concessioni with size: " + listOfConcessioni.size());
		return listOfConcessioni;
	}

	/**
	 * Query on mongo.
	 *
	 * @param offset            the offset
	 * @param limit             the limit
	 * @param filter            the filter
	 * @param recordType        the record type
	 * @return the result set paginated data
	 * @throws Exception the exception
	 */
	public ResultSetPaginatedData queryOnMongo(Integer totalItems, Integer offset, Integer limit, SearchingFilter filter, String recordType) throws Exception {

		try {

			if (recordType.equalsIgnoreCase("concessione")) {
				MongoConcessioni clientMongo = getInstanceMongoConcessioni();

				if(totalItems==null || totalItems < 0) {
					// TODO MUST BE REPLACED BY COUNT
					List<Concessione> listOfConcessioni = getListOfConcessioni();
					int listConcessioniSize = listOfConcessioni.size();
					totalItems = listConcessioniSize;
				}
				
				Integer offsetIndex = offset;
				Integer limitIndex = limit;

				if (offset == null || offset<0) {
					offsetIndex = 0;
				}
				if (limit == null || limit<0) {
					limitIndex = totalItems;
				}

				ResultSetPaginatedData searchedData = new ResultSetPaginatedData(offsetIndex, limitIndex, false);
				searchedData.setTotalItems(totalItems);

				List<ConcessioneDV> toReturnList = new ArrayList<ConcessioneDV>();
				Direction sDirection = null;
				List<String> orderingFields = new ArrayList<String>();

				if (filter == null) {
					LOG.info("No filter found, creating empty filter");
					filter = new SearchingFilter();
				}

				ORDER order = filter.getOrder();

				if (order == null) {
					order = ORDER.ASC;
					LOG.info("No direction/order found, using default: " + order);
				}

				switch (order) {
				case ASC:
					sDirection = Direction.ASCENDING;
					break;
				case DESC:
					sDirection = Direction.DESCENDING;
					break;
				}

				List<ItemField> orderByFields = filter.getOrderByFields();
				
				if(orderByFields==null) {
					orderByFields = new ArrayList<ItemField>();
				}
				
				if(orderByFields.isEmpty()) {
					ItemField orderD = new ItemField("", Arrays.asList("name"), false, false, false);
					LOG.info("Order by is null, adding default: " + orderD);
					orderByFields.add(orderD);
				}
				
				for (ItemField itemField : orderByFields) {
					if (itemField.getJsonFields() != null) {
						for (String field : itemField.getJsonFields()) {
							orderingFields.add(field);
						}
					}

				}

				QueryRequest request = new QueryRequest();
				PagedRequest paging = new PagedRequest();
				paging.setOffset(offsetIndex);
				paging.setLimit(limitIndex);
				request.setPaging(paging);

				OrderedRequest ordering = new OrderedRequest();
				ordering.setDirection(sDirection);
				ordering.setFields(orderingFields);

				request.setOrdering(ordering);
				
				Document query = new Document();
				if(filter.getConditions()!=null) {
					for (WhereClause whereClause : filter.getConditions()) {
						
						LOGICAL_OP searchWithOperator = whereClause.getOperator();
						if(searchWithOperator==null) {
							searchWithOperator = LOGICAL_OP.OR;
						}
	
						if (whereClause.getSearchInto() != null) {
							Map<String, Object> searchFields = whereClause.getSearchInto();
							BasicDBObjectBuilder builder = BasicDBObjectBuilder.start();
							for (String key : searchFields.keySet()) {
								// using regex and case-insensitive
								BasicDBObject bs = new BasicDBObject();
								bs.append("$regex", searchFields.get(key));
								bs.append("$options", "i");
								builder.append(key, bs);
								
								
							}
							//Building list of Document in OR clause
							BasicDBList list = new BasicDBList();
							Map map = builder.get().toMap();
							for (Object key : map.keySet()) {
								
								BasicDBObject value = (BasicDBObject) map.get(key);
								Document doc = new Document((String) key, value);
								list.add(doc);
							}
							
							//query = new Document();
							query.put(searchWithOperator.getOperator(), list);
							
//							BasicDBObject bs = new BasicDBObject();
//							bs.append("$eq", "PASSED");
//							query.put("report.status", bs);
							
						}
					}
				}
				
				request.setFilter(query);
				
				LOG.info("Paging offset: " + offsetIndex + ", limit: " + limitIndex);
				LOG.info("Direction: " + sDirection);
				LOG.info("Order by Fields: " + orderingFields);
				LOG.info("Search for conditions: " + filter.getConditions());
				if (query != null) {
					LOG.info("Search query to JSON: " + query.toJson());
				}

				Iterator<Concessione> concessioni = clientMongo.query(request);
				int i = 0;
				while (concessioni.hasNext()) {
					Concessione concessione = concessioni.next();
					ConcessioneDV concessioneDV = ConvertToDataViewModel.toMetadataConcessione(concessione, true);
					toReturnList.add(concessioneDV);
					i++;
					LOG.trace(i+") converted: " + concessioneDV);
				}
				LOG.debug("read " + toReturnList + " project/s");

				searchedData.setData(toReturnList);

				// TODO WORKAROUND MUST BE REMOVE AFTER THE QUERY COUNT
				// AND LIST.SIZE WILL BE AVAILABLE IN THE SERVICE
				if (filter.getConditions() != null) {
					searchedData.setTotalItems(toReturnList.size());
					totalItems = toReturnList.size();
				}
				
				if (totalItems == limit || totalItems == 0) {
					LOG.debug("Page completed returning " + totalItems + " items");
					int newOffset = offsetIndex + limitIndex;
					searchedData.setServerSearchFinished(newOffset > totalItems || totalItems == 0);
					LOG.debug("is Search finished: " + searchedData.isServerSearchFinished());

				}

				return searchedData;
			}

		} catch (Exception e) {
			LOG.error("Error on loading paginated and filtered list of concessioni: ", e);
			throw new Exception("Error occurred on loading list of Concessioni. Error: " + e.getMessage());
		}

		return null;

	}

}
